/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  FmConfirm.c

  (c) Simon Marlow 1990-1993

  modified 2005,2006 by Bernhard R. Link (see Changelog)

  Module for creating and implementing the confirmation window
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <xfmconfig.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw3d/Form.h>
#include <X11/Xaw3d/Label.h>
#include <X11/Xaw3d/Box.h>

#include "global.h"

#include "Am.h"
#include "Fm.h"

#define LABEL_WIDTH 400

/*-----------------------------------------------------------------------------
  STATIC DATA
-----------------------------------------------------------------------------*/

struct ConfirmationDialog {
	Widget shell, label1, label2, label3;
	enum { DontKnow, Ok, Cancel, Abort } confirm_flag;
	Boolean aborted;
};

/*-----------------------------------------------------------------------------
  Widget Argument Lists
-----------------------------------------------------------------------------*/

static Arg shell_args[] = {
  { XtNtitle, (XtArgVal) "Confirm" },
  { XtNtransientFor, (XtArgVal) NULL }
};

static Arg *form_args = NULL;

static Arg label_args[] = {
  { XtNfromHoriz, (XtArgVal) NULL },
  { XtNfromVert, (XtArgVal) NULL },
  { XtNlabel, (XtArgVal) NULL },
  { XtNwidth, (XtArgVal) LABEL_WIDTH },
  { XtNfont, (XtArgVal) NULL },
  { XtNresize, (XtArgVal) False },
  { XtNtop, XtChainTop },
  { XtNbottom, XtChainTop },
  { XtNleft, XtChainLeft },
  { XtNright, XtChainRight }
};

static Arg button_box_args[] = {
  { XtNfromHoriz, (XtArgVal) NULL },
  { XtNfromVert, (XtArgVal) NULL },
  { XtNtop, XtChainTop },
  { XtNbottom, XtChainTop },
  { XtNleft, XtChainLeft },
  { XtNright, XtChainLeft }
};

/*-----------------------------------------------------------------------------
  PRIVATE FUNCTIONS
-----------------------------------------------------------------------------*/

static void confirmOkCb(UNUSED(Widget w), void *client_data,
		        UNUSED(void *call_data)) {
	struct ConfirmationDialog *d = client_data;

	XtPopdown(d->shell);
	d->confirm_flag = Ok;
}

/*---------------------------------------------------------------------------*/

static void confirmCancelCb(UNUSED(Widget w), void *client_data,
			    UNUSED(void *call_data)) {
	struct ConfirmationDialog *d = client_data;

	XtPopdown(d->shell);
	d->confirm_flag = Cancel;
}

/*---------------------------------------------------------------------------*/

static void confirmAbortCb(UNUSED(Widget w), void *client_data,
			   UNUSED(void *call_data)) {
	struct ConfirmationDialog *d = client_data;

	XtPopdown(d->shell);
	d->confirm_flag = Abort;
}

/*-----------------------------------------------------------------------------
  Button Data
-----------------------------------------------------------------------------*/

static const ButtonRec confirm_buttons[] = {
  { "ok", "Continue", confirmOkCb },
  { "cancel", "Cancel", confirmCancelCb },
  { "abort", "Abort", confirmAbortCb },
};


/*-----------------------------------------------------------------------------
  PUBLIC FUNCTIONS
-----------------------------------------------------------------------------*/

static void createConfirmPopup(Widget parent, struct ConfirmationDialog **out)
{
  Widget form, button_box;
  struct ConfirmationDialog *d;

  d = (struct ConfirmationDialog*)XtMalloc(sizeof(struct ConfirmationDialog));

  /* create shell */
  shell_args[1].value = (XtArgVal) parent;
  d->shell = XtCreatePopupShell("confirm", transientShellWidgetClass,
			     parent, shell_args, XtNumber(shell_args));

  /* create outer form */
  form = XtCreateManagedWidget("form", formWidgetClass, d->shell,
				      form_args, XtNumber(form_args) );

  /* create label 1 */
  label_args[4].value = (XtArgVal) resources.label_font;
  d->label1 = XtCreateManagedWidget("label1",labelWidgetClass, form,
				 label_args, XtNumber(label_args) );

  /* create label 2 */
  label_args[1].value = (XtArgVal) d->label1;
  d->label2 = XtCreateManagedWidget("label2",labelWidgetClass, form,
				 label_args, XtNumber(label_args) );

  /* create label 3 */
  label_args[1].value = (XtArgVal) d->label2;
  d->label3 = XtCreateManagedWidget("label3",labelWidgetClass, form,
				 label_args, XtNumber(label_args) );

  /* create button box */
  button_box_args[1].value = (XtArgVal) d->label3;
  button_box = XtCreateManagedWidget("button box", boxWidgetClass, form,
				     button_box_args,
				     XtNumber(button_box_args) );
  createButtons(confirm_buttons, XtNumber(confirm_buttons), button_box, d);

#ifdef ENHANCE_POP_ACCEL
  XtInstallAllAccelerators(form, form);
  XtInstallAllAccelerators(d->shell, form);
#endif

  XtRealizeWidget(d->shell);
  setWMProps(d->shell);
  *out = d;
}

/*---------------------------------------------------------------------------*/

static int confirm(struct ConfirmationDialog *d, const char *s1, const char *s2, const char *s3,
		Boolean *aborted)
{
  XEvent e;

  XtVaSetValues(d->label1, XtNlabel, s1, NULL);
  XtVaSetValues(d->label2, XtNlabel, s2, NULL);
  XtVaSetValues(d->label3, XtNlabel, s3, NULL);
  popupByCursor(d->shell, XtGrabExclusive);

  d->confirm_flag = DontKnow;

  do {
    XtAppNextEvent(app_context, &e);
    XtDispatchEvent(&e);
  } while (d->confirm_flag == DontKnow);

  if (aborted != NULL)
    *aborted = (d->confirm_flag == Abort);
  return (d->confirm_flag == Ok);
}

int fwConfirm(FileWindowRec *fw, const char *s1, const char *s2, const char *s3, Boolean *aborted) {
	if (fw->confirmationDialog == NULL)
		createConfirmPopup(fw->shell, &fw->confirmationDialog);
	return confirm(fw->confirmationDialog, s1, s2, s3, aborted);
}

static struct ConfirmationDialog *amConfirmDialog = NULL;

int amConfirm(const char *s1, const char *s2, const char *s3, Boolean *aborted) {
	if (amConfirmDialog == NULL)
		createConfirmPopup(aw.shell, &amConfirmDialog);
	return confirm(amConfirmDialog, s1, s2, s3, aborted);
}
