/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * Copyright (C) 2019 Tianjin KYLIN Information Technology Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */
#include "screensaver.h"
#include "ui_screensaver.h"

#include <QDebug>
#include <QBoxLayout>

#define SSTHEMEPATH "/usr/share/applications/screensavers/"
#define ID_PREFIX "screensavers-ukui-"

#define SCREENSAVER_SCHEMA "org.ukui.screensaver"
#define MODE_KEY "mode"
#define THEMES_KEY "themes"
#define LOCK_KEY "lock-enabled"
#define ACTIVE_KEY "idle-activation-enabled"

#define SESSION_SCHEMA "org.ukui.session"
#define IDLE_DELAY_KEY "idle-delay"

#define IDLEMIN 1
#define IDLEMAX 120
#define IDLESTEP 1

typedef enum
{
    MODE_BLANK_ONLY,
    MODE_RANDOM,
    MODE_SINGLE,
    MODE_IMAGE,
    MODE_DEFAULT_UKUI,
}SaverMode;

void PreviewWidget::paintEvent(QPaintEvent *e){
//    QPainter painter(this);
//    painter.setCompositionMode(QPainter::CompositionMode_DestinationOver);
//    qDebug() << Q_FUNC_INFO << this->rect();
//    painter.fillRect(this->rect(), Qt::red);
}


Screensaver::Screensaver()
{
    ui = new Ui::Screensaver;
    itemDelege = new QStyledItemDelegate();
    pluginWidget = new QWidget;
    pluginWidget->setAttribute(Qt::WA_DeleteOnClose);
    ui->setupUi(pluginWidget);

    pluginName = tr("screensaver");
    pluginType = PERSONALIZED;

    pluginWidget->setStyleSheet("background: #ffffff;");

    ui->previewWidget->setStyleSheet("#previewWidget{background: black; border-radius: 6px;}");
    ui->previewWidget->setAutoFillBackground(true);

    mPreviewWidget = new PreviewWidget;
    mPreviewWidget->resize(400,222);
    QBoxLayout *mPreviewLayout = new QBoxLayout(QBoxLayout::TopToBottom);
    mPreviewLayout->addWidget(mPreviewWidget);
    ui->previewWidget->setLayout(mPreviewLayout);

    ui->enableWidget->setStyleSheet("QWidget{background: #F4F4F4; border-top-left-radius: 6px; border-top-right-radius: 6px;}");

    ui->programWidget->setStyleSheet("QWidget#programWidget{background: #F4F4F4;}");
    ui->programLabel->setStyleSheet("QLabel{background: #F4F4F4;}");

    ui->idleWidget->setStyleSheet("QWidget{background: #F4F4F4; border-bottom-left-radius: 6px; border-bottom-right-radius: 6px;}");
    ui->idleLineEdit->setStyleSheet("QLineEdit{background: #ffffff; border-radius: 0px;}");

    ui->lockWidget->setStyleSheet("QWidget{background: #F4F4F4; border-radius: 6px;}");

    process = new QProcess();

    _acquireThemeinfoList();
    initComponent();
    initEnableBtnStatus();
    initThemeStatus();
    initIdleSliderStatus();

    init_theme_info_map();
    component_init();
    status_init();
}

Screensaver::~Screensaver()
{
    delete ui;
    delete process;
    process = nullptr;
}

QString Screensaver::get_plugin_name(){
    return pluginName;
}

int Screensaver::get_plugin_type(){
    return pluginType;
}

QWidget *Screensaver::get_plugin_ui(){
    return pluginWidget;
}

void Screensaver::plugin_delay_control(){
    // 初始化屏保预览Widget
    initPreviewWidget();
}

void Screensaver::initComponent(){

    //添加开启屏保按钮
    enableSwitchBtn = new SwitchButton(ui->enableWidget);
    ui->enableHorLayout->addStretch();
    ui->enableHorLayout->addWidget(enableSwitchBtn);

    //添加锁定屏幕开关按钮
    lockSwitchBtn = new SwitchButton(ui->lockWidget);
    ui->lockHorLayout->addStretch();
    ui->lockHorLayout->addWidget(lockSwitchBtn);

    //初始化屏保程序下拉列表
    ui->comboBox->setItemDelegate(itemDelege);
    ui->comboBox->setMaxVisibleItems(5);
    ui->comboBox->addItem(tr("Default_ukui"));
    ui->comboBox->addItem(tr("Blank_Only"));
    ui->comboBox->addItem(tr("Random"));
    QMap<QString, SSThemeInfo>::iterator it = infoMap.begin();
    for (int index = 3; it != infoMap.end(); it++, index++){
        SSThemeInfo info = (SSThemeInfo)it.value();
        ui->comboBox->addItem(info.name);
        ui->comboBox->setItemData(index, QVariant::fromValue(info));
    }

    //初始化滑动条
    ui->idleSlider->setMinimum(IDLEMIN);
    ui->idleSlider->setMaximum(IDLEMAX);
    ui->idleSlider->setSingleStep(IDLESTEP);
    ui->idleSlider->setPageStep(IDLESTEP);

    connect(enableSwitchBtn, &SwitchButton::checkedChanged, this, [=](bool checked){
        screensaver_settings = g_settings_new(SCREENSAVER_SCHEMA);
        g_settings_set_boolean(screensaver_settings, ACTIVE_KEY, checked);

        //刷新LockWidget状态
        ui->lockWidget->setVisible(checked);
        g_object_unref(screensaver_settings);
    });

    connect(lockSwitchBtn, &SwitchButton::checkedChanged, this, [=](bool checked){
        //REVIEW*** g_object_unref faild
        //    screensaver_settings = g_settings_new(SCREENSAVER_SCHEMA);
        //    g_settings_set_boolean(screensaver_settings, LOCK_KEY, status);
        //    if (screensaver_settings)
        //            g_object_unref(screensaver_settings);
        const QByteArray ba(SCREENSAVER_SCHEMA);
        QGSettings * settings = new QGSettings(ba);
        settings->set(LOCK_KEY, checked);
        delete settings;
    });

    connect(ui->idleSlider, &QSlider::valueChanged, this, [=](int value){
        //刷新分钟显示
        ui->idleLineEdit->blockSignals(true);
        ui->idleLineEdit->setText(QString::number(value));
        ui->idleLineEdit->blockSignals(false);
    });
    connect(ui->idleSlider, &QSlider::sliderReleased, this, [=]{
        int value = ui->idleSlider->value();

        session_settings = g_settings_new(SESSION_SCHEMA);
        g_settings_set_int(session_settings, IDLE_DELAY_KEY, value);
        g_object_unref(session_settings);
    });

}

void Screensaver::initPreviewWidget(){
    startupScreensaver();
}

void Screensaver::initEnableBtnStatus(){

    const QByteArray ba(SCREENSAVER_SCHEMA);
    QGSettings * settings = new QGSettings(ba);

    //初始化EnableBtn状态
    bool active;
    active = settings->get(ACTIVE_KEY).toBool();
    enableSwitchBtn->blockSignals(true);
    enableSwitchBtn->setChecked(active);
    enableSwitchBtn->blockSignals(false);

    //初始化LockWidget状态
    ui->lockWidget->setVisible(active);

    bool locked;
    locked = settings->get(LOCK_KEY).toBool();
    initLockBtnStatus(locked);


    delete settings;

}

void Screensaver::initLockBtnStatus(bool status){
    lockSwitchBtn->blockSignals(true);
    lockSwitchBtn->setChecked(status);
    lockSwitchBtn->blockSignals(false);
}

void Screensaver::initThemeStatus(){
    int mode;
    char * name;

    ui->comboBox->blockSignals(true);

    screensaver_settings = g_settings_new(SCREENSAVER_SCHEMA);
    mode = g_settings_get_enum(screensaver_settings, MODE_KEY);

    if (mode == MODE_DEFAULT_UKUI){
        ui->comboBox->setCurrentIndex(0); //UKUI
    }
    else if (mode == MODE_BLANK_ONLY){
        ui->comboBox->setCurrentIndex(1); //Black_Only
    }
    else if (mode == MODE_RANDOM){
        ui->comboBox->setCurrentIndex(2); //Random
    }
    else{
        gchar ** strv;
        strv = g_settings_get_strv(screensaver_settings, THEMES_KEY);

        if (strv != NULL){
            name = g_strdup(strv[0]);
            QString nameqt = QString(name);
            SSThemeInfo info = (SSThemeInfo)infoMap.find(name).value();
            ui->comboBox->setCurrentText(info.name);
        }
        else
            ui->comboBox->setCurrentIndex(0); //no data, default Blank_Only
        g_strfreev(strv);
    }
    g_object_unref(screensaver_settings);

    ui->comboBox->blockSignals(false);
}

void Screensaver::initIdleSliderStatus(){
    int minutes;
    session_settings = g_settings_new(SESSION_SCHEMA);
    minutes = g_settings_get_int(session_settings, IDLE_DELAY_KEY);

    ui->idleSlider->blockSignals(true);
    ui->idleSlider->setValue(minutes);
    ui->idleSlider->blockSignals(false);

    ui->idleLineEdit->blockSignals(true);
    ui->idleLineEdit->setText(QString::number(minutes));
    ui->idleLineEdit->blockSignals(true);

    g_object_unref(session_settings);
}

void Screensaver::component_init(){

    //设置屏保预览widget的背景为黑色
//    mPreviewWidget->setStyleSheet("#previewWidget{background: black}");

    //
//    activeswitchbtn = new SwitchButton();
//    activeswitchbtn->setAttribute(Qt::WA_DeleteOnClose);
//    ui->activeHLayout->addWidget(activeswitchbtn);
//    ui->activeHLayout->addStretch();

//    lockswitchbtn = new SwitchButton();
//    lockswitchbtn->setAttribute(Qt::WA_DeleteOnClose);
//    ui->lockHLayout->addWidget(lockswitchbtn);
//    ui->lockHLayout->addStretch();

//    ui->comboBox->addItem(tr("Blank_Only"));
//    ui->comboBox->addItem(tr("Random"));

//    QMap<QString, SSThemeInfo>::iterator it = infoMap.begin();
//    for (int itemsindex = 2; it != infoMap.end(); it++, itemsindex++){
//        SSThemeInfo info = (SSThemeInfo)it.value();
//        ui->comboBox->addItem(info.name);
//        ui->comboBox->setItemData(itemsindex, QVariant::fromValue(info));
//    }

    //init slider
//    int min = 1;
//    int max = 120;
//    int singlestep = 1;

//    ui->idleSlider->setMinimum(min);
//    ui->idleSlider->setMaximum(max);
//    ui->idleSlider->setSingleStep(singlestep);
//    ui->idleSlider->installEventFilter(this);

//    connect(this, SIGNAL(kill_signals()), this, SLOT(kill_screensaver_preview()));
}

void Screensaver::status_init(){
    int mode;
    char * name;

    screensaver_settings = g_settings_new(SCREENSAVER_SCHEMA);
    mode = g_settings_get_enum(screensaver_settings, MODE_KEY);
    if (mode == MODE_DEFAULT_UKUI){
        ui->comboBox->setCurrentIndex(0); //UKUI
    }
    else if(mode == MODE_BLANK_ONLY){
        ui->comboBox->setCurrentIndex(1); //Black_Only
    }
    else if (mode == MODE_RANDOM){
        ui->comboBox->setCurrentIndex(2); //Random
    }
    else{
        gchar ** strv;
        strv = g_settings_get_strv(screensaver_settings, THEMES_KEY);

        if (strv != NULL){
            name = g_strdup(strv[0]);
            QString nameqt = QString(name);
            SSThemeInfo info = (SSThemeInfo)infoMap.find(name).value();
            ui->comboBox->setCurrentText(info.name);
        }
        else
            ui->comboBox->setCurrentIndex(0); //no data, default Blank_Only
        g_strfreev(strv);
    }

    //init
//    bool activation; bool lockable;
//    activation = g_settings_get_boolean(screensaver_settings, ACTIVE_KEY);
//    activeswitchbtn->setChecked(activation);
//    if (activation){
//        lockable = g_settings_get_boolean(screensaver_settings, LOCK_KEY);
//        lockswitchbtn->setChecked(lockable);
//        ui->widget->show();
//    }
//    else{
//        lockswitchbtn->setChecked(false);
//        ui->widget->hide();
//    }

    g_object_unref(screensaver_settings);

    //获取空闲时间
    int minutes;
    session_settings = g_settings_new(SESSION_SCHEMA);
    minutes = g_settings_get_int(session_settings, IDLE_DELAY_KEY);
//    ui->idleSlider->setValue(minutes);
//    ui->idleLabel->setText(QString("%1%2").arg(minutes).arg(tr("minutes")));

    g_object_unref(session_settings);

    //获取功能列表
//    PublicData * publicdata = new PublicData();
//    QStringList tmpList = publicdata->subfuncList[SYSTEM];

    //connect
//    connect(ui->powerBtn, &QPushButton::clicked, this, [=]{pluginWidget->emitting_toggle_signal(tmpList.at(2), SYSTEM, 0);});

    connect(ui->idleSlider, SIGNAL(sliderReleased()), this, SLOT(slider_released_slot())); //改gsettings
//    connect(activeswitchbtn, SIGNAL(checkedChanged(bool)), this, SLOT(activebtn_changed_slot(bool)));
//    connect(lockswitchbtn, SIGNAL(checkedChanged(bool)), this, SLOT(lockbtn_changed_slot(bool)));
    connect(ui->comboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(combobox_changed_slot(int)));

    connect(mPreviewWidget, SIGNAL(destroyed(QObject*)), this, SLOT(kill_screensaver_preview()));

//    delete publicdata;
}

#define UKUI_SCREENSAVER_BIN "/usr/lib/ukui-screensaver/ukui-screensaver-default"
void Screensaver::startupScreensaver(){
    //关闭屏保
    closeScreensaver();
    qDebug() << Q_FUNC_INFO << mPreviewWidget->winId();
    if (ui->comboBox->currentIndex() == 0){//UKUI
        QStringList args;
        args << "-window-id" << QString::number(mPreviewWidget->winId());
        //启动屏保
        process->startDetached(UKUI_SCREENSAVER_BIN, args);
        runStringList.append(UKUI_SCREENSAVER_BIN);
    }
    else if (ui->comboBox->currentIndex() == 1){//黑屏
        mPreviewWidget->update();
    }
    else if (ui->comboBox->currentIndex() == 2){//随机
        mPreviewWidget->update();
    }
    else{//屏保

        SSThemeInfo info = ui->comboBox->currentData().value<SSThemeInfo>();
        QStringList args;
        args << "-window-id" << QString::number(mPreviewWidget->winId());
        //启动屏保
        process->startDetached(info.exec, args);
        runStringList.append(info.exec);
    }

}

void Screensaver::closeScreensaver(){
    //杀死分离启动的屏保预览程序
    if (!runStringList.isEmpty()){
        process->start(QString("killall"), runStringList);
        process->waitForStarted();
        process->waitForFinished(2000);

        runStringList.clear();
    }
}

void Screensaver::kill_and_start(){
    emit kill_signals(); //如果有屏保先杀死
    if (ui->comboBox->currentIndex() == 0){//UKUI
        QStringList args;
        args << "-window-id" << QString::number(mPreviewWidget->winId());
        //启动屏保
        process->startDetached(UKUI_SCREENSAVER_BIN, args);
        runStringList.append(UKUI_SCREENSAVER_BIN);
    }
    else if (ui->comboBox->currentIndex() == 1){//黑屏
        mPreviewWidget->update();
    }
    else if (ui->comboBox->currentIndex() == 2){//随机
        mPreviewWidget->update();
    }
    else{//屏保
        SSThemeInfo info = ui->comboBox->currentData().value<SSThemeInfo>();
        QStringList args;
        args << "-window-id" << QString::number(mPreviewWidget->winId());
        //启动屏保
        process->startDetached(info.exec, args);
        killList.clear(); killList.append(info.exec);
    }
}


void Screensaver::set_idle_gsettings_value(int value){
    session_settings = g_settings_new(SESSION_SCHEMA);
    g_settings_set_int(session_settings, IDLE_DELAY_KEY, value);
}

void Screensaver::slider_released_slot(){
    int minutes;
    minutes = ui->idleSlider->value();
    set_idle_gsettings_value(minutes);
}

void Screensaver::lockbtn_changed_slot(bool status){
    //REVIEW***  setchecked(false) -> g_object_unref faild
//    screensaver_settings = g_settings_new(SCREENSAVER_SCHEMA);
//    g_settings_set_boolean(screensaver_settings, LOCK_KEY, status);
//    if (screensaver_settings)
//            g_object_unref(screensaver_settings);
    const QByteArray ba(SCREENSAVER_SCHEMA);
    QGSettings * settings = new QGSettings(ba);
    settings->set(LOCK_KEY, status);
    delete settings;
}

void Screensaver::activebtn_changed_slot(bool status){
    screensaver_settings = g_settings_new(SCREENSAVER_SCHEMA);
    g_settings_set_boolean(screensaver_settings, ACTIVE_KEY, status);

//    if (status){
//        ui->widget->show();
//    }
//    else{
//        lockswitchbtn->setChecked(false);
//        ui->widget->hide();

//    }
    g_object_unref(screensaver_settings);

}

void Screensaver::combobox_changed_slot(int index){
    char ** strv = NULL;


    screensaver_settings = g_settings_new(SCREENSAVER_SCHEMA);
    if (index == 0){ //ukui
        g_settings_set_enum(screensaver_settings, MODE_KEY, MODE_DEFAULT_UKUI);
    }
    else if (index == 1){ //Blank_Only
        g_settings_set_enum(screensaver_settings, MODE_KEY, MODE_BLANK_ONLY);
    }
    else if (index == 2){ //Random
//        int mode = MODE_RANDOM;
        g_settings_set_enum(screensaver_settings, MODE_KEY, MODE_RANDOM);
        //REVIEW*** 二维字符数组赋值字符串段错误？
//        QMap<QString, SSThemeInfo>::iterator it = infoMap.begin();
//        for (guint num = 0; it != infoMap.end(); it++){
//            QString id = QString(it.key());
//            QByteArray ba = id.toLatin1();
//            char * info_id = ba.data();
//            strv[num++] = g_strdup)(info_id);
//        }
        //改用qt的gsetting
        QStringList valueStringList;
        const QByteArray ba(SCREENSAVER_SCHEMA);
        QGSettings * settings = new QGSettings(ba);
        QMap<QString, SSThemeInfo>::iterator it = infoMap.begin();
        for (; it != infoMap.end(); it++){
            QString id = QString(it.key());
            valueStringList.append(id);
        }
        settings->set(THEMES_KEY, QVariant(valueStringList));
        delete settings;
    }
    else{
        g_settings_set_enum(screensaver_settings, MODE_KEY, MODE_SINGLE);
        //获取当前屏保的id
        QVariant variant = ui->comboBox->itemData(index);
        SSThemeInfo info = variant.value<SSThemeInfo>();
        QByteArray ba = info.id.toLatin1();
        strv = g_strsplit(ba.data(), "%%%", 1);
        qDebug() << Q_FUNC_INFO << "wxy-----------" <<strv;
        g_settings_set_strv(screensaver_settings, THEMES_KEY, (const gchar * const*)strv);
    }
    //set mode
//    g_settings_set_enum(screensaver_settings, MODE_KEY, mode);
    //set themes
//    g_settings_set_strv(screensaver_settings, THEMES_KEY, (const gchar * const*)strv);

    g_object_unref(screensaver_settings);
    g_strfreev(strv);

    //启动屏保
    kill_and_start();
}

void Screensaver::kill_screensaver_preview(){
    //需要手动杀死分离启动的屏保预览程序
//    if (!killList.isEmpty()){
//        process->start(QString("killall"), killList);
//        process->waitForStarted();
//        process->waitForFinished(2000);

//        killList.clear();
//    }
}

SSThemeInfo Screensaver::_info_new(const char *path){
    SSThemeInfo info;
    GKeyFile * keyfile;
    char * name, * exec;

    keyfile = g_key_file_new();
    if (!g_key_file_load_from_file(keyfile, path, G_KEY_FILE_NONE, NULL)){
        g_key_file_free (keyfile);
        return info;
    }

    name = g_key_file_get_locale_string(keyfile, G_KEY_FILE_DESKTOP_GROUP, G_KEY_FILE_DESKTOP_KEY_NAME, NULL, NULL);
    exec = g_key_file_get_string(keyfile, G_KEY_FILE_DESKTOP_GROUP, G_KEY_FILE_DESKTOP_KEY_TRY_EXEC, NULL);

    info.name = QString::fromUtf8(name);
    info.exec = QString::fromUtf8(exec);
    info.id = ID_PREFIX + info.name.toLower();

    g_key_file_free(keyfile);

    return info;
}

SSThemeInfo Screensaver::_newThemeinfo(const char * path){
    SSThemeInfo info;
    GKeyFile * keyfile;
    char * name, * exec;

    keyfile = g_key_file_new();
    if (!g_key_file_load_from_file(keyfile, path, G_KEY_FILE_NONE, NULL)){
        g_key_file_free (keyfile);
        return info;
    }

    name = g_key_file_get_locale_string(keyfile, G_KEY_FILE_DESKTOP_GROUP, G_KEY_FILE_DESKTOP_KEY_NAME, NULL, NULL);
    exec = g_key_file_get_string(keyfile, G_KEY_FILE_DESKTOP_GROUP, G_KEY_FILE_DESKTOP_KEY_TRY_EXEC, NULL);


    info.name = QString::fromUtf8(name);
    info.exec = QString::fromUtf8(exec);
    info.id = ID_PREFIX + info.name.toLower();

    g_key_file_free(keyfile);

    return info;
}

void Screensaver::_acquireThemeinfoList(){
    GDir * dir;
    const char * name;

    infoMap.clear();

    dir = g_dir_open(SSTHEMEPATH, 0, NULL);
    if (!dir)
        return;
    while ((name = g_dir_read_name(dir))) {
        SSThemeInfo info;
        char * desktopfilepath;
        if (!g_str_has_suffix(name, ".desktop"))
            continue;
        desktopfilepath = g_build_filename(SSTHEMEPATH, name, NULL);
        info = _newThemeinfo(desktopfilepath);
        infoMap.insert(info.id, info);
        g_free (desktopfilepath);
    }
    g_dir_close(dir);
}

void Screensaver::init_theme_info_map(){
    GDir * dir;
    const char * name;

    infoMap.clear();

    dir = g_dir_open(SSTHEMEPATH, 0, NULL);
    if (!dir)
        return;
    while ((name = g_dir_read_name(dir))) {
        SSThemeInfo info;
        char * desktopfilepath;
        if (!g_str_has_suffix(name, ".desktop"))
            continue;
        desktopfilepath = g_build_filename(SSTHEMEPATH, name, NULL);
        info = _info_new(desktopfilepath);
        infoMap.insert(info.id, info);
        g_free (desktopfilepath);
    }
    g_dir_close(dir);
}
