/*
 * Copyright 2013-2014 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 3 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ubuntupurchase.h"
#include <QUrl>
#include <QUrlQuery>
#include <QDBusConnection>
#include <QProcessEnvironment>

#ifdef TESTS
#include "tests/fakedbus.h"
#endif

#define PURCHASE_URL "https://software-center.ubuntu.com/api/2.0/click-purchase/"

namespace UbuntuPurchases {

UbuntuPurchase::UbuntuPurchase(QQuickItem *parent):
    QQuickItem(parent)
{
    this->m_service = nullptr;
    this->m_serviceAdaptor = nullptr;
#ifdef TESTS
    QObject::connect(&m_ssoService, &FakeSsoService::credentialsFound,
                  this, &UbuntuPurchase::handleCredentialsFound);
    QObject::connect(&m_ssoService, &FakeSsoService::credentialsNotFound,
                  this, &UbuntuPurchase::handleCredentialsNotFound);
#else
    QObject::connect(&m_ssoService, &SSOService::credentialsFound,
                  this, &UbuntuPurchase::handleCredentialsFound);
    QObject::connect(&m_ssoService, &SSOService::credentialsNotFound,
                  this, &UbuntuPurchase::handleCredentialsNotFound);
#endif
}

UbuntuPurchase::~UbuntuPurchase()
{
}

void UbuntuPurchase::launchService()
{
    if(this->m_service == nullptr && this->m_serviceAdaptor == nullptr) {
        this->m_service = new PurchaseService(this);
        this->m_serviceAdaptor = new PurchaseServiceAdaptor(this->m_service);

#ifdef TESTS
        FakeDbus connection(this);
#else
        QDBusConnection connection = QDBusConnection::sessionBus();
#endif
        bool valid = connection.registerService("com.canonical.Purchase");
        valid = valid && connection.registerObject("/", this->m_service);

        if(!valid) {
            Q_EMIT this->serviceRegistrationFailed();
            return;
        }

        QObject::connect(this->m_service, &PurchaseService::purchaseItem,
                      this, &UbuntuPurchase::setPurchaseItemId);

        this->m_ssoService.getCredentials();
    }
}

void UbuntuPurchase::setPurchaseItemId(QString& id)
{
    this->m_itemPurchaseId = id;
    this->triggerPurchase();
}

void UbuntuPurchase::handleCredentialsFound(Token token)
{
    qDebug() << "Credentials found.";
    this->m_token = token;
    this->triggerPurchase();
}

void UbuntuPurchase::handleCredentialsNotFound()
{
    qDebug() << "No credentials were found.";
    Q_EMIT this->credentialsNotFound();
}

QString UbuntuPurchase::getPurchaseUrl() const
{
    QProcessEnvironment environment = QProcessEnvironment::systemEnvironment();
    QString url = environment.value("PURCHASE_URL", QString(PURCHASE_URL));
    return url;
}

void UbuntuPurchase::triggerPurchase()
{
    if(this->m_token.isValid() && !this->m_itemPurchaseId.isEmpty()) {
        QString url = this->getPurchaseUrl();
        QUrlQuery query;
        query.addQueryItem(QString("package_name"), this->m_itemPurchaseId);
        QUrl storeUrl(url);
        storeUrl.setQuery(query);
        QString authHeader = this->m_token.signUrl(storeUrl.toString(), QStringLiteral("GET"), true);
        storeUrl.setQuery(authHeader);
        this->m_url = storeUrl.toString();
        Q_EMIT this->urlChanged();
    }
}

void UbuntuPurchase::purchaseSucceeded()
{
    const QString packageName = this->m_service->getPackageName();
    Q_EMIT this->m_service->success(packageName);
}

void UbuntuPurchase::purchaseCanceled()
{
    const QString packageName = this->m_service->getPackageName();
    Q_EMIT this->m_service->failure(packageName);
}

}
