#This file is part of Tryton.  The COPYRIGHT file at the top level of
#this repository contains the full copyright notices and license terms.
from functools import wraps
from sql import Table

from trytond.model import ModelView, Workflow, fields
from trytond.pool import Pool, PoolMeta
from trytond.transaction import Transaction
from trytond import backend

__all__ = ['Invoice', 'InvoiceLine']
__metaclass__ = PoolMeta


def process_purchase(func):
    @wraps(func)
    def wrapper(cls, invoices):
        pool = Pool()
        Purchase = pool.get('purchase.purchase')
        func(cls, invoices)
        with Transaction().set_user(0, set_context=True):
            Purchase.process([p for i in cls.browse(invoices)
                    for p in i.purchases])
    return wrapper


class Invoice:
    __name__ = 'account.invoice'
    purchase_exception_state = fields.Function(fields.Selection([
        ('', ''),
        ('ignored', 'Ignored'),
        ('recreated', 'Recreated'),
        ], 'Exception State'), 'get_purchase_exception_state')
    purchases = fields.Many2Many('purchase.purchase-account.invoice',
            'invoice', 'purchase', 'Purchases', readonly=True)

    @classmethod
    def __setup__(cls):
        super(Invoice, cls).__setup__()
        cls._error_messages.update({
                'delete_purchase_invoice': ('You can not delete invoices '
                    'that come from a purchase.'),
                'reset_invoice_purchase': ('You cannot reset to draft '
                    'an invoice generated by a purchase.'),
                })

    @classmethod
    def get_purchase_exception_state(cls, invoices, name):
        Purchase = Pool().get('purchase.purchase')
        with Transaction().set_user(0, set_context=True):
            purchases = Purchase.search([
                    ('invoices', 'in', [i.id for i in invoices]),
                    ])

        recreated = tuple(i for p in purchases for i in p.invoices_recreated)
        ignored = tuple(i for p in purchases for i in p.invoices_ignored)

        res = {}
        for invoice in invoices:
            if invoice in recreated:
                res[invoice.id] = 'recreated'
            elif invoice in ignored:
                res[invoice.id] = 'ignored'
            else:
                res[invoice.id] = ''
        return res

    @classmethod
    def delete(cls, invoices):
        pool = Pool()
        Purchase_Invoice = pool.get('purchase.purchase-account.invoice')
        purchase_invoice = Purchase_Invoice.__table__()
        cursor = Transaction().cursor
        if invoices:
            cursor.execute(*purchase_invoice.select(purchase_invoice.id,
                    where=purchase_invoice.invoice.in_(
                        [i.id for i in invoices])))
            if cursor.fetchone():
                cls.raise_user_error('delete_purchase_invoice')
        super(Invoice, cls).delete(invoices)

    @classmethod
    def copy(cls, invoices, default=None):
        if default is None:
            default = {}
        default = default.copy()
        default.setdefault('purchases', None)
        return super(Invoice, cls).copy(invoices, default=default)

    @classmethod
    @ModelView.button
    @Workflow.transition('draft')
    def draft(cls, invoices):
        Purchase = Pool().get('purchase.purchase')
        with Transaction().set_user(0, set_context=True):
            purchases = Purchase.search([
                    ('invoices', 'in', [i.id for i in invoices]),
                    ])
        if purchases and any(i.state == 'cancel' for i in invoices):
            cls.raise_user_error('reset_invoice_purchase')

        return super(Invoice, cls).draft(invoices)

    @classmethod
    @process_purchase
    def post(cls, invoices):
        super(Invoice, cls).post(invoices)

    @classmethod
    @process_purchase
    def paid(cls, invoices):
        super(Invoice, cls).paid(invoices)

    @classmethod
    @process_purchase
    def cancel(cls, invoices):
        super(Invoice, cls).cancel(invoices)


class InvoiceLine:
    __name__ = 'account.invoice.line'

    @classmethod
    def __register__(cls, module_name):
        TableHandler = backend.get('TableHandler')
        cursor = Transaction().cursor
        sql_table = cls.__table__()

        super(InvoiceLine, cls).__register__(module_name)

        # Migration from 2.6: remove purchase_lines
        rel_table_name = 'purchase_line_invoice_lines_rel'
        if TableHandler.table_exist(cursor, rel_table_name):
            rel_table = Table(rel_table_name)
            cursor.execute(*rel_table.select(
                    rel_table.purchase_line, rel_table.invoice_line))
            for purchase_line, invoice_line in cursor.fetchall():
                cursor.execute(*sql_table.update(
                        columns=[sql_table.origin],
                        values=['purchase.line,%s' % purchase_line],
                        where=sql_table.id == invoice_line))
            TableHandler.drop_table(cursor,
                'purchase.line-account.invoice.line', rel_table_name)

    @property
    def origin_name(self):
        pool = Pool()
        PurchaseLine = pool.get('purchase.line')
        name = super(InvoiceLine, self).origin_name
        if isinstance(self.origin, PurchaseLine):
            name = self.origin.purchase.rec_name
        return name

    @classmethod
    def _get_origin(cls):
        models = super(InvoiceLine, cls)._get_origin()
        models.append('purchase.line')
        return models
