/*	Connect_View

PIRL CVS ID: Connect_View.java,v 1.11 2012/04/16 06:08:56 castalia Exp

Copyright (C) 2005-2007  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Database;

import PIRL.PVL.*;
import PIRL.Viewers.*;
import PIRL.Configuration.*;

import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
import java.util.Iterator;
import java.io.File;
import java.io.IOException;

/**	<I>Connect_View</I> provides a dialog used to interactively obtain the
	information needed to make a connection to a Database.
<p>
	A Connect_View dialog, which may be modal or non-modal, may contain
	a user supplied decorative icon on the left side of the panel. The
	right side of the panel contains a list of label fields used to
	gather the database connection parameters. Buttons across the bottom
	allow the user to Connect - accept the current connection parameters
	and close the dialog - Clear - clear all connection parameters - or
	Cancel - reject all parameters and close the dialog.
<p>
	A user supplied Configuration provides an initial source of database
	connection parameters. If no Configuration is supplied, an attempt
	is made to read the {@link Database#DEFAULT_CONFIGURATION_FILENAME
	default configuration file}. If this fails the user is offered the
	opportunity to browse for a configuration file to read. If this is
	rejected an empty Configuration is used.
<p>
	A specific set of Server configuration parameters is assembled from
	the connection parameter fields of the dialog. There are two
	categories of fields:
<h4>
	Required Fields
</h4><p>
	The fields for the required parameters will always be present. They
	are drawn from the user supplied Configuration:
	<dl>
	<dt>Server
		<dd>The database {@link Database#SERVER Server} parameters. This
		field is a drop-down list of each valid Server group; the user
		is notified when a Server list name does not have a
		corresponding group of parameters in the Configuration.
		Selecting an entry in the Server list results in the parameters
		of the Server group being used to set the values of all other
		fields, both required and additional, from each parameter that
		has a name matching one of the field names. The first Server
		name is used by default.
	<p>
		<b>Note</b>: When a parameter name in a selected Server group matches
		an additional field name, the parameter value is used to set the
		field value only when the field is currently empty or its value
		is not the user specified value.
	<p>
		<b>Note</b>: When the Server field is cleared the required
		fields are also cleared, but not any additional fields.
	<dt>Type
		<dd>The type of database server. This is used to select the
		appropriate {@link Data_Port} for managing the connection to
		the database server. A typical database type is MySQL. An entry
		in this field is required.
	<dt>Host
		<dd>The hostname of the computer system where the database server
		can be contacted. An entry in this field is required.
	<dt>User
		<dd>The username to supply when making a connection to the
		database server. Depending on the database connection
		requirements, this may be left empty.
	<dt>Password
		<dd>The password to supply when making a connection to the
		database server. The entry in this field will be masked.
		Depending on the database connection requirements, this may be
		left empty.
	</dl>
<h4>
	Additional Fields
</h4><p>
	An optional list containing the names of additional connection
	parameters to be obtained may be specified. Each field name may
	optionally be paired with an initial value for the field. All
	duplicate field names, including names that are the same (ignoring
	case) as any of the required field names, are removed leaving only
	the first unique name.
<h4>
	Buttons
</h4><p>
	<dl>
	<dt>Connect
		<dd>Completion of user connection parameter entries. The field
		values are assembled into a server specific Configuration
		suitable for configuring a {@link Database} object. This
		Configuration will be named for the selected Server.
		<b>N.B.</b>: If the Server field is empty the Server name will
		be "unnamed". The Configuration will contain a {@link
		Database#SERVER Server} parameter with a value that is the
		selected Server name and a group with the same name containing a
		parameter for each field where the name of each parameter is the
		field name and its value is the contents of the field. The
		Connect_View dialog will be closed (setVisible (false)).
	<p>
		<b>N.B.</b>: If the Type or Host field is empty a notice
		dialog will be preseted, the server specific Configuration will
		not be assembled, and the Connect_View dialog will not close.
	<dt>Reset
		<dd>Reset the additional, non-required fields to their
		original values. If there are noe user specified additional
		fields this button will not be present.
	<dt>Clear
		<dd>All fields, both required and any additional, are cleared.
	<dt>Cancel
		<dd>The server configuration is set to null to indicate that
		the database connection was canceled and the Connect_View
		dialog is closed.
	</dl>
<p>
	Connect_View is a reusable dialog. Once constructed it only needs to
	be setVisible to be activated. If it had been used previously the
	previous field values will remain. The current server configuration
	is renewed after each use unless the Type or Host fields are empty
	when Connect is pressed.
<p>
	@author	Bradford Castalia, UA/PIRL
	@version 1.11
	@see	Database
	@see	Configuration
*/
public class Connect_View
	extends JDialog
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Database.Connect_View (1.11 2012/04/16 06:08:56)";

//	Configuration.
private Configuration
	User_Configuration,
	Server_Configuration		= null;

//	Parameter names:
private static final String
	PASSWORD_PARAMETER			= "password",
	MASKED_VALUE				= "*******";

//	Icon displayed on the dialog box.
private Icon
	Decorative_Icon				= null;

//	Server configuration selection.
private String
	Selected_Server				= "";
private Vector
	Servers;
private	JComboBox
	Servers_List				= new JComboBox ();

//	Required information fields.
private	JTextField
	Server_Type_Field			= new JTextField (),
	Hostname_Field				= new JTextField (),
	Username_Field				= new JTextField ();
private JPasswordField
	Password_Field				= new JPasswordField ();
	
//	Additonal information fields.
private Vector
	Additional_Field_Names,
	Additional_Field_Values		= new Vector (),
	Additional_Fields			= new Vector ();

// Current Working Directory.
private String
	CWD							= System.getProperty ("user.dir");
//	System new-line.
private static final String
	NL							= Database.NL;


//  DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_SETUP		= 1 << 0,
	DEBUG_GET		= 1 << 1,
	DEBUG_CONFIG	= 1 << 2,
	DEBUG_UI		= 1 << 3,
	DEBUG_HELPERS	= 1 << 4,
	DEBUG_ALL		= -1,

	DEBUG       	= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a Database Connect_View dialog.
<p>
	@param	title	The title String for the dialog window.
	@param	configuration	The Configuration containing default
		Database access parameters. If null the {@link
		Database#DEFAULT_CONFIGURATION_FILENAME default configuration
		file} will be used. If this can found or read the user will be
		offered the opportunity to browser for a configuration file. If
		rejected, an empty Configuration will be supplied.
	@param	icon	The Icon to be displayed on the dialog. If null no
		Icon is displayed.
	@param	additional_field_names	A Vector of String values (or any
		Object that can be converted to a String) which are the names of
		additional parameters that are to be obtained beyond the
		required parameters. Instead of a String entry a Vector
		containing a pair of String values (or any Objects that can be
		converted to Strings) may be used in which the first value is
		the parameter name and the second value is its initial value.
		The Vector may be empty or null.
	@param	owner	The Frame with which the dialog is associated,
		which may be null.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public Connect_View
	(
	String			title,
	Configuration	configuration,
	Icon			icon,
	Vector			additional_field_names,
	Frame			owner,
	boolean			modal
	)
{
super (owner, (title == null ? "Database Connection" : title), modal);
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">>> Connect_View");

Decorative_Icon = icon;

if ((Additional_Field_Names = additional_field_names) == null)
	Additional_Field_Names = new Vector ();
Initialize_Additional_Values ();

//	Setup the configuration.
Configure (configuration);

//	Build the panels.
getContentPane ().add (Panels (), BorderLayout.CENTER);

//	Initialize the field values.
Server_Selected ();

pack ();
getRootPane ().setMinimumSize (getContentPane ().getSize ());
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("<<< Connect_View");
}

/**	Constructs a Database Connect_View dialog.
<p>
	@param	title	The title String for the dialog window.
	@param	configuration	The Configuration containing default
		Database access parameters. If null the {@link
		Database#DEFAULT_CONFIGURATION_FILENAME default configuration
		file} will be used. If this can found or read the user will be
		offered the opportunity to browser for a configuration file. If
		rejected, an empty Configuration will be supplied.
	@param	icon	The Icon to be displayed on the dialog. If null no
		Icon is displayed.
	@param	additional_field_names	A Vector of String values (or any
		Object that can be converted to a String) which are the names of
		additional parameters that are to be obtained beyond the
		required parameters. Instead of a String entry a Vector
		containing a pair of String values (or any Objects that can be
		converted to Strings) may be used in which the first value is
		the parameter name and the second value is its initial value.
		The Vector may be empty or null.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public Connect_View
	(
	String			title,
	Configuration	configuration,
	Icon			icon,
	Vector			additional_field_names,
	boolean			modal
	)
{this (title, configuration, icon, additional_field_names, null, modal);}

/**	Constructs a Database Connect_View dialog.
<p>
	@param	title	The title String for the dialog window.
	@param	configuration	The Configuration containing default
		Database access parameters. If null the {@link
		Database#DEFAULT_CONFIGURATION_FILENAME default configuration
		file} will be used. If this can found or read the user will be
		offered the opportunity to browser for a configuration file. If
		rejected, an empty Configuration will be supplied.
	@param	icon	The Icon to be displayed on the dialog. If null no
		Icon is displayed.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public Connect_View
	(
	String			title,
	Configuration	configuration,
	Icon			icon,
	boolean			modal
	)
{this (title, configuration, icon, null, null, modal);}

/**	Constructs a Database Connect_View dialog.
<p>
	@param	title	The title String for the dialog window.
	@param	configuration	The Configuration containing default
		Database access parameters. If null the {@link
		Database#DEFAULT_CONFIGURATION_FILENAME default configuration
		file} will be used. If this can found or read the user will be
		offered the opportunity to browser for a configuration file. If
		rejected, an empty Configuration will be supplied.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public Connect_View
	(
	String			title,
	Configuration	configuration,
	boolean			modal
	)
{this (title, configuration, null, null, null, modal);}

/**	Constructs a Database Connect_View dialog.
<p>
	@param	title	The title String for the dialog window.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public Connect_View
	(
	String			title,
	boolean			modal
	)
{this (title, null, null, null, null, modal);}

/**	Constructs a Database Connect_View dialog.
<p>
	@param	configuration	The Configuration containing default
		Database access parameters. If null the {@link
		Database#DEFAULT_CONFIGURATION_FILENAME default configuration
		file} will be used. If this can found or read the user will be
		offered the opportunity to browser for a configuration file. If
		rejected, an empty Configuration will be supplied.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public Connect_View
	(
	Configuration	configuration,
	boolean			modal
	)
{this (null, configuration, null, null, null, modal);}

/**	Constructs a default Connect_View.
*/
public Connect_View ()
{this (null, null, null, null, false);}

/*------------------------------------------------------------------------------
	PUBLIC INTERFACE
*/
/**	Gets a Database Configuration.
<p>
	@param	title	The title String for the dialog window.
	@param	configuration	The Configuration containing default
		Database access parameters. If null the {@link
		Database#DEFAULT_CONFIGURATION_FILENAME default configuration
		file} will be used. If this can found or read the user will be
		offered the opportunity to browser for a configuration file. If
		rejected, an empty Configuration will be supplied.
	@param	icon	The Icon to be displayed on the dialog. If null no
		Icon is displayed.
	@param	additional_field_names	A Vector of String values (or any
		Object that can be converted to a String) which are the names of
		additional parameters that are to be obtained beyond the
		required parameters. Instead of a String entry a Vector
		containing a pair of String values (or any Objects that can be
		converted to Strings) may be used in which the first value is
		the parameter name and the second value is its initial value.
		The Vector may be empty or null.
	@param	parent	The parent window that dialog is associated with,
		which may be null.
	@return	A Configuration containing the parameters obtained, which
		should be suitable for configuring a Database to make the
		user defined connection. This will be null if the user
		Canelled the dialog.
*/
public static Configuration Get_Database_Configuration
	(
	String			title,
	Configuration	configuration,
	Icon			icon,
	Vector			additional_field_names,
	JFrame			parent
	)
{
if ((DEBUG & DEBUG_GET) != 0)
	System.out.println
		(">>> Connect_View.Get_Database_Configuration");
Connect_View
	connect_view = new Connect_View
		(title, configuration, icon, additional_field_names, parent, true);
connect_view.setVisible (true);
if ((DEBUG & DEBUG_GET) != 0)
	System.out.println
		("<<< Connect_View.Get_Database_Configuration");
return connect_view.Server_Configuration;
}

/**	Gets the current database server connection parameters.
<p>
	@return	A Configuration containing the current database connection
		parameters. This may be null.
*/
public Configuration Database_Configuration ()
{return Server_Configuration;}

/*==============================================================================
	Configuration

	The Configuration is expected to have a list of database server
	names and a Group of parameters for each named database server.
*/
private void Configure
	(
	Configuration	configuration
	)
{
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">>> Connect_View.Configure");
if (configuration == null)
	{
	//	Try the default configuration.
	try {configuration =
			new Configuration (Database.DEFAULT_CONFIGURATION_FILENAME);}
	catch (Exception exception)
		{
		if (Dialog_Box.Confirm (
			"Unable to configure with the default file "
				+ Database.DEFAULT_CONFIGURATION_FILENAME + NL
			+ NL
			+ exception.getMessage () + NL
			+ NL
			+ "Would you like to browse for a configuration file?"))
			//	File selection dialog.
			configuration = Open_Configuration ();
		else
			//	Use an empty configuration.
			configuration = new Configuration ();
		}
	}

User_Configuration = configuration;
boolean
	case_sensitive = configuration.Case_Sensitive (false);

//	Database Servers.
Servers = configuration.Get
	(Configuration.Absolute_Pathname (Database.SERVER));
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("          Servers: " + Servers);

//	Validate all the server groups.
int
	index = Servers.size ();
while (--index >= 0)
	{
	try
		{
		if (Server_Group ((String)Servers.get (index)) == null)
			//	Remove invalid server names.
			Servers.remove (index);
		}
	catch (Configuration_Exception e)
		{
		Servers.remove (index);
		}
	}
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("    Valid Servers: " + Servers);

configuration.Case_Sensitive (case_sensitive);
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("<<< Connect_View.Configure");
}


private Configuration Open_Configuration ()
{
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println (">>> Connect_View.Open_Configuration");
Configuration
	configuration;
JFileChooser
	file_chooser = new JFileChooser (CWD);
file_chooser.setFileSelectionMode (JFileChooser.FILES_ONLY);
while (true)
	{
	if (file_chooser.showOpenDialog (rootPane)
		== JFileChooser.APPROVE_OPTION)
		{
		File
			file = file_chooser.getSelectedFile ();
		try
			{
			file = file.getCanonicalFile ();
			CWD = file.getParent ();
			try
				{
				configuration = new Configuration (file.getPath ());
				break;
				}
			catch (Configuration_Exception except)
				{
				Dialog_Box.Error
					("Unable to use the file -" + NL
					+ file.getPath () + NL
					+ NL
					+ except.getMessage (),
					this);
				}
			}
		catch (IOException exception)
			{
			Dialog_Box.Error
				("Unable to obtain the canonical pathname" + NL
				+"for the selected file -" + NL
				+ file.getPath () + NL
				+ NL
				+ exception.getMessage (),
				this);
			}
		}
	else
		{
		configuration = new Configuration ();
		break;
		}
	}
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println ("<<< Connect_View.Open_Configuration");
return configuration;
}

/**	Gets the Configuration for a specific Server.
<p>
	@param	server	The name of the group of parameters in the current
		Configuration that is expected to contain database server access
		parameters.
	@return	The Configuration for the server group of parameters, or
		null if the server group was not found.
*/
public Configuration Server_Configuration
	(
	String	server
	)
{
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">>> Connect_View.Server_Configuration: " + server);
Configuration
	server_configuration = null;
if (server != null)
	{
	try
		{
		if ((server_configuration = Server_Group (server)) == null)
			Dialog_Box.Notice
				("Database server group \"" + server + '"' + NL
				+"is not in the configuration.",
				this);
		}
	catch (Configuration_Exception exception)
		{
		Dialog_Box.Error
			("Database server group \"" + server + '"' + NL
			+"could not be assembled from the configuration." + NL
			+ NL
			+ ID + NL
			+ exception.getMessage (),
			this);
		}
	}
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("<<< Connect_View.Server_Configuration:" + NL
		+ ((server_configuration == null) ?
			"not found" : server_configuration.Description ()));
return server_configuration;	
}


private Configuration Server_Group
	(
	String	server
	)
	throws Configuration_Exception
{
try
	{
	return
		User_Configuration.Group
			(Configuration.Absolute_Pathname (server));
	}
catch (Configuration_Exception exception)
	{
	Dialog_Box.Error
		("Unable to assemble the \"" + server
			+ "\" Configuration Group." + NL
		+ NL
		+ ID + NL
		+ exception.getMessage (),
		this);
	throw exception;
	}
}

/*==============================================================================
	Panels
*/
private JPanel Panels ()
{
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println (">>> Connect_View.Panels");
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

//	Icon or empty space.
location.gridx		= GridBagConstraints.RELATIVE;
location.gridy		= GridBagConstraints.RELATIVE;
location.gridwidth	= 1;
location.gridheight	= 5 + Additional_Field_Names.size ();
location.weightx	= 0.0;
location.weighty	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.anchor		= GridBagConstraints.NORTH;
location.insets		= new Insets (10, 10, 10, 5);
if (Decorative_Icon == null)
	//	Place holder.
	panel.add (Box.createHorizontalGlue (), location);
else
	//	Decorative icon.
	panel.add (new JLabel (Decorative_Icon), location);


//	Servers.
location.gridheight	= 1;
location.anchor		= GridBagConstraints.EAST;
location.insets		= new Insets (10, 0, 5, 5);
panel.add (new JLabel("Server:"), location);

for (Iterator
		names = Servers.iterator ();
		names.hasNext ();
		Servers_List.addItem (names.next ()));
Servers_List.addActionListener(new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Server_Selected ();}});
Servers_List.setEditable (true);
Servers_List.setEnabled (true);
Servers_List.setToolTipText ("Server configuration group");
location.gridwidth	= GridBagConstraints.REMAINDER;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
location.insets		= new Insets (10, 0, 5, 10);
panel.add (Servers_List, location);
			
//	Type.
location.gridx		= 1;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 5, 5);
panel.add (new JLabel("Type:"), location);

Server_Type_Field.setToolTipText ("Type of database server");
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 5, 10);
panel.add (Server_Type_Field, location);

//	Hostname.
location.gridx		= 1;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 5, 5);
panel.add (new JLabel("Host:"), location);

Hostname_Field.setToolTipText ("Host name of the database server");
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 5, 10);
panel.add (Hostname_Field, location);

//	Username.
location.gridx		= 1;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 5, 5);
panel.add (new JLabel("User:"), location);

Username_Field.setToolTipText ("User name for the database server");
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 5, 10);
panel.add (Username_Field, location);

//	Password.
location.gridx		= 1;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 15, 5);
panel.add (new JLabel("Password:"), location);

Password_Field.setToolTipText ("Password for the database server");
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 15, 10);
panel.add (Password_Field, location);


//	Additional user fields.
Iterator
	names  = Additional_Field_Names.iterator (),
	values = Additional_Field_Values.iterator ();
while (names.hasNext ())
	{
	location.gridx		= 1;
	location.gridwidth	= 1;
	location.weightx	= 0.0;
	location.anchor		= GridBagConstraints.EAST;
	location.fill		= GridBagConstraints.NONE;
	location.insets		= new Insets (0, 0, 5, 5);
	panel.add (new JLabel((String)names.next () + ':'), location);

	JTextField field	= new JTextField ((String)values.next ());
	location.gridx		= GridBagConstraints.RELATIVE;
	location.gridwidth	= GridBagConstraints.REMAINDER;
	location.weightx	= 1.0;
	location.fill		= GridBagConstraints.HORIZONTAL;
	location.insets		= new Insets (0, 0, 5, 10);
	panel.add (field, location);
	Additional_Fields.add (field);
	}


//	Buttons panel:

JPanel
	button_panel = new JPanel (new GridBagLayout ());
JButton
	button;
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 0, 0);

//	Connect.
button				= new JButton ("Connect");
button.setMnemonic ('C');
button.setToolTipText ("Proceed with the database connection");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Connect (true);}});
button.setDefaultCapable (true);
getRootPane ().setDefaultButton (button);
location.anchor		= GridBagConstraints.WEST;
button_panel.add (button, location);

location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
button_panel.add (Box.createHorizontalGlue (), location);

if (Additional_Field_Names.size () != 0)
	{
	//	Reset.
	button				= new JButton ("Reset");
	button.setMnemonic ('R');
	button.setToolTipText ("Reset non-required fields");
	button.addActionListener (new ActionListener ()
		{public void actionPerformed (ActionEvent event)
			{Reset ();}});
	location.weightx	= 0.0;
	location.fill		= GridBagConstraints.NONE;
	location.anchor		= GridBagConstraints.CENTER;
	button_panel.add (button, location);

	location.weightx	= 1.0;
	location.fill		= GridBagConstraints.HORIZONTAL;
	button_panel.add (Box.createHorizontalGlue (), location);
	}

//	Clear.
button				= new JButton ("Clear");
button.setMnemonic ('l');
button.setToolTipText ("Clear all fields");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Clear (true);}});
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.anchor		= GridBagConstraints.CENTER;
button_panel.add (button, location);

location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
button_panel.add (Box.createHorizontalGlue (), location);

//	Cancel.
button		= new JButton ("Cancel");
button.setMnemonic ('n');
button.setToolTipText ("Cancel the database connection");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Connect (false);}});
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.anchor		= GridBagConstraints.EAST;
button_panel.add (button, location);

location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (10, 10, 10, 10);
panel.add (button_panel, location);

if ((DEBUG & DEBUG_UI) != 0)
	System.out.println ("<<< Connect_View.Panels");
return panel;
}

/*==============================================================================
	Actions
*/
//	Server Selected.
private void Server_Selected ()
{
String
	selected_server = (String)Servers_List.getSelectedItem ();
if (selected_server == null)
	selected_server = "";
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println
		(">>> Connect_View.Server_Selected: " + selected_server);

Configuration
	server_configuration = null;
if (selected_server.length () != 0)
	server_configuration = Server_Configuration (selected_server);
if (server_configuration != null)
	{
	//	Reset info fields from the configuration parameter values.
	Selected_Server = selected_server;
	if ((DEBUG & (DEBUG_UI | DEBUG_CONFIG)) != 0)
		System.out.println
			("    Connect_View.Server_Selected: Server Configuration -"
			+ server_configuration.Description ());
	Server_Type_Field.setText
		(server_configuration.Get_Specific_One (Database.TYPE));
	Hostname_Field.setText
		(server_configuration.Get_Specific_One (Configuration.HOST));
	Username_Field.setText
		(server_configuration.Get_Specific_One (Configuration.USER));
	Password_Field.setText
		(server_configuration.Get_Specific_One (PASSWORD_PARAMETER));

	for (int
			index = 0,
			total_additional_fields = Additional_Field_Names.size ();
			index < total_additional_fields;
			index++)
		{
		if ((DEBUG & (DEBUG_UI | DEBUG_CONFIG)) != 0)
			System.out.println
				("    Field " + index + ": " + Field_Name (index));
		String
			value = server_configuration.Get_Specific_One (Field_Name (index));
		if (value != null)
			{
			//	Server config contains additional field name parameter.
			String
				field_value = Field_Value (index);
			if ((DEBUG & (DEBUG_UI | DEBUG_CONFIG)) != 0)
				System.out.println
					("      Config value - " + value + NL
					+"       Field value - " + field_value + NL
					+"        User value - " + Additional_Field_Value (index));
			if (field_value.length () == 0 ||
				! field_value.equals (Additional_Field_Value (index)))
				{
				if ((DEBUG & (DEBUG_UI | DEBUG_CONFIG)) != 0)
					System.out.println
						("         New value - " + value);
				Field_Value (index, value);
				}
			}
		}
	int
		index = 0,
		total_items = Servers_List.getItemCount ();
	while (index < total_items)
		{
		if (selected_server.equals ((String)Servers_List.getItemAt (index)))
			break;
		++index;
		}
	if (index == total_items)
		{
		//	Add this unexpected server name to the list.
		Servers_List.addItem (selected_server);
		Servers_List.setSelectedIndex (index);
		}
	}
else
	{
	if (selected_server.length () == 0)
		//	Clear the server and required fields only.
		Clear (false);
	else
		//	Restore the previously selected server.
		Servers_List.setSelectedItem (Selected_Server);
	}
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println ("<<< Connect_View.Server_Selected");
}


private void Connect
	(
	boolean		connect
	)
{
if (connect)
	{
	Configuration
		server_configuration = new Configuration ();
	server_configuration.Remove_All ();	//	Clear out.
	server_configuration.Name ((Selected_Server.length () == 0) ?
		"unnamed" : Selected_Server);
	String
		value;
	try
		{
		if ((value = Server_Type_Field.getText ()).length () == 0)
			{
			Dialog_Box.Notice
				("The type of database server must be specified.",
				this);
			return;
			}
		server_configuration.Set (Database.TYPE, value);
		if ((value = Hostname_Field.getText ()).length () == 0)
			{
			Dialog_Box.Notice
				("The hostname for the database server must be specified.",
				this);
			return;
			}
		server_configuration.Set (Configuration.HOST, value);
		server_configuration.Set (Configuration.USER,
			Username_Field.getText ());
		server_configuration.Set (PASSWORD_PARAMETER,
			new String (Password_Field.getPassword ()));

		for (int
				index = 0,
				total_additional_fields = Additional_Field_Names.size ();
				index < total_additional_fields;
				index++)
			server_configuration.Set (Field_Name (index), Field_Value (index));

		//	Put the server configuration in a container configuration.
		Server_Configuration = new Configuration ();
		Server_Configuration.Remove_All ();	//	Clear out.
		Server_Configuration.Name (server_configuration.Name ());
		Server_Configuration.Set (Database.SERVER,
			server_configuration.Name ());
		Server_Configuration.Add (server_configuration);
		}
	catch (Configuration_Exception exception)
		{
		Dialog_Box.Error
			("Unable to assemble the server configuration." + NL
			+ NL
			+ ID + NL
			+ exception.getMessage (),
			this);
		Server_Configuration = null;
		return;
		}
	catch (PVL_Exception exception)
		{
		Dialog_Box.Error
			("Unable to assemble the server configuration." + NL
			+ NL
			+ ID + NL
			+ exception.getMessage (),
			this);
		Server_Configuration = null;
		return;
		}
	}
setVisible (false);
}

/**	Resets the non-interactive fields.
*/
private void Reset ()
{
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println
		(">-< Connect_View.Reset");
for (int
		index = 0,
		total_additional_fields = Additional_Field_Names.size ();
		index < total_additional_fields;
		index++)
	Field_Value (index, Additional_Field_Value (index));
}


/**	Clears information fields.
<p>
	@param	clear_all	If true all fields are cleared;
		otherwise only the required fields are cleared.
*/
private void Clear
	(
	boolean	clear_all
	)
{
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println
		(">-< Connect_View.Clear: clear_all = " + clear_all);
Servers_List.getEditor ().setItem (Selected_Server = "");
Server_Type_Field.setText (null);
Hostname_Field.setText (null);
Username_Field.setText (null);
Password_Field.setText (null);
if (clear_all)
	{
	Iterator
		fields = Additional_Fields.iterator ();
	while (fields.hasNext ())
		((JTextField)fields.next ()).setText (null);
	}
}

/*==============================================================================
	Helpers
*/
private void Initialize_Additional_Values ()
{
if ((DEBUG & DEBUG_HELPERS) != 0)
	System.out.println (">>> Connect_View.Initialize_Additional_Values: "
		+ Additional_Field_Names);
String
	name,
	value;
int
	count = Additional_Field_Names.size (),
	index;

//	Generate the mirror Additional_Field_Values list.
for (index = 0;
	 index < count;
	 index++)
	{
	value = "";
	Object
		object = Additional_Field_Names.get (index);
	if (object instanceof Vector)
		{
		//	Name, value pair.
		Additional_Field_Names.set (index, ((Vector)object).get (0));
		object = ((Vector)object).get (1);
		if (object != null)
			value = object.toString ();
		}
	Additional_Field_Values.add (value);
	}

while (--index >= 0)
	{
	name = Field_Name (index);
	if (name.equalsIgnoreCase (Database.TYPE) ||
		name.equalsIgnoreCase (Configuration.HOST) ||
		name.equalsIgnoreCase (Configuration.USER) ||
		name.equalsIgnoreCase (PASSWORD_PARAMETER))
		{
		Additional_Field_Names.remove (index);
		Additional_Field_Values.remove (index);
		}
	else
		{
		for (count = 0;
			 count < index;
			 count++)
			{
			if (name.equalsIgnoreCase (Field_Name (count)))
				{
				Additional_Field_Names.remove (index);
				Additional_Field_Values.remove (index);
				break;
				}
			}
		}
	}
if ((DEBUG & DEBUG_HELPERS) != 0)
	System.out.println
		("    Additional_Field_Names  - " + Additional_Field_Names + NL
		+"    Additional_Field_Values - " + Additional_Field_Values + NL
		+"<<< Connect_View.Initialize_Additional_Values");
}


private String Field_Name
	(
	int		index
	)
{return (String)Additional_Field_Names.get (index);}


private String Field_Value
	(
	int		index
	)
{return ((JTextField)Additional_Fields.get (index)).getText ();}


private void Field_Value
	(
	int		index,
	String	value
	)
{((JTextField)Additional_Fields.get (index)).setText (value);}


private String Additional_Field_Value
	(
	int		index
	)
{return (String)Additional_Field_Values.get (index);}

/*==============================================================================
	Application test stub
*/
private static final Point
	DEFAULT_INITIAL_LOCATION	= new Point (50, 25);

public static void main (String[] arguments) 
{
if (DEBUG != DEBUG_OFF)
	System.out.println ("*** " + ID);
Vector
	additional_fields = new Vector (),
	field;
field = new Vector ();
field.add ("Catalog");
field.add ("catalog");
additional_fields.add (field);
field = new Vector ();
field.add ("Pipeline");
field.add ("pipeline");
additional_fields.add (field);
try
	{
	Configuration
		configuration = null;
	if (arguments.length > 0)
		{
		if ((DEBUG & DEBUG_SETUP) != 0)
			System.out.println
				("    Configuration: " + arguments[0]);
		configuration = new Configuration (arguments[0]);
		}
	if ((DEBUG & DEBUG_SETUP) != 0)
		System.out.println ("    User Configuration -" + NL
			+ ((configuration == null) ?
				"null" : configuration.Description ()));

	configuration = Get_Database_Configuration ("*** Connect_View ***",
		configuration,
		Icons.Load_Icon ("PIRL_Logo.gif"),
		additional_fields,
		null);

	if ((DEBUG & DEBUG_SETUP) != 0)
		System.out.println ("    Database Configuration -" + NL
			+ ((configuration == null) ?
				"null" : configuration.Description ()));
	}
catch (Exception exception)
	{
	System.out.println (exception.getMessage ());
	System.exit (1);
	}
System.exit (0);
}

}	//	End of Connect_View class.

