/* 
  Copyright (C) 2008 Kai Hertel

	This file is part of mmpong.

	mmpong is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	mmpong is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <string.h>
#include "hashmap.h"



static unsigned findPrime(number)
unsigned number; 	// find the next best prime number starting from `number`
{
	unsigned check= number >> 1, iter;
	while (check*check > number) check>>= 1; 	// rough int sqrt approximation
	for (;;number++) {
		if (check*check < number) check++;
		for (iter= 2; iter< check; iter++)
			if ((number % iter) == 0) break;
		if (iter >= check) return number;
	}
	return number;
}



struct hashmap *hashmap_create(size, rehash, map)
unsigned size;
const struct hashmap *rehash; 	// copy elements from the optional `rehash` map into the new one (can be NULL)
unsigned (*map)(const unsigned, const void *);
{
	if (size< 2) size= 2;
	size= findPrime(size -2); 	// actual hashmap size
	struct hashmap *hash= malloc(sizeof(struct hashmap));
	if (!hash) return NULL;

	hash->size= size;
	hash->table= calloc(size, sizeof(struct hashentry));
	if (!hash->table) {
		free(hash);
		return NULL;
	}

	if ((!rehash)||(!map)) return hash;
	for (int idx= 0; idx< rehash->size; idx++) {
		for (int ent= 0; ent< rehash->table[idx].size; ent++) {
			struct hashentry *dest= hash->table + (map(size, rehash->table[idx].keys[ent]) %size);
			dest->keys= realloc(dest->keys, (dest->size +1)*sizeof(void *));
			dest->vals= realloc(dest->vals, (dest->size +1)*sizeof(void *));
			if ((!dest->keys) || (!dest->vals)) {
				hashmap_destroy(hash);
				return NULL;
			}
			dest->keys[dest->size]= rehash->table[idx].keys[ent];
			dest->vals[dest->size]= rehash->table[idx].vals[ent];
			dest->size++;
		}
	}
	return hash;
}



void hashmap_destroy(hash)
struct hashmap *hash;
{
	if (!hash) return;
	if (hash->table) {
		for (int idx= 0; idx< hash->size; idx++) {
			if (hash->table[idx].keys) free(hash->table[idx].keys);
			if (hash->table[idx].vals) free(hash->table[idx].vals);
		}
		free(hash->table);
	}
	free(hash);
}



void **hashmap_lookup(hash, key, map, cmp)
struct hashmap *hash;
const void *key;
unsigned (*map)(const unsigned, const void *);
int (*cmp)(const void *, const void *);
{
	if ((!hash)||(!map)||(!cmp)) return NULL;
	struct hashentry *dest= hash->table + (map(hash->size, key) %hash->size);
	for (int idx= 0; idx< dest->size; idx++)
		if (!cmp(key, dest->keys[idx])) return dest->vals+idx;
	return NULL;
}



// overwrites key if present before
// return values: <0: fatal, ==0: regular, >0: overwrite
int hashmap_insert(hash, key, val, map, cmp)
struct hashmap *hash;
void *key, *val;
unsigned (*map)(const unsigned, const void *);
int (*cmp)(const void *, const void *);
{
	if ((!hash)||(!map)||(!cmp)) return (-1);
	struct hashentry *dest= hash->table + (map(hash->size, key) %hash->size);
	for (int idx= 0; idx< dest->size; idx++)
		if (!cmp(key, dest->keys[idx])) {
			dest->vals[idx]= val;
			return 1;
		}
	dest->keys= realloc(dest->keys, (dest->size +1)*sizeof(void *));
	dest->vals= realloc(dest->vals, (dest->size +1)*sizeof(void *));
	if ((!dest->keys) || (!dest->vals)) {
		if (dest->keys) free(dest->keys);
		if (dest->vals) free(dest->vals);
		dest->keys= dest->vals= NULL;
		dest->size= 0;
		return (-2);
	}
	dest->keys[dest->size]= key;
	dest->vals[dest->size]= val;
	dest->size++;
	return 0;
}



// return values: <0: fatal, ==0: regular, >0: not found
int hashmap_remove(hash, key, map, cmp)
struct hashmap *hash;
const void *key;
unsigned (*map)(const unsigned, const void *);
int (*cmp)(const void *, const void *);
{
	if ((!hash)||(!map)||(!cmp)) return (-1);
	struct hashentry *dest= hash->table + (map(hash->size, key) %hash->size);
	for (int idx= 0; idx< dest->size; idx++) {
		if (cmp(key, dest->keys[idx])) continue;
		memmove(dest->keys+idx, dest->keys+idx+1, (dest->size -idx -1)*sizeof(void *));
		memmove(dest->vals+idx, dest->vals+idx+1, (dest->size -idx -1)*sizeof(void *));
		dest->size--;
		dest->keys= realloc(dest->keys, (dest->size)*sizeof(void *));
		dest->vals= realloc(dest->vals, (dest->size)*sizeof(void *));
		if ((!dest->keys) || (!dest->vals)) {
			if (dest->keys) free(dest->keys);
			if (dest->vals) free(dest->vals);
			dest->keys= dest->vals= NULL;
			dest->size= 0;
			return (-2);
		}
		return 0;
	}
	return 1;
}



/*
unsigned hashmap_fast_inc(hash, key, offset, map)
struct hashmap *hash;
unsigned key;
int offset;
unsigned (*map)(unsigned, void *);
{
	if ((!hash)||(!map)) return (-1);
	struct hashentry *dest= hash->table + (map(hash->size, (void *)key) %hash->size);
	for (int idx= 0; idx< dest->size; idx++)
		if (key == (unsigned) dest->keys[idx]) {
			dest->vals[idx]= (void *)( ((unsigned) dest->vals[idx]) + offset );
			return dest->vals[idx];
		}
	dest->keys= realloc(dest->keys, (dest->size +1)*sizeof(void *));
	dest->vals= realloc(dest->vals, (dest->size +1)*sizeof(void *));
	if ((!dest->keys) || (!dest->vals)) {
		if (dest->keys) free(dest->keys);
		if (dest->vals) free(dest->vals);
		dest->keys= dest->vals= NULL;
		dest->size= 0;
		return 0;
	}
	dest->keys[dest->size]= (void *)key;
	dest->vals[dest->size]= (void *)offset;
	dest->size++;
	return offset;
}
*/
