// SPDX-License-Identifier: Apache-2.0
/*
 * Copyright IBM Corp. 2021
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "testsupport.h"

void setUp(void) { /* This is run before EACH TEST */
  VERIFY_HW_ENV;
}

void tearDown(void) { /* This is run after EACH TEST */
}

zdnn_ztensor *
test_layer(zdnn_ztensor *input, uint32_t *h0_shape, void *h0_values,
           uint32_t *c0_shape, void *c0_values, uint32_t *weights_shape,
           void *weights_values, uint32_t *biases_shape, void *biases_values,
           uint32_t *hidden_weights_shape, void *hidden_weights_values,
           uint32_t *hidden_biases_shape, void *hidden_biases_values,
           uint32_t *all_ts_out_shape, void *all_ts_out_exp_values,
           uint32_t *cell_out_shape, void *cell_out_exp_values,
           bool is_prev_layer_bidir, bool is_this_layer_bidir) {

  zdnn_ztensor *h0, *c0, *weights, *biases, *hidden_weights, *hidden_biases,
      *all_ts_out, *cell_out;

  h0 = alloc_ztensor_with_values(h0_shape, ZDNN_3DS, test_datatype, NO_CONCAT,
                                 false, (float *)h0_values);

  c0 = alloc_ztensor_with_values(c0_shape, ZDNN_3DS, test_datatype, NO_CONCAT,
                                 false, (float *)c0_values);

  // FICO/ZRH elements coming in as one pointer instead of four or three
  // pointers

  uint32_t num_elements_weights =
      weights_shape[0] * weights_shape[1] * weights_shape[2];
  weights = alloc_ztensor_with_values(
      weights_shape, ZDNN_3DS, test_datatype,
      RNN_TYPE_LSTM |
          (is_prev_layer_bidir ? PREV_LAYER_BIDIR : PREV_LAYER_UNI) |
          USAGE_WEIGHTS,
      false, (float *)weights_values,
      (float *)weights_values + num_elements_weights,
      (float *)weights_values + 2 * num_elements_weights,
      (float *)weights_values + 3 * num_elements_weights);

  uint32_t num_elements_biases = biases_shape[0] * biases_shape[1];
  biases = alloc_ztensor_with_values(
      biases_shape, ZDNN_2DS, test_datatype, RNN_TYPE_LSTM | USAGE_BIASES,
      false, (float *)biases_values,
      (float *)biases_values + num_elements_biases,
      (float *)biases_values + 2 * num_elements_biases,
      (float *)biases_values + 3 * num_elements_biases);

  uint32_t num_elements_hidden_weights = hidden_weights_shape[0] *
                                         hidden_weights_shape[1] *
                                         hidden_weights_shape[2];
  hidden_weights = alloc_ztensor_with_values(
      hidden_weights_shape, ZDNN_3DS, test_datatype,
      RNN_TYPE_LSTM | USAGE_HIDDEN_WEIGHTS, false,
      (float *)hidden_weights_values,
      (float *)hidden_weights_values + num_elements_hidden_weights,
      (float *)hidden_weights_values + 2 * num_elements_hidden_weights,
      (float *)hidden_weights_values + 3 * num_elements_hidden_weights);

  uint32_t num_elements_hidden_biases =
      hidden_biases_shape[0] * hidden_biases_shape[1];
  hidden_biases = alloc_ztensor_with_values(
      hidden_biases_shape, ZDNN_2DS, test_datatype,
      RNN_TYPE_LSTM | USAGE_HIDDEN_BIASES, false, (float *)hidden_biases_values,
      (float *)hidden_biases_values + num_elements_hidden_biases,
      (float *)hidden_biases_values + 2 * num_elements_hidden_biases,
      (float *)hidden_biases_values + 3 * num_elements_hidden_biases);

  all_ts_out = alloc_ztensor_with_values(
      all_ts_out_shape, ZDNN_4DS, test_datatype, NO_CONCAT, true, ZERO_ARRAY);

  cell_out = alloc_ztensor_with_values(cell_out_shape, ZDNN_4DS, test_datatype,
                                       NO_CONCAT, true, ZERO_ARRAY);

  zdnn_status status =
      zdnn_lstm(input, h0, c0, weights, biases, hidden_weights, hidden_biases,
                is_this_layer_bidir ? BIDIR : FWD, NULL, all_ts_out, cell_out);

  if (status != ZDNN_OK) {
    TEST_FAIL_MESSAGE_FORMATTED("%s() - zdnn_lstm() not ZDNN_OK, status = %08x",
                                __func__, status);
  }

  assert_ztensor_values(all_ts_out, false, all_ts_out_exp_values);
  assert_ztensor_values(cell_out, false, cell_out_exp_values);

  free_ztensor_buffers(7, h0, c0, weights, biases, hidden_weights,
                       hidden_biases, cell_out);

  return all_ts_out;
}

void lstm_fwd_to_fwd() {

  // num_timesteps = 5
  // num_batches = 2
  // num_features = 4
  // num_hidden = 4, 5

  bool is_layer_bidir[] = {false, false};

  // first layer

  uint32_t input0_shape[] = {5, 2, 4};
  uint32_t input0_values[] = {
      0x3f80f554, 0x3eed5744, 0x3fe9598b, 0x3fde3340, 0x3fb14cbd, 0x3f3b5a0a,
      0x3f82893d, 0x3e5414c8, 0x3f8b5bf7, 0x3f3c425a, 0x3fa6aeeb, 0x3f99290e,
      0x3ffa48dc, 0x3fd4c5a9, 0x3fb4c3ba, 0x3f768450, 0x3f1acb50, 0x3eccc9d0,
      0x3fd6c6c6, 0x3fb7bd3f, 0x3f230434, 0x3e2daec8, 0x3f9a57a9, 0x3e80dd48,
      0x3f94a1a8, 0x3f64e95e, 0x3dc195b0, 0x3ff6bde7, 0x3fd094b3, 0x3fa067b8,
      0x3fb1e4f7, 0x3e0b4360, 0x3fd2f78d, 0x3fbaec30, 0x3fd96d0d, 0x3ff7e13b,
      0x3fcab802, 0x3e0fc588, 0x3f0dc4a2, 0x3f03ec80};
  uint32_t h00_shape[] = {1, 2, 4};
  uint32_t h00_values[] = {0x3f72895c, 0x3fc19f9d, 0x3f54b050, 0x3ff7834f,
                           0x3fdc7d0d, 0x3fc1fce3, 0x3ebcf5b4, 0x3ed3cdb4};
  uint32_t c00_shape[] = {1, 2, 4};
  uint32_t c00_values[] = {0x3fb8c472, 0x3f849e59, 0x3eb88b80, 0x3bc03f00,
                           0x3f1a65ee, 0x3f5d6a8e, 0x3ea8b604, 0x3fcb5de0};
  uint32_t weights0_shape[] = {1, 4, 4};
  uint32_t weights0_values[] = {
      0x3e493898, 0x3dcbca78, 0xbeee948c, 0x3dbfaa08, 0x3ed41bd0, 0xbede9cf8,
      0x3ee3743e, 0xbdac80c8, 0x3edec5e8, 0x3d3c6690, 0x3ec2a6f0, 0xbda882b8,
      0x3ee1e222, 0xbea027ac, 0xbeff5dfe, 0xbe6a5f1c, 0x3dbed0f8, 0x3e67aa8c,
      0x3e8c896e, 0x3e9ed100, 0xbec67a6c, 0x3e4de7f8, 0xbd813f20, 0x3ef5cf48,
      0xbeb16e18, 0xbe97a46a, 0x3c9c6440, 0xbec54796, 0xbe843ed2, 0x3e1aadc8,
      0x3ded4400, 0xbe3ba1ec, 0x3e44e48c, 0x3eb7435c, 0x3e7fa638, 0x3ef0d4f2,
      0xbe97a134, 0x3e3f7148, 0x3dd65318, 0x3eac7f54, 0x3e3bb1dc, 0xbefd5f4a,
      0xbec7b396, 0xbe5f3eb0, 0x3e817616, 0xbea61100, 0xbe9368e8, 0xbe00dcd4,
      0xbef3dd78, 0xbce1b020, 0xbe9bc938, 0xbdfedb88, 0xbc133e80, 0x3d99bfa0,
      0x3ee84968, 0x3cb8d280, 0xbec0c878, 0xbe51adf4, 0x3eaf6fd0, 0x3d956718,
      0xbec577a2, 0x3e97e798, 0xbed7e164, 0x3df2ddd0};
  uint32_t biases0_shape[] = {1, 4};
  uint32_t biases0_values[] = {0xbed2f700, 0x3e8ab64c, 0x3ecb4226, 0x3eaf295c,
                               0x3e7604cc, 0x3e34d140, 0xbdf63f00, 0xbdd75a50,
                               0xbeb493ac, 0x3cb6ae60, 0xbeb47690, 0xbe8cec88,
                               0xbe952c30, 0x3e2ef934, 0xbe988dc4, 0xbc32ba00};
  uint32_t hidden_weights0_shape[] = {1, 4, 4};
  uint32_t hidden_weights0_values[] = {
      0x3c63ac80, 0x3ef27eba, 0xbee5f866, 0xbe7e7cdc, 0xbd30adc0, 0x3ea29306,
      0xbe72ba40, 0xbec42d02, 0x3dcf3d10, 0x3ef30cc4, 0x3eae4fce, 0xbeef9400,
      0x3ea2c660, 0xbd141d60, 0xbb546b00, 0xbed810ec, 0x3e48ec5c, 0xbef59156,
      0xbe331d5c, 0xbea6c676, 0x3cf559e0, 0xbe97bba0, 0xbed1d2ba, 0xbcd23440,
      0xbe79d1cc, 0xbe002b3c, 0xbdd9d200, 0x3eb74200, 0x3e7245d4, 0xbe7966ec,
      0x3ddae2f8, 0xbd5288f0, 0xbdcfb470, 0x3e9fb02e, 0xbdc53cf0, 0xbda03c28,
      0x3e8c6456, 0xbec8528a, 0xbdc90e10, 0x3bd3e180, 0x3e8a6774, 0xbdd87bf0,
      0xbee5b8ba, 0xbe6896b8, 0xbef6e502, 0xbe4f9a1c, 0xbedd0a44, 0x3e40deb8,
      0xbee66a3a, 0x3ee72b36, 0xbd6c53f0, 0x3d5bc2b0, 0xbd0a36c0, 0x3e396c38,
      0xbe648f70, 0xbdd664c0, 0x3ee121a2, 0xbee707ae, 0x3eccb614, 0x3eb6d016,
      0xbe50d738, 0x3ea1f874, 0xbecedf54, 0x3e0eec08};
  uint32_t hidden_biases0_shape[] = {1, 4};
  uint32_t hidden_biases0_values[] = {
      0xbe94a63c, 0x3eb32ed6, 0xbe380bcc, 0x3ed7eee0, 0x3cac4fa0, 0x3ea50604,
      0xbec183fa, 0xbeafbf44, 0x3e3924a0, 0x3d81aa40, 0xbb73ed00, 0xbdca6d08,
      0x3d807a40, 0xbde9d330, 0xbb663e00, 0x3d82a7c0};
  uint32_t all_ts_out0_shape[] = {5, 1, 2, 4};
  uint32_t all_ts_out0_exp_values[] = {
      0x3d7dec3b, 0x3ec321c6, 0xbd33532c, 0xbd2813ae, 0x3d517efa, 0x3ee7cc40,
      0xbd18113a, 0x3e9092b4, 0x3df21ddc, 0xbb310d91, 0xbe18e945, 0xbce56e8b,
      0x3d359756, 0x3e238b20, 0xbe0c1333, 0x3e987c64, 0x3dd11579, 0xbe65fb8f,
      0xbe55c118, 0xbda5413b, 0x3d7d912d, 0xbcbef32e, 0xbe3ba9c3, 0x3e30db74,
      0x3dc41819, 0xbd0f17f6, 0xbdbe76a3, 0x3beeb838, 0x3cc76662, 0x3d477816,
      0xbe6299de, 0x3e62029a, 0x3d3e7a2d, 0xbe47d763, 0xbe4eef76, 0x3d90d525,
      0x3cfd8dfe, 0x3de03933, 0xbdb0d31e, 0x3e51f80d};
  uint32_t cell_out0_shape[] = {1, 1, 2, 4};
  uint32_t cell_out0_values[] = {0x3f3626d4, 0xbeab179c, 0xbf0f2c94,
                                 0x3e03c918, 0x3e366377, 0x3e49c51e,
                                 0xbea2454d, 0x3eeb4eda};

  zdnn_ztensor *input0 =
      alloc_ztensor_with_values(input0_shape, ZDNN_3DS, test_datatype,
                                NO_CONCAT, false, (void *)input0_values);

  zdnn_ztensor *all_ts_out0 =
      test_layer(input0, h00_shape, (void *)h00_values, c00_shape,
                 (void *)c00_values, weights0_shape, (void *)weights0_values,
                 biases0_shape, (void *)biases0_values, hidden_weights0_shape,
                 (void *)hidden_weights0_values, hidden_biases0_shape,
                 (void *)hidden_biases0_values, all_ts_out0_shape,
                 (void *)all_ts_out0_exp_values, cell_out0_shape,
                 (void *)cell_out0_values, false, is_layer_bidir[0]);

  // second layer

  uint32_t h01_shape[] = {1, 2, 5};
  uint32_t h01_values[] = {0x3fe41f4c, 0x3fc316bd, 0x3e4520a0, 0x3fe7e0c3,
                           0x3f8930f2, 0x3c305000, 0x3f2385f8, 0x3f78c07a,
                           0x3feeed13, 0x3f012eea};
  uint32_t c01_shape[] = {1, 2, 5};
  uint32_t c01_values[] = {0x3fdb2c04, 0x3fa455aa, 0x3faaf233, 0x3f92f487,
                           0x3f7d3326, 0x3e3365a8, 0x3f600a90, 0x3dd59f00,
                           0x3ec6cda0, 0x3fd0ec63};
  uint32_t weights1_shape[] = {1, 4, 5};
  uint32_t weights1_values[] = {
      0xbee2f252, 0xbe83e971, 0x3e261b9c, 0x3ebc395c, 0xbed87cba, 0x3e4eb6e0,
      0x3e83a64e, 0xbe18ef8e, 0x3df90638, 0x3dc5e080, 0xbe5bef69, 0x3e72843c,
      0xbdfff1d8, 0xbe58ace5, 0xbe807ef0, 0xbe98cd9b, 0x3eafbdf8, 0x3e074a8c,
      0xbe574539, 0xbecc25a9, 0x3e4d6418, 0x3e1735dc, 0x3e6bc304, 0x3bc9f900,
      0x3ebd57a4, 0xbea1dc41, 0x3eabc840, 0xbedd7037, 0xbd9bb79c, 0xbe05dde4,
      0xbe4216de, 0xbe136b9a, 0x3ea58b16, 0xbe71302b, 0xbe87f0ac, 0x3e06b148,
      0x3a3bae00, 0xbebd57dc, 0x3d721bf0, 0xbe295dd2, 0x3ec8c806, 0x3c696e40,
      0x3d952498, 0xbdd1716c, 0x3e10c984, 0xbe94b2a7, 0xbe9a126e, 0x3ebdf640,
      0xbca2ec60, 0xbc172600, 0xbd506310, 0xbeb9158a, 0xbe1985b8, 0xbe8b7474,
      0xbdcdfc84, 0x3e0332e4, 0x3eb682ba, 0x3e06e404, 0x3e1ae0c4, 0xbe8d2560,
      0x3e9e5868, 0xbe5cf1d4, 0x3db3dd28, 0x3e354500, 0x3e1f0c64, 0xbeaa740b,
      0x3da48928, 0xbeaa02a3, 0xbd669538, 0xbe271822, 0x3e3b9ae0, 0xbe6de235,
      0x3d987eb0, 0xbebbb5a4, 0x3e2dd3f4, 0xbe19fc78, 0x3dd306b8, 0x3e430d88,
      0xbd8d3050, 0x3e987cda};
  uint32_t biases1_shape[] = {1, 5};
  uint32_t biases1_values[] = {0xbe8d6097, 0x3cbbc620, 0x3e5b9460, 0x3e6328f4,
                               0xbed14480, 0xbdf89d64, 0xbdb0b3c4, 0x3d2f4d98,
                               0x3ed3c2e8, 0x3cc42620, 0xbda3e468, 0xbeaa2909,
                               0xbe436636, 0x3e24fec4, 0xbea299d4, 0x3e2a3b28,
                               0x3ec258fa, 0x3cf049b0, 0xbe989ba1, 0xbe24134e};
  uint32_t hidden_weights1_shape[] = {1, 5, 5};
  uint32_t hidden_weights1_values[] = {
      0xbe20b66c, 0x3e5d42c8, 0x3eb2e8ec, 0xbe1b9f76, 0xbee1e40d, 0x3ebbff92,
      0x3e79a49c, 0xbda4ce70, 0x3e5f481c, 0xbeb7e0dd, 0x3e804fe0, 0xbe8f83dc,
      0x3e3248cc, 0xbe9fee66, 0x3eb4c482, 0xbe89ca96, 0x3e036284, 0x3da2aec0,
      0x3dcedbf0, 0x3e77c3f8, 0x3ecdd9da, 0xbe2089f6, 0x3e42d780, 0xbe9aebe8,
      0x3ed6d390, 0xbe6f1e1b, 0x3d64ba10, 0x3e86944e, 0xbec4e626, 0x3eace9cc,
      0x3b9084c0, 0xbeb401c3, 0x3d757710, 0x3ee46d12, 0x3ee4e29a, 0x3e3d5c4c,
      0x3d0ff4f8, 0x3e55de1c, 0x3e915990, 0x3ec83690, 0x3d0f9250, 0x3e9e8ea0,
      0xbe5ec821, 0xbe9e462c, 0x3eb165e2, 0x3d190310, 0x3ece54c0, 0xbebdbf60,
      0x3e332b14, 0xbdd1fa20, 0x3eb76f78, 0x3e9eff90, 0x3ebc5350, 0xbdea86c4,
      0x3e6d851c, 0xbecc7bce, 0xbead1b0c, 0x3ebb7968, 0x3e497f5c, 0x3e8e5746,
      0xbe9c4230, 0xbe5f6ed0, 0x3ea3e864, 0x3ecbffd4, 0xbe20471c, 0xbd93fe10,
      0xbedd358e, 0x3eb0cbec, 0x3e177f54, 0x3e5bbc44, 0xbe94b3e4, 0xbe81ffa5,
      0x3ecb999a, 0x3ee4e636, 0x3d527bf8, 0xbddabb30, 0x3ea2c8c8, 0x3d082a00,
      0x3edb2580, 0xbd8f889c, 0xbe811315, 0xbd507b08, 0xbe58367b, 0x3eade05a,
      0x3ec26bea, 0xbe807b12, 0xbe8480f1, 0x3ed26ffe, 0xbe26eada, 0x3c976030,
      0xbeb030cc, 0x3eb0f98a, 0x3e3b45e4, 0x3e80b7ea, 0xbea1ef22, 0x3e99b77e,
      0x3e926d0a, 0xbeadd2f6, 0xbe8219a4, 0xbe190f96};
  uint32_t hidden_biases1_shape[] = {1, 5};
  uint32_t hidden_biases1_values[] = {
      0xbede7c01, 0xbdb9cd1c, 0x3e99f81e, 0xbed8b7ed, 0x3ebe51d8,
      0x3dc7ff90, 0xbeae8cee, 0x3e63833c, 0xbecfe0c1, 0xbedc1c4e,
      0xbe37306c, 0x3e062014, 0x3ca711d0, 0xbece783b, 0x3ebde4ee,
      0x3e769414, 0x3ee39938, 0x3e675c3c, 0xbe972362, 0x3ebf18f2};
  uint32_t all_ts_out1_shape[] = {5, 1, 2, 5};
  uint32_t all_ts_out1_exp_values[] = {
      0x3e10591e, 0x3ea4c525, 0x3ede5521, 0x3ee29046, 0x3ebfbb06, 0xbdf93f6b,
      0xbd444231, 0x3e873334, 0x3e5763af, 0x3ecbdbf2, 0xbd65671e, 0xbd55c4a2,
      0x3efdcab4, 0x3e2c772b, 0x3eac09f7, 0xbe4fcd44, 0xbe0e3377, 0x3e535bcd,
      0x3dbbe197, 0x3ead8e9b, 0xbe1ad9da, 0xbdfa7c72, 0x3ee06835, 0x3dbbcd47,
      0x3e6c8300, 0xbe44c5f7, 0xbe0ce1ad, 0x3e0a1251, 0x3d4e3a71, 0x3e54c2a1,
      0xbe35533e, 0xbe198119, 0x3ea44292, 0x3d3cfc26, 0x3e0b0c38, 0xbe2a6eaa,
      0xbdee377c, 0x3d8c79a9, 0x3cb2aedd, 0x3ddb9885, 0xbe238eaf, 0xbde48ec9,
      0x3e3951bc, 0x3cb1df06, 0x3d89af34, 0xbe15c676, 0xbde42498, 0x3c0f48aa,
      0xbc2b888c, 0x3d376f4d};
  uint32_t cell_out1_shape[] = {1, 1, 2, 5};
  uint32_t cell_out1_values[] = {0xbe89ae92, 0xbe286430, 0x3e9ceb6f, 0x3d86d03b,
                                 0x3dffb46f, 0xbe804ee8, 0xbe24adf4, 0x3c7cc930,
                                 0xbd0309c4, 0x3da7c588};

  zdnn_ztensor *all_ts_out1 = test_layer(
      all_ts_out0, h01_shape, (void *)h01_values, c01_shape, (void *)c01_values,
      weights1_shape, (void *)weights1_values, biases1_shape,
      (void *)biases1_values, hidden_weights1_shape,
      (void *)hidden_weights1_values, hidden_biases1_shape,
      (void *)hidden_biases1_values, all_ts_out1_shape,
      (void *)all_ts_out1_exp_values, cell_out1_shape, (void *)cell_out1_values,
      is_layer_bidir[0], is_layer_bidir[1]);

  free_ztensor_buffers(3, input0, all_ts_out0, all_ts_out1);
}

void lstm_fwd_to_bidir() {

  // num_timesteps = 5
  // num_batches = 2
  // num_features = 4
  // num_hidden = 4, 5

  bool is_layer_bidir[] = {false, true};

  // first layer

  uint32_t input0_shape[] = {5, 2, 4};
  uint32_t input0_values[] = {
      0x3f80f554, 0x3eed5744, 0x3fe9598b, 0x3fde3340, 0x3fb14cbd, 0x3f3b5a0a,
      0x3f82893d, 0x3e5414c8, 0x3f8b5bf7, 0x3f3c425a, 0x3fa6aeeb, 0x3f99290e,
      0x3ffa48dc, 0x3fd4c5a9, 0x3fb4c3ba, 0x3f768450, 0x3f1acb50, 0x3eccc9d0,
      0x3fd6c6c6, 0x3fb7bd3f, 0x3f230434, 0x3e2daec8, 0x3f9a57a9, 0x3e80dd48,
      0x3f94a1a8, 0x3f64e95e, 0x3dc195b0, 0x3ff6bde7, 0x3fd094b3, 0x3fa067b8,
      0x3fb1e4f7, 0x3e0b4360, 0x3fd2f78d, 0x3fbaec30, 0x3fd96d0d, 0x3ff7e13b,
      0x3fcab802, 0x3e0fc588, 0x3f0dc4a2, 0x3f03ec80};
  uint32_t h00_shape[] = {1, 2, 4};
  uint32_t h00_values[] = {0x3f72895c, 0x3fc19f9d, 0x3f54b050, 0x3ff7834f,
                           0x3fdc7d0d, 0x3fc1fce3, 0x3ebcf5b4, 0x3ed3cdb4};
  uint32_t c00_shape[] = {1, 2, 4};
  uint32_t c00_values[] = {0x3fb8c472, 0x3f849e59, 0x3eb88b80, 0x3bc03f00,
                           0x3f1a65ee, 0x3f5d6a8e, 0x3ea8b604, 0x3fcb5de0};
  uint32_t weights0_shape[] = {1, 4, 4};
  uint32_t weights0_values[] = {
      0x3e493898, 0x3dcbca78, 0xbeee948c, 0x3dbfaa08, 0x3ed41bd0, 0xbede9cf8,
      0x3ee3743e, 0xbdac80c8, 0x3edec5e8, 0x3d3c6690, 0x3ec2a6f0, 0xbda882b8,
      0x3ee1e222, 0xbea027ac, 0xbeff5dfe, 0xbe6a5f1c, 0x3dbed0f8, 0x3e67aa8c,
      0x3e8c896e, 0x3e9ed100, 0xbec67a6c, 0x3e4de7f8, 0xbd813f20, 0x3ef5cf48,
      0xbeb16e18, 0xbe97a46a, 0x3c9c6440, 0xbec54796, 0xbe843ed2, 0x3e1aadc8,
      0x3ded4400, 0xbe3ba1ec, 0x3e44e48c, 0x3eb7435c, 0x3e7fa638, 0x3ef0d4f2,
      0xbe97a134, 0x3e3f7148, 0x3dd65318, 0x3eac7f54, 0x3e3bb1dc, 0xbefd5f4a,
      0xbec7b396, 0xbe5f3eb0, 0x3e817616, 0xbea61100, 0xbe9368e8, 0xbe00dcd4,
      0xbef3dd78, 0xbce1b020, 0xbe9bc938, 0xbdfedb88, 0xbc133e80, 0x3d99bfa0,
      0x3ee84968, 0x3cb8d280, 0xbec0c878, 0xbe51adf4, 0x3eaf6fd0, 0x3d956718,
      0xbec577a2, 0x3e97e798, 0xbed7e164, 0x3df2ddd0};
  uint32_t biases0_shape[] = {1, 4};
  uint32_t biases0_values[] = {0xbed2f700, 0x3e8ab64c, 0x3ecb4226, 0x3eaf295c,
                               0x3e7604cc, 0x3e34d140, 0xbdf63f00, 0xbdd75a50,
                               0xbeb493ac, 0x3cb6ae60, 0xbeb47690, 0xbe8cec88,
                               0xbe952c30, 0x3e2ef934, 0xbe988dc4, 0xbc32ba00};
  uint32_t hidden_weights0_shape[] = {1, 4, 4};
  uint32_t hidden_weights0_values[] = {
      0x3c63ac80, 0x3ef27eba, 0xbee5f866, 0xbe7e7cdc, 0xbd30adc0, 0x3ea29306,
      0xbe72ba40, 0xbec42d02, 0x3dcf3d10, 0x3ef30cc4, 0x3eae4fce, 0xbeef9400,
      0x3ea2c660, 0xbd141d60, 0xbb546b00, 0xbed810ec, 0x3e48ec5c, 0xbef59156,
      0xbe331d5c, 0xbea6c676, 0x3cf559e0, 0xbe97bba0, 0xbed1d2ba, 0xbcd23440,
      0xbe79d1cc, 0xbe002b3c, 0xbdd9d200, 0x3eb74200, 0x3e7245d4, 0xbe7966ec,
      0x3ddae2f8, 0xbd5288f0, 0xbdcfb470, 0x3e9fb02e, 0xbdc53cf0, 0xbda03c28,
      0x3e8c6456, 0xbec8528a, 0xbdc90e10, 0x3bd3e180, 0x3e8a6774, 0xbdd87bf0,
      0xbee5b8ba, 0xbe6896b8, 0xbef6e502, 0xbe4f9a1c, 0xbedd0a44, 0x3e40deb8,
      0xbee66a3a, 0x3ee72b36, 0xbd6c53f0, 0x3d5bc2b0, 0xbd0a36c0, 0x3e396c38,
      0xbe648f70, 0xbdd664c0, 0x3ee121a2, 0xbee707ae, 0x3eccb614, 0x3eb6d016,
      0xbe50d738, 0x3ea1f874, 0xbecedf54, 0x3e0eec08};
  uint32_t hidden_biases0_shape[] = {1, 4};
  uint32_t hidden_biases0_values[] = {
      0xbe94a63c, 0x3eb32ed6, 0xbe380bcc, 0x3ed7eee0, 0x3cac4fa0, 0x3ea50604,
      0xbec183fa, 0xbeafbf44, 0x3e3924a0, 0x3d81aa40, 0xbb73ed00, 0xbdca6d08,
      0x3d807a40, 0xbde9d330, 0xbb663e00, 0x3d82a7c0};
  uint32_t all_ts_out0_shape[] = {5, 1, 2, 4};
  uint32_t all_ts_out0_exp_values[] = {
      0x3d7dec3b, 0x3ec321c6, 0xbd33532c, 0xbd2813ae, 0x3d517efa, 0x3ee7cc40,
      0xbd18113a, 0x3e9092b4, 0x3df21ddc, 0xbb310d91, 0xbe18e945, 0xbce56e8b,
      0x3d359756, 0x3e238b20, 0xbe0c1333, 0x3e987c64, 0x3dd11579, 0xbe65fb8f,
      0xbe55c118, 0xbda5413b, 0x3d7d912d, 0xbcbef32e, 0xbe3ba9c3, 0x3e30db74,
      0x3dc41819, 0xbd0f17f6, 0xbdbe76a3, 0x3beeb838, 0x3cc76662, 0x3d477816,
      0xbe6299de, 0x3e62029a, 0x3d3e7a2d, 0xbe47d763, 0xbe4eef76, 0x3d90d525,
      0x3cfd8dfe, 0x3de03933, 0xbdb0d31e, 0x3e51f80d};
  uint32_t cell_out0_shape[] = {1, 1, 2, 4};
  uint32_t cell_out0_values[] = {0x3f3626d4, 0xbeab179c, 0xbf0f2c94,
                                 0x3e03c918, 0x3e366377, 0x3e49c51e,
                                 0xbea2454d, 0x3eeb4eda};

  zdnn_ztensor *input0 =
      alloc_ztensor_with_values(input0_shape, ZDNN_3DS, test_datatype,
                                NO_CONCAT, false, (void *)input0_values);

  zdnn_ztensor *all_ts_out0 =
      test_layer(input0, h00_shape, (void *)h00_values, c00_shape,
                 (void *)c00_values, weights0_shape, (void *)weights0_values,
                 biases0_shape, (void *)biases0_values, hidden_weights0_shape,
                 (void *)hidden_weights0_values, hidden_biases0_shape,
                 (void *)hidden_biases0_values, all_ts_out0_shape,
                 (void *)all_ts_out0_exp_values, cell_out0_shape,
                 (void *)cell_out0_values, false, is_layer_bidir[0]);

  // second layer

  uint32_t h01_shape[] = {2, 2, 5};
  uint32_t h01_values[] = {0x3fe41f4c, 0x3fc316bd, 0x3e4520a0, 0x3fe7e0c3,
                           0x3f8930f2, 0x3c305000, 0x3f2385f8, 0x3f78c07a,
                           0x3feeed13, 0x3f012eea, 0x3fdb2c04, 0x3fa455aa,
                           0x3faaf233, 0x3f92f487, 0x3f7d3326, 0x3e3365a8,
                           0x3f600a90, 0x3dd59f00, 0x3ec6cda0, 0x3fd0ec63};
  uint32_t c01_shape[] = {2, 2, 5};
  uint32_t c01_values[] = {0x3f0d2ed6, 0x3fda7b92, 0x3fb63fe7, 0x3f34b460,
                           0x3f2b7888, 0x3e7fc438, 0x3fa9348a, 0x3f7f9716,
                           0x3ef8690c, 0x3ffbc9ad, 0x3e8dd57c, 0x3fe9d898,
                           0x3f7c78c0, 0x3f95c31c, 0x3fc36a05, 0x3f5e2a0a,
                           0x3e313c38, 0x3fa56aba, 0x3fcbfe2b, 0x3faf56e1};
  uint32_t weights1_shape[] = {2, 4, 5};
  uint32_t weights1_values[] = {
      0x3ec8c806, 0x3c696e40, 0x3d952498, 0xbdd1716c, 0x3e10c984, 0xbe94b2a7,
      0xbe9a126e, 0x3ebdf640, 0xbca2ec60, 0xbc172600, 0xbd506310, 0xbeb9158a,
      0xbe1985b8, 0xbe8b7474, 0xbdcdfc84, 0x3e0332e4, 0x3eb682ba, 0x3e06e404,
      0x3e1ae0c4, 0xbe8d2560, 0xbebfde02, 0xbed417e7, 0x3ec1d528, 0xbe0751fa,
      0xbdfe1e6c, 0xbe77c691, 0x3ea17e98, 0xbe76cf7f, 0x3e1940d8, 0xbe2ab878,
      0x3eca9984, 0x3e658114, 0x3e109bc4, 0xbe03c1e6, 0x3de69348, 0x3ed16702,
      0x3e878898, 0x3e3b6830, 0x3e8d90bc, 0x3e226e48, 0xbee2f252, 0xbe83e971,
      0x3e261b9c, 0x3ebc395c, 0xbed87cba, 0x3e4eb6e0, 0x3e83a64e, 0xbe18ef8e,
      0x3df90638, 0x3dc5e080, 0xbe5bef69, 0x3e72843c, 0xbdfff1d8, 0xbe58ace5,
      0xbe807ef0, 0xbe98cd9b, 0x3eafbdf8, 0x3e074a8c, 0xbe574539, 0xbecc25a9,
      0xbec39102, 0xbea1a3c0, 0xbd3aa670, 0x3c6b9ce0, 0x3e630230, 0x3e55ae7c,
      0xbe62d375, 0x3eb037d8, 0xbe0d9648, 0xbea06a9a, 0xbe81b1a9, 0xbebc9a53,
      0x3e8db48a, 0xbdc724ec, 0x3ec02c1a, 0x3e5c50f0, 0x3e6ef9a8, 0x3e7d66c4,
      0x3d737210, 0xbd472b98, 0x3e9e5868, 0xbe5cf1d4, 0x3db3dd28, 0x3e354500,
      0x3e1f0c64, 0xbeaa740b, 0x3da48928, 0xbeaa02a3, 0xbd669538, 0xbe271822,
      0x3e3b9ae0, 0xbe6de235, 0x3d987eb0, 0xbebbb5a4, 0x3e2dd3f4, 0xbe19fc78,
      0x3dd306b8, 0x3e430d88, 0xbd8d3050, 0x3e987cda, 0xbe5fbd62, 0x3e0b5e64,
      0xbe86a497, 0xbaa00c00, 0xbeacb04b, 0x3d21ed48, 0xbddce6cc, 0xbe68730d,
      0xbe5ddf86, 0xbeb99f2f, 0x3e84b2c0, 0x3e208298, 0x3ed26dd4, 0x3ee494f2,
      0xbead8f69, 0xbd3641c0, 0xbea5ddf2, 0xbdf673cc, 0x3edfa1a6, 0xbb0d5900,
      0xbe6f1e1b, 0x3d190310, 0x3e9e8ea0, 0x3e55de1c, 0x3ee46d12, 0x3b9084c0,
      0x3d64ba10, 0x3ece54c0, 0xbe5ec821, 0x3e915990, 0x3e3d5c4c, 0xbeb401c3,
      0x3e86944e, 0xbebdbf60, 0xbe9e462c, 0x3d0f9250, 0x3d0ff4f8, 0x3d757710,
      0xbec4e626, 0x3e332b14, 0xbde97700, 0xbe6f45de, 0x3d7ba930, 0xbe28e040,
      0x3ee1a07c, 0xbe95df9f, 0x3dcaf230, 0x3ebc4676, 0x3ee0b168, 0xbe90de80,
      0xbe2440b2, 0xbdd20768, 0xbe9acddc, 0xbed93dd3, 0x3daf9920, 0x3dad0a60,
      0xbe5de779, 0x3caa1db0, 0xbedb8204, 0xbd1e1828};
  uint32_t biases1_shape[] = {2, 5};
  uint32_t biases1_values[] = {
      0xbede7c01, 0xbdb9cd1c, 0x3e99f81e, 0xbed8b7ed, 0x3ebe51d8, 0xbdc69f70,
      0x3ea85d6a, 0xbeb1737d, 0x3d428f68, 0x3ed75422, 0x3dc7ff90, 0xbeae8cee,
      0x3e63833c, 0xbecfe0c1, 0xbedc1c4e, 0x3d940df8, 0x3ed2d41c, 0xbe5a9fca,
      0x3e23c650, 0xbde59ef4, 0xbe37306c, 0x3e062014, 0x3ca711d0, 0xbece783b,
      0x3ebde4ee, 0xbd7bb5b8, 0x3eb1c89c, 0xbe0d071e, 0x3eb8509c, 0xbedd7e2d,
      0x3e769414, 0x3ee39938, 0x3e675c3c, 0xbe972362, 0x3ebf18f2, 0xbd840080,
      0xbda0df98, 0x3e1469e4, 0x3e33aa40, 0x3eafcf42};
  uint32_t hidden_weights1_shape[] = {2, 5, 5};
  uint32_t hidden_weights1_values[] = {
      0xbee1e40d, 0x3eb76f78, 0x3e9eff90, 0x3ebc5350, 0xbdea86c4, 0xbeb7e0dd,
      0xbecc7bce, 0xbead1b0c, 0x3ebb7968, 0x3e497f5c, 0x3eb4c482, 0xbe9c4230,
      0xbe5f6ed0, 0x3ea3e864, 0x3ecbffd4, 0x3e77c3f8, 0xbd93fe10, 0xbedd358e,
      0x3eb0cbec, 0x3e177f54, 0x3ed6d390, 0xbe94b3e4, 0xbe81ffa5, 0x3ecb999a,
      0x3ee4e636, 0xbebd9868, 0x3db92198, 0xbec9e6b4, 0xbec61cd1, 0xbe2ccb44,
      0xbecfb148, 0x3e2de8c8, 0xbecee7d6, 0x3ed4086e, 0xbe9d7ac6, 0x3de585b8,
      0x3eb61b5a, 0x3ed5ca40, 0x3ed8ea94, 0x3ed8d474, 0xbd0ab3d0, 0x3eb1c556,
      0x3e4a7010, 0x3ecebb20, 0xbe44c542, 0xbe6741db, 0xbd891828, 0x3e479f54,
      0xbec12893, 0xbe5113e1, 0x3eace9cc, 0xbe20b66c, 0x3e5d42c8, 0x3eb2e8ec,
      0xbe1b9f76, 0x3ee4e29a, 0x3ebbff92, 0x3e79a49c, 0xbda4ce70, 0x3e5f481c,
      0x3ec83690, 0x3e804fe0, 0xbe8f83dc, 0x3e3248cc, 0xbe9fee66, 0x3eb165e2,
      0xbe89ca96, 0x3e036284, 0x3da2aec0, 0x3dcedbf0, 0xbdd1fa20, 0x3ecdd9da,
      0xbe2089f6, 0x3e42d780, 0xbe9aebe8, 0xbdc0f07c, 0xbe872d40, 0xbdcbff10,
      0x3e8472c6, 0xbe19b22c, 0xbdcc9010, 0xbe1c1d3a, 0xbda475ac, 0xbe0aeb80,
      0xbed457d0, 0x3e917fbc, 0x3e667240, 0x3eb369f6, 0xbe97eae2, 0x3e3bb9b4,
      0x3caf1b90, 0xbe942d27, 0x3e662ae4, 0xbd084a60, 0x3edee626, 0xbedeee8a,
      0x3ed7e74a, 0x3ec2326c, 0x3ebd81c8, 0xbe025ea8, 0x3e6d851c, 0xbddabb30,
      0x3ea2c8c8, 0x3d082a00, 0x3edb2580, 0x3e8e5746, 0xbe811315, 0xbd507b08,
      0xbe58367b, 0x3eade05a, 0xbe20471c, 0xbe807b12, 0xbe8480f1, 0x3ed26ffe,
      0xbe26eada, 0x3e5bbc44, 0xbeb030cc, 0x3eb0f98a, 0x3e3b45e4, 0x3e80b7ea,
      0x3d527bf8, 0x3e99b77e, 0x3e926d0a, 0xbeadd2f6, 0xbe8219a4, 0x3e6fec98,
      0xbeb25d85, 0x3e66f338, 0x3ed89bd2, 0x3ec8c0ca, 0xbe53d7b8, 0x3ebee346,
      0x3d81ac10, 0x3dd8c630, 0xbd97418c, 0xbe618c84, 0xbe4a029f, 0x3ec2d2d6,
      0xbedf67a9, 0xbed0b705, 0xbd203aa8, 0x3e2270c4, 0x3d763d80, 0xbe025fa2,
      0xbce64df0, 0xbd301208, 0x3ec72844, 0xbe53df41, 0xbe9bf81e, 0x3e8fcc58,
      0xbd8f889c, 0xbdf89d64, 0xbe8d6097, 0xbda3e468, 0x3e2a3b28, 0x3ec26bea,
      0xbdb0b3c4, 0x3cbbc620, 0xbeaa2909, 0x3ec258fa, 0x3c976030, 0x3d2f4d98,
      0x3e5b9460, 0xbe436636, 0x3cf049b0, 0xbea1ef22, 0x3ed3c2e8, 0x3e6328f4,
      0x3e24fec4, 0xbe989ba1, 0xbe190f96, 0x3cc42620, 0xbed14480, 0xbea299d4,
      0xbe24134e, 0x3dedf310, 0xbe362bda, 0x3d836668, 0xbe8525dc, 0xbe3b1bb2,
      0x3e10ce08, 0xbed605fa, 0x3e122c34, 0x3ebc54aa, 0x3ec058f2, 0x3d2a1fb8,
      0xbeac7e7c, 0x3d01b298, 0xbeb62674, 0xbe9d91cb, 0x3e2abb28, 0x3e4679ac,
      0xbe94746f, 0xbddc5118, 0xbec0490a, 0xbddf28c4, 0xbe879404, 0x3edaf946,
      0x3e791bd4, 0xbe4e7f38};
  uint32_t hidden_biases1_shape[] = {2, 5};
  uint32_t hidden_biases1_values[] = {
      0xbe325de8, 0x3dc59638, 0xbeb3c7f8, 0x3e11fa20, 0x3e75d434, 0xbe987408,
      0xbd4637a8, 0xbcc4c620, 0x3e4c5720, 0x3e9c8b2a, 0x3cc4e590, 0x3e592a78,
      0xbeb798f6, 0xbe03b7b6, 0xbee0d2e5, 0xbd88748c, 0xbc914780, 0x3e9ccdb4,
      0xbdf7d0f0, 0x3ec8b9ca, 0x3d7f2f50, 0xbe9933c8, 0xbeb1870e, 0xbe0d48c0,
      0x3e4904fc, 0xbd912c2c, 0xbebdb332, 0x3e62e8b8, 0x3e08fc84, 0x3e37f4f4,
      0x3ee1dbc6, 0x3e83aa94, 0xbd4e46b0, 0x3e20904c, 0xbee0a324, 0xbe8d3f0b,
      0x3e935dc2, 0x3ed8df8e, 0x3d1ef258, 0xbed5df49};
  uint32_t all_ts_out1_shape[] = {5, 2, 2, 5};
  uint32_t all_ts_out1_exp_values[] = {
      0x3ece7c30, 0x3e0ade44, 0x3ea77833, 0x3d8a1542, 0x3ec14d90, 0x3e0f6de1,
      0xbcc5b015, 0x3e3d01bd, 0x3ddc92d0, 0x3ed872c6, 0xbdb78493, 0x3e4d107e,
      0x3d629dbb, 0x3e81d6dc, 0xbe282ecb, 0xbda067b0, 0x3d215190, 0xbc606d8e,
      0x3e9d2a63, 0xbe17d42e, 0x3e4adf52, 0xbc961bd7, 0x3e1f1361, 0xbcd7e3b3,
      0x3ea4f150, 0x3b1279df, 0xbd0fc6b2, 0x3ce5e595, 0xbc901e15, 0x3ee2c9e3,
      0xbde63869, 0x3e83c0d4, 0x3ddda844, 0x3e83a426, 0xbd9c714e, 0xbdbdde95,
      0x3dbfb398, 0x3ca6b42f, 0x3e9b0b9e, 0xbd832c54, 0x3da00061, 0xbd3cde78,
      0x3d0c9de8, 0xbdce55e5, 0x3e8c68d8, 0xbd389116, 0xbc3aea77, 0xbcdead41,
      0xbdcf5f14, 0x3ec552b2, 0xbdf64801, 0x3ea190d7, 0x3e43aea4, 0x3e90136c,
      0x3d1c6530, 0xbdb73956, 0x3e1a74cf, 0x3d91c0d8, 0x3e9f6f28, 0x3d0d3abe,
      0xbb2c613b, 0xbd29ebe1, 0xbd1e8311, 0xbe0a8488, 0x3e7dc919, 0xbdadd9a7,
      0x3c9a6539, 0xbd8b9ccc, 0xbe0bde89, 0x3eaf4cea, 0xbdcda1c7, 0x3ec24b51,
      0x3e9f2fd6, 0x3ea62629, 0x3e1f64a3, 0xbd7dae5d, 0x3e4a2633, 0x3e13742c,
      0x3eaf0a3c, 0x3e0b9059, 0xbd110365, 0xbcb1df00, 0xbd897780, 0xbe1bafd2,
      0x3e63dc44, 0xbdc911b3, 0x3cd0cccd, 0xbdcc05ba, 0xbe19b36a, 0x3e95fa6d,
      0xbd9d8379, 0x3e60bed7, 0x3f00a623, 0x3eb132b8, 0x3e57d24f, 0x3d00a823,
      0x3e3e6cd3, 0x3e9d532b, 0x3f01cead, 0x3e8b3a37};
  uint32_t cell_out1_shape[] = {1, 2, 2, 5};
  uint32_t cell_out1_values[] = {
      0xbd5edee1, 0xbd04585f, 0xbe0fefd7, 0xbeab8c8a, 0x3efb1f43,
      0xbe19186a, 0x3d1c1ce0, 0xbe48d1fc, 0xbebf107e, 0x3f29bd16,
      0xbe65346c, 0x3ec43b22, 0x3dacd5ad, 0x3ee2a358, 0xbebe8bd4,
      0xbe48f9a6, 0x3d90ece4, 0xbcaaee82, 0x3f187263, 0xbeb46694};

  zdnn_ztensor *all_ts_out1 = test_layer(
      all_ts_out0, h01_shape, (void *)h01_values, c01_shape, (void *)c01_values,
      weights1_shape, (void *)weights1_values, biases1_shape,
      (void *)biases1_values, hidden_weights1_shape,
      (void *)hidden_weights1_values, hidden_biases1_shape,
      (void *)hidden_biases1_values, all_ts_out1_shape,
      (void *)all_ts_out1_exp_values, cell_out1_shape, (void *)cell_out1_values,
      is_layer_bidir[0], is_layer_bidir[1]);

  free_ztensor_buffers(3, input0, all_ts_out0, all_ts_out1);
}

void lstm_bidir_to_bidir() {

  // num_timesteps = 5
  // num_batches = 2
  // num_features = 4
  // num_hidden = 4, 5

  bool is_layer_bidir[] = {true, true};

  // first layer

  uint32_t input0_shape[] = {5, 2, 4};
  uint32_t input0_values[] = {
      0x3f80f554, 0x3eed5744, 0x3fe9598b, 0x3fde3340, 0x3fb14cbd, 0x3f3b5a0a,
      0x3f82893d, 0x3e5414c8, 0x3f8b5bf7, 0x3f3c425a, 0x3fa6aeeb, 0x3f99290e,
      0x3ffa48dc, 0x3fd4c5a9, 0x3fb4c3ba, 0x3f768450, 0x3f1acb50, 0x3eccc9d0,
      0x3fd6c6c6, 0x3fb7bd3f, 0x3f230434, 0x3e2daec8, 0x3f9a57a9, 0x3e80dd48,
      0x3f94a1a8, 0x3f64e95e, 0x3dc195b0, 0x3ff6bde7, 0x3fd094b3, 0x3fa067b8,
      0x3fb1e4f7, 0x3e0b4360, 0x3fd2f78d, 0x3fbaec30, 0x3fd96d0d, 0x3ff7e13b,
      0x3fcab802, 0x3e0fc588, 0x3f0dc4a2, 0x3f03ec80};
  uint32_t h00_shape[] = {2, 2, 4};
  uint32_t h00_values[] = {0x3f72895c, 0x3fc19f9d, 0x3f54b050, 0x3ff7834f,
                           0x3fdc7d0d, 0x3fc1fce3, 0x3ebcf5b4, 0x3ed3cdb4,
                           0x3fb8c472, 0x3f849e59, 0x3eb88b80, 0x3bc03f00,
                           0x3f1a65ee, 0x3f5d6a8e, 0x3ea8b604, 0x3fcb5de0};
  uint32_t c00_shape[] = {2, 2, 4};
  uint32_t c00_values[] = {0x3f504bc2, 0x3fe33d36, 0x3fd8b70c, 0x3fc21f69,
                           0x3f0c2aba, 0x3f190c04, 0x3fcbd235, 0x3f32a91c,
                           0x3ee6ed24, 0x3f9027e4, 0x3f7639bc, 0x3f44af00,
                           0x3ec25e00, 0x3d230b80, 0x3fe2a3cb, 0x3faee87b};
  uint32_t weights0_shape[] = {2, 4, 4};
  uint32_t weights0_values[] = {
      0x3e44e48c, 0x3eb7435c, 0x3e7fa638, 0x3ef0d4f2, 0xbe97a134, 0x3e3f7148,
      0x3dd65318, 0x3eac7f54, 0x3e3bb1dc, 0xbefd5f4a, 0xbec7b396, 0xbe5f3eb0,
      0x3e817616, 0xbea61100, 0xbe9368e8, 0xbe00dcd4, 0x3be1d000, 0x3ed3b0f2,
      0xbefdbbe6, 0xbe937b62, 0xbdae18e0, 0xbe15aae8, 0x3e671d1c, 0x3e933052,
      0xbe86d40a, 0xbe97fc56, 0xbe75e520, 0x3e879224, 0x3d8757d8, 0xbe3d5b84,
      0xbeaad6d0, 0x3ec47c50, 0x3e493898, 0x3dcbca78, 0xbeee948c, 0x3dbfaa08,
      0x3ed41bd0, 0xbede9cf8, 0x3ee3743e, 0xbdac80c8, 0x3edec5e8, 0x3d3c6690,
      0x3ec2a6f0, 0xbda882b8, 0x3ee1e222, 0xbea027ac, 0xbeff5dfe, 0xbe6a5f1c,
      0x3d7fab80, 0x3e65a254, 0x3e290ef0, 0x3e83cb7a, 0x3ee54c20, 0xbeb4f724,
      0x3ec00ef2, 0xbef7935a, 0x3e9c9930, 0xbe58ff9c, 0xbe24d228, 0x3eb91542,
      0xbea1d8c6, 0x3e169740, 0x3a51d400, 0xbed3b130, 0xbef3dd78, 0xbce1b020,
      0xbe9bc938, 0xbdfedb88, 0xbc133e80, 0x3d99bfa0, 0x3ee84968, 0x3cb8d280,
      0xbec0c878, 0xbe51adf4, 0x3eaf6fd0, 0x3d956718, 0xbec577a2, 0x3e97e798,
      0xbed7e164, 0x3df2ddd0, 0x3e39b6d8, 0x3ed270de, 0xbef20a42, 0x3ee07afa,
      0xbe2afcc4, 0x3e0b3574, 0x3ddd3bb0, 0xbea63fd0, 0xbe0f13d4, 0xbe72401c,
      0xbe8fa9a8, 0xbd68fbd0, 0x3e174298, 0xbe70adfc, 0xbee43e50, 0x3e12af48,
      0x3e48ec5c, 0xbef59156, 0xbe331d5c, 0xbea6c676, 0x3cf559e0, 0xbe97bba0,
      0xbed1d2ba, 0xbcd23440, 0xbe79d1cc, 0xbe002b3c, 0xbdd9d200, 0x3eb74200,
      0x3e7245d4, 0xbe7966ec, 0x3ddae2f8, 0xbd5288f0, 0x3c827de0, 0x3da6bf30,
      0xbdea2a18, 0x3e21e080, 0xbeac41fa, 0x3ed46246, 0xbcb62760, 0xbc28fd40,
      0xbeceee2c, 0xbe2ba4a8, 0xbe9bea52, 0xbde64cc0, 0x3ecc0d98, 0x3e16cff8,
      0x3e2d28ac, 0xbe9dce58};
  uint32_t biases0_shape[] = {2, 4};
  uint32_t biases0_values[] = {
      0xbe94a63c, 0x3eb32ed6, 0xbe380bcc, 0x3ed7eee0, 0x3e8a8150, 0x3ef02ee8,
      0x3ecd1648, 0xbee49ea0, 0x3cac4fa0, 0x3ea50604, 0xbec183fa, 0xbeafbf44,
      0xbead3520, 0xbefc8dba, 0xbecd9510, 0x3eca1ab6, 0x3e3924a0, 0x3d81aa40,
      0xbb73ed00, 0xbdca6d08, 0xbeaeb3ce, 0xbda575e8, 0xbea64132, 0x3eb1c3f8,
      0x3d807a40, 0xbde9d330, 0xbb663e00, 0x3d82a7c0, 0xbec189ba, 0xbe79ce38,
      0xbef751c4, 0xbe9157c6};
  uint32_t hidden_weights0_shape[] = {2, 4, 4};
  uint32_t hidden_weights0_values[] = {
      0xbdcfb470, 0x3e9fb02e, 0xbdc53cf0, 0xbda03c28, 0x3e8c6456, 0xbec8528a,
      0xbdc90e10, 0x3bd3e180, 0x3e8a6774, 0xbdd87bf0, 0xbee5b8ba, 0xbe6896b8,
      0xbef6e502, 0xbe4f9a1c, 0xbedd0a44, 0x3e40deb8, 0x3e31d250, 0xbe85abba,
      0x3d2b1290, 0x3eb145b4, 0xbe3ad12c, 0x3ba19380, 0x3d7fb970, 0x3ee6af64,
      0x3e425874, 0x3e53b624, 0xbec940fa, 0x3e9676d8, 0x3eaa7c86, 0x3d208490,
      0x3d20f2e0, 0x3d893818, 0x3c63ac80, 0x3ef27eba, 0xbee5f866, 0xbe7e7cdc,
      0xbd30adc0, 0x3ea29306, 0xbe72ba40, 0xbec42d02, 0x3dcf3d10, 0x3ef30cc4,
      0x3eae4fce, 0xbeef9400, 0x3ea2c660, 0xbd141d60, 0xbb546b00, 0xbed810ec,
      0x3eb10914, 0xbe77060c, 0x3dc91810, 0x3e4aaa5c, 0xbebe9294, 0x3db7f4e0,
      0xbebe13ca, 0xbd80e658, 0x3e51bfac, 0xbe84fb22, 0x3daa7e98, 0xbed1dd9a,
      0xbe2c296c, 0x3debef40, 0x3e5a1364, 0xbd9dda90, 0xbee66a3a, 0x3ee72b36,
      0xbd6c53f0, 0x3d5bc2b0, 0xbd0a36c0, 0x3e396c38, 0xbe648f70, 0xbdd664c0,
      0x3ee121a2, 0xbee707ae, 0x3eccb614, 0x3eb6d016, 0xbe50d738, 0x3ea1f874,
      0xbecedf54, 0x3e0eec08, 0x3e6f1c7c, 0x3eff635a, 0x3ec152aa, 0xbdeac2f0,
      0xbe7913dc, 0x3ea2818e, 0x3effe6c2, 0xbe33aea0, 0xbed424ec, 0xbeb0f4b2,
      0x3edfd858, 0x3ed23042, 0xbedc23ca, 0x3e4850f4, 0x3ec65644, 0x3e8f750a,
      0x3e7604cc, 0xbed2f700, 0xbeb493ac, 0xbe952c30, 0x3e34d140, 0x3e8ab64c,
      0x3cb6ae60, 0x3e2ef934, 0xbdf63f00, 0x3ecb4226, 0xbeb47690, 0xbe988dc4,
      0xbdd75a50, 0x3eaf295c, 0xbe8cec88, 0xbc32ba00, 0xbe9a0e30, 0xbea0746a,
      0xbdb84258, 0xbe2dfde0, 0x3ee625fe, 0x3e12e488, 0x3e4753f8, 0x3e79a2f4,
      0x3e776090, 0xbe337cec, 0x3db5e280, 0xbeb2cefe, 0x3e8b8e00, 0x3ec806fc,
      0x3e59d6f8, 0x3de74688};
  uint32_t hidden_biases0_shape[] = {2, 4};
  uint32_t hidden_biases0_values[] = {
      0xbd130f20, 0x3efd3ec0, 0x3e38f410, 0x3c67f0c0, 0x3ee3a1ba, 0xbe031ab0,
      0x3e6147f4, 0x3ee41404, 0xbec83e98, 0xbe862d7a, 0x3eceb7d8, 0xbecfc186,
      0x3ed28de2, 0x3ed19a42, 0x3eb74124, 0x3ec5aa22, 0xbeddda26, 0x3e7da22c,
      0xbeb65808, 0xbe1156a8, 0x3e296114, 0x3effeaca, 0x3e84c718, 0x3e9f2458,
      0xbe2bc8cc, 0xbd97a438, 0x3bb33680, 0x3ed32696, 0xbe33322c, 0x3e75abf4,
      0x3d6b5420, 0xbdf48c88};
  uint32_t all_ts_out0_shape[] = {5, 2, 2, 4};
  uint32_t all_ts_out0_exp_values[] = {
      0xbe7a53bc, 0x3e9f7692, 0xbc9ed153, 0x3ee831ab, 0xbe77d13d, 0x3e31f762,
      0x3d8ddd1a, 0x3e6762b7, 0xbd9698a5, 0x3e4d00b0, 0xbe496231, 0x3e91c291,
      0xbe1c9299, 0x3ed36114, 0xbe45f6fb, 0x3ed3744e, 0xbec68241, 0x3e276d3e,
      0xbdd85a96, 0x3ec34484, 0xbed55ee0, 0x3dc4d879, 0x3c76e7a9, 0x3e49a5fe,
      0xbd67a64d, 0x3e9192cf, 0xbe51be30, 0x3ea200ca, 0xbe09e0cc, 0x3ed8b4cc,
      0xbe37dffa, 0x3eba17c2, 0xbecd3af4, 0x3dfa0768, 0xbe162afb, 0x3e8c28d0,
      0xbea2f4b5, 0x3e259552, 0xbd4ff47a, 0x3deba7d9, 0xbcda2de0, 0x3e82d495,
      0xbe3a1843, 0x3e90a6e1, 0xbe23555a, 0x3e8f0959, 0xbda17677, 0x3ebd8b84,
      0xbf0b5dde, 0x3dc3eca8, 0xbe264ef9, 0x3e0dfb28, 0xbec6e89f, 0x3d867695,
      0xbc05f6c6, 0x3cc15a89, 0x3d7d9314, 0x3eeaeca2, 0xbe61c09e, 0x3e8a8119,
      0xbdcf0300, 0x3eb7170f, 0xbc19745a, 0x3ec402a2, 0xbf02ead9, 0x3d31e00b,
      0xbdf09f2e, 0x3ddd5180, 0xbede6b61, 0x3da42fd6, 0xbdeab752, 0xbd45daf2,
      0x3d57812b, 0x3ec346a9, 0xbd12e422, 0x3e53ccf4, 0xbc8ff60e, 0x3ea5b06e,
      0x3e9c34b9, 0x3ed3c79c};
  uint32_t cell_out0_shape[] = {1, 2, 2, 4};
  uint32_t cell_out0_values[] = {
      0xbfd5981b, 0x3eab93df, 0xbeb6282e, 0x3e2b0ac4, 0xbf94f0fd, 0x3e9d97ae,
      0xbe856685, 0xbdb00498, 0xbe5181d2, 0x3eaf261e, 0xbf2f0982, 0x3fcb5ba8,
      0xbf0caf39, 0x3f471fab, 0xbf2da98a, 0x3fb4fadd};

  zdnn_ztensor *input0 =
      alloc_ztensor_with_values(input0_shape, ZDNN_3DS, test_datatype,
                                NO_CONCAT, false, (void *)input0_values);

  zdnn_ztensor *all_ts_out0 =
      test_layer(input0, h00_shape, (void *)h00_values, c00_shape,
                 (void *)c00_values, weights0_shape, (void *)weights0_values,
                 biases0_shape, (void *)biases0_values, hidden_weights0_shape,
                 (void *)hidden_weights0_values, hidden_biases0_shape,
                 (void *)hidden_biases0_values, all_ts_out0_shape,
                 (void *)all_ts_out0_exp_values, cell_out0_shape,
                 (void *)cell_out0_values, false, is_layer_bidir[0]);

  // second layer

  uint32_t h01_shape[] = {2, 2, 5};
  uint32_t h01_values[] = {0x3fc827a5, 0x3fc7d2ab, 0x3fe27e59, 0x3ea84764,
                           0x3e9400d4, 0x3f8e916a, 0x3fca1262, 0x3e688b78,
                           0x3eb894e8, 0x3f6cf872, 0x3fbc6eee, 0x3da5ca40,
                           0x3f174faa, 0x3fe12bad, 0x3d2fc9e0, 0x3e7666b8,
                           0x3faea7a3, 0x3ee02d48, 0x3fc8ba6b, 0x3f940f37};
  uint32_t c01_shape[] = {2, 2, 5};
  uint32_t c01_values[] = {0x3e1a8538, 0x3f756c1c, 0x3fda8620, 0x3faac825,
                           0x3fa2beb7, 0x3f98b1e4, 0x3f67802a, 0x3d99f2f0,
                           0x3f724b2e, 0x3fcf0846, 0x3f72e100, 0x3f054054,
                           0x3f010382, 0x3ff4fbf0, 0x3f96e796, 0x3fdf1f5c,
                           0x3fb69da2, 0x3f23c3d0, 0x3fdae58c, 0x3f20d682};
  uint32_t weights1_shape[] = {2, 8, 5};
  uint32_t weights1_values[] = {
      0x3e83aa94, 0xbea1a3c0, 0x3c6b9ce0, 0xbebfde02, 0x3ec1d528, 0xbd4e46b0,
      0xbe62d375, 0xbe0d9648, 0xbe77c691, 0xbe76cf7f, 0x3e20904c, 0xbebc9a53,
      0xbdc724ec, 0x3eca9984, 0x3e109bc4, 0xbee0a324, 0x3e6ef9a8, 0x3d737210,
      0x3ed16702, 0x3e3b6830, 0xbec39102, 0xbd3aa670, 0x3e630230, 0xbed417e7,
      0xbe0751fa, 0x3e55ae7c, 0x3eb037d8, 0xbea06a9a, 0x3ea17e98, 0x3e1940d8,
      0xbe81b1a9, 0x3e8db48a, 0x3ec02c1a, 0x3e658114, 0xbe03c1e6, 0x3e5c50f0,
      0x3e7d66c4, 0xbd472b98, 0x3e878898, 0x3e8d90bc, 0x3e7b0bac, 0xbecc0226,
      0x3cbe6420, 0xbe927f99, 0x3aa69a00, 0x3e6e6210, 0x3e8ca274, 0x3da1fbe0,
      0xbe9eba88, 0x3ecdd426, 0x3e8e02b2, 0x3d3f7208, 0xb9c68000, 0xbd938128,
      0x3ee3b00e, 0xbe91fb37, 0x3cd35960, 0x3e13e288, 0xbda1fd74, 0xbe84a2b8,
      0x3ee40ec6, 0xbe7d9782, 0x3ed942f8, 0x3e4bb92c, 0x3da325a8, 0xbe87ba02,
      0xbe4018f0, 0x3df38580, 0xbe43be48, 0x3d586020, 0x3ee497e8, 0xbe05e5dc,
      0xbe27a444, 0x3eb689d4, 0xbe56b587, 0xbedbbe59, 0xbedf0e3e, 0xbe3c776a,
      0xbea0aa84, 0xbe1e37de, 0x3ec258fa, 0xbedc1c4e, 0x3ca711d0, 0x3cc4e590,
      0x3e11fa20, 0x3cf049b0, 0xbede7c01, 0xbece783b, 0x3e592a78, 0x3e75d434,
      0xbe989ba1, 0xbdb9cd1c, 0x3ebde4ee, 0xbeb798f6, 0x3d7f2f50, 0xbe24134e,
      0x3e99f81e, 0x3e769414, 0xbe03b7b6, 0xbe9933c8, 0x3dc7ff90, 0xbed8b7ed,
      0x3ee39938, 0xbee0d2e5, 0xbeb1870e, 0xbeae8cee, 0x3ebe51d8, 0x3e675c3c,
      0xbe325de8, 0xbe0d48c0, 0x3e63833c, 0xbe37306c, 0xbe972362, 0x3dc59638,
      0x3e4904fc, 0xbecfe0c1, 0x3e062014, 0x3ebf18f2, 0xbeb3c7f8, 0x3ee1dbc6,
      0xbdcf25dc, 0xbeb5b12a, 0xbd48ccd8, 0xbe46414a, 0x3e24bcd8, 0x3d26cb70,
      0x3cdf2e90, 0xbe101be0, 0x3eacd6bc, 0xbea06bf2, 0xbea00f51, 0x3eb58a42,
      0xbeb81f6b, 0xbe3e8e36, 0xbdc54a50, 0x3ec2e956, 0xbe019774, 0x3941c000,
      0xbe9f0e96, 0x3ec6a716, 0xbd6ccb78, 0x3e578a54, 0x3eac49a0, 0x3d8d0aa8,
      0xbeafad22, 0xbea7780c, 0x3ec4c2fc, 0x3ec0e9e4, 0x3e893e48, 0xbe05d8ee,
      0x3e061770, 0x3e97de7c, 0x3d2ae830, 0xbece70c1, 0x3e163290, 0xbea2cdd5,
      0xbe703894, 0x3dd00540, 0x3ecefc06, 0xbec899a2, 0xbdfe1e6c, 0x3e0b5e64,
      0xbaa00c00, 0xbde97700, 0x3d7ba930, 0xbe2ab878, 0xbddce6cc, 0xbe5ddf86,
      0xbe95df9f, 0x3ebc4676, 0x3de69348, 0x3e208298, 0x3ee494f2, 0xbe2440b2,
      0xbe9acddc, 0x3e226e48, 0xbea5ddf2, 0x3edfa1a6, 0x3dad0a60, 0x3caa1db0,
      0xbe5fbd62, 0xbe86a497, 0xbeacb04b, 0xbe6f45de, 0xbe28e040, 0x3d21ed48,
      0xbe68730d, 0xbeb99f2f, 0x3dcaf230, 0x3ee0b168, 0x3e84b2c0, 0x3ed26dd4,
      0xbead8f69, 0xbdd20768, 0xbed93dd3, 0xbd3641c0, 0xbdf673cc, 0xbb0d5900,
      0xbe5de779, 0xbedb8204, 0x3df812c8, 0x3ee2c0f8, 0x3dd6ac68, 0x3d6a6440,
      0x3e478690, 0xbe9f3858, 0xbe0bfad6, 0x3c8a0b80, 0xbe376674, 0xbde0babc,
      0x3d971e50, 0x3e78da1c, 0x3e9124d4, 0xbe1ad584, 0x3e462330, 0x3e462a34,
      0xbe02fc74, 0xbdb961b8, 0x3def8690, 0x3ea8f792, 0xbe347690, 0xbd85c98c,
      0x3d37b120, 0xbda59be8, 0x3ca89770, 0x3ebe31d8, 0xbd9b37b0, 0xbe2a5a0c,
      0x3e95a0ea, 0x3db33be0, 0xbcf119f0, 0xbb361500, 0xbebfc12d, 0x3ccf8430,
      0x3c851e40, 0x3e4fc6b8, 0xbe2cff70, 0x3e3950e8, 0x3e2cfbb8, 0x3e9e88a0,
      0x3ee1a07c, 0xbedeee8a, 0x3eb369f6, 0xbe19b22c, 0xbd0ab3d0, 0xbe90de80,
      0xbe872d40, 0x3e662ae4, 0xbed457d0, 0xbe6741db, 0x3daf9920, 0xbe1c1d3a,
      0x3ec2326c, 0x3e3bb9b4, 0x3db92198, 0xbd1e1828, 0x3e667240, 0x3e8472c6,
      0x3edee626, 0x3e2de8c8, 0xbdc0f07c, 0xbe942d27, 0xbe0aeb80, 0xbe025ea8,
      0x3eb61b5a, 0xbdcc9010, 0x3ed7e74a, 0xbe97eae2, 0xbebd9868, 0x3eb1c556,
      0x3e917fbc, 0xbdcbff10, 0xbd084a60, 0xbecfb148, 0xbd891828, 0x3caf1b90,
      0xbda475ac, 0x3ebd81c8, 0x3de585b8, 0xbec9e6b4, 0xbe4157cc, 0x3e8ad580,
      0xbd7cdcb8, 0x3eaf736e, 0x3e858166, 0xbe736e40, 0x3ee2894e, 0x3dc70f30,
      0x3ede9074, 0x3e75fc90, 0x3e478d4c, 0x3db95270, 0x3d74a7f0, 0xbe87d88c,
      0x3e7e8034, 0xbec7475c, 0xbcf41780, 0xbdacfd44, 0xbce470b0, 0xbeb5ea1e,
      0x3beaf1c0, 0x3ee163d0, 0xbee4efd6, 0xbe377cb8, 0x3d405f70, 0xbe529b09,
      0xbe43b460, 0x3cbb9700, 0xbed30845, 0x3ed51bde, 0x3e97214e, 0xbd12c9a0,
      0xbc590b60, 0xbea69d53, 0xbe7f92d8, 0xbed52ee3, 0xbe488982, 0x3d89c8b8,
      0x3ed6e7ce, 0x3ecbb182};
  uint32_t biases1_shape[] = {2, 5};
  uint32_t biases1_values[] = {
      0xbea9c5c8, 0x3dd289d0, 0x3d851878, 0xbe5655f7, 0x3e1f747c, 0xbe134938,
      0x3ad25d00, 0xbdb01a08, 0xbd1bbbd0, 0xbeb29254, 0x3e935dc2, 0x3ed8df8e,
      0x3d1ef258, 0xbed5df49, 0x3ea0bcaa, 0x3daba420, 0xbe13420a, 0x3e9b1762,
      0x3cc83240, 0x3dabe7d8, 0xbecccd1f, 0x3dd09a78, 0xbe91286a, 0xbd4613f0,
      0x3d6b9ee0, 0xbb343e80, 0xbebe0edb, 0xbdc50970, 0x3e84b35e, 0xbe9f3779,
      0x3ec31294, 0xbe5a7ee4, 0x3e92b048, 0xbd68b2e8, 0xbe597ddb, 0xbe9a5704,
      0xbddfa3c0, 0xbdfb13dc, 0x3d9b0ca0, 0x3d980d78};
  uint32_t hidden_weights1_shape[] = {2, 5, 5};
  uint32_t hidden_weights1_values[] = {
      0x3d81ac10, 0x3dd8c630, 0xbd97418c, 0x3e10ce08, 0xbed605fa, 0x3ec2d2d6,
      0xbedf67a9, 0xbed0b705, 0x3d2a1fb8, 0xbeac7e7c, 0x3d763d80, 0xbe025fa2,
      0xbce64df0, 0x3e2abb28, 0x3e4679ac, 0xbe53df41, 0xbe9bf81e, 0x3e8fcc58,
      0xbddf28c4, 0xbe879404, 0x3ed89bd2, 0x3ec8c0ca, 0x3dedf310, 0xbe362bda,
      0x3d836668, 0xbea63515, 0xbed34fff, 0xbeb2cae3, 0xbedc6d5d, 0xbec0db92,
      0x3e49d700, 0x3e8c699a, 0x3ead673e, 0x3e9acf32, 0x3ea5bbea, 0x3d79a270,
      0x3e92763a, 0x3e0fb304, 0x3ecb49b0, 0xbed82f3e, 0x3e5f5638, 0xbecf279c,
      0x3ee267e6, 0x3e8c4992, 0x3dcc9cb0, 0xbee4ae25, 0x3e2dd470, 0xbee1c411,
      0x3e983a74, 0xbe95fc4a, 0xbecee7d6, 0x3ed4086e, 0xbe9d7ac6, 0xbe53d7b8,
      0x3ebee346, 0x3ed5ca40, 0x3ed8ea94, 0x3ed8d474, 0xbe618c84, 0xbe4a029f,
      0x3e4a7010, 0x3ecebb20, 0xbe44c542, 0xbd203aa8, 0x3e2270c4, 0x3e479f54,
      0xbec12893, 0xbe5113e1, 0xbd301208, 0x3ec72844, 0xbec61cd1, 0xbe2ccb44,
      0x3e6fec98, 0xbeb25d85, 0x3e66f338, 0x3df0bb68, 0xbce89ee0, 0x3ed04f64,
      0xbe2a0094, 0x3d93c7f8, 0x3ea117be, 0x3e18bfa8, 0x3e99bb1e, 0xbd4da508,
      0x3ddd3e70, 0xbe442dc0, 0x3e0955f0, 0x3ea0fb84, 0xbe7777df, 0x3ec92466,
      0x3e531f20, 0x3ebf9b54, 0xbd6c5ae0, 0x3e6a16c8, 0x3e26cc6c, 0xbecafb69,
      0x3ee10096, 0xbeb1b1ae, 0x3e20c074, 0xbec8cbb7, 0x3e122c34, 0x3ebc54aa,
      0x3ec058f2, 0x3ed2d41c, 0x3ea85d6a, 0x3d01b298, 0xbeb62674, 0xbe9d91cb,
      0xbe5a9fca, 0xbeb1737d, 0xbe94746f, 0xbddc5118, 0xbec0490a, 0x3e23c650,
      0x3d428f68, 0x3edaf946, 0x3e791bd4, 0xbe4e7f38, 0xbde59ef4, 0x3ed75422,
      0xbe8525dc, 0xbe3b1bb2, 0x3d940df8, 0xbdc69f70, 0xbd7bb5b8, 0xbe35e4f4,
      0xbed7e492, 0xbebb3390, 0xbe7a2866, 0x3ed07d84, 0x3da22d18, 0x3e316444,
      0xbeb70a96, 0x3e185bfc, 0xbee1383b, 0xbe340e34, 0xbe41d6c8, 0x3e8902e6,
      0x3ca49640, 0xbee3b077, 0xbd90ea54, 0xbe8b4e16, 0x3e68bf70, 0xbea3a41a,
      0x3d6ab290, 0xbed906ca, 0xbe34b29a, 0x3d740020, 0x3dc51748, 0x3eac6c0c,
      0x3eb1c89c, 0xbda0df98, 0xbc914780, 0xbd4637a8, 0xbebdb332, 0xbe0d071e,
      0x3e1469e4, 0x3e9ccdb4, 0xbcc4c620, 0x3e62e8b8, 0x3eb8509c, 0x3e33aa40,
      0xbdf7d0f0, 0x3e4c5720, 0x3e08fc84, 0xbedd7e2d, 0x3eafcf42, 0x3ec8b9ca,
      0x3e9c8b2a, 0x3e37f4f4, 0xbd840080, 0xbd88748c, 0xbe987408, 0xbd912c2c,
      0xbe8d3f0b, 0xbe8b53f4, 0x3ee473e2, 0x3ec9ef0c, 0x3dcb4df8, 0xbd1b6dd0,
      0x3dc99f48, 0x3e952b7e, 0xbee3d029, 0xbe794ffd, 0xbe19d608, 0x3ea0f704,
      0xbe80c7e3, 0x3e77fb08, 0x3d81cb10, 0x3e85eb20, 0x3e0d3144, 0x3d1e5550,
      0xbe04f4de, 0xbe94a906, 0x3d1deee8, 0x3dc63590, 0x3e69f3cc, 0xbee03730,
      0xbeac4f21, 0x3eb2ba24};
  uint32_t hidden_biases1_shape[] = {2, 5};
  uint32_t hidden_biases1_values[] = {
      0x3df84f58, 0x3e9ba3fc, 0x3ec36c40, 0x3eb9b38a, 0xbea9a47a, 0xbd059ee0,
      0xbe847d16, 0x3e03a480, 0x3e826528, 0xbe8f0d14, 0xbec763d2, 0xbed21657,
      0x3edf9e2a, 0xbde46c64, 0x3cb6fdc0, 0x3e683924, 0xbe9dbc3c, 0xbd014578,
      0x3e801014, 0x3ec60e30, 0xbd92bdd4, 0xbe877dfe, 0xbe82d308, 0x3dd941a0,
      0x3ed6ece6, 0xbe84df2a, 0x3e4a6960, 0xbd005890, 0xbed843d9, 0xbe8405ca,
      0xbeb76e74, 0x3e8a9360, 0xbe6d6e1d, 0xbdb26478, 0x3eb5c09c, 0x3e39b660,
      0xbe1b530c, 0x3e9e0b48, 0x3eb50338, 0x3d853c28};
  uint32_t all_ts_out1_shape[] = {5, 2, 2, 5};
  uint32_t all_ts_out1_exp_values[] = {
      0xbdd1fa2b, 0xbd3f4ead, 0x3e0461ac, 0x3ee96b3b, 0x3eb74b45, 0x3e90e20c,
      0x3cc3de88, 0xbe6c0e8e, 0x3e8982be, 0x3e9c9c61, 0xbcc322be, 0xbde8fc7c,
      0xbd1c8d3e, 0xbd2b6c1d, 0xbe42fb01, 0x3c57f78f, 0xbdc04b1a, 0xbd33a392,
      0xbb95a554, 0xbe5066f9, 0xbd9fc1a4, 0xbe477008, 0xbdf132ab, 0x3e9bcf26,
      0x3eb93873, 0x3ddaf977, 0xbdd3166c, 0xbe619ffb, 0x3e4adfd0, 0x3e9fccd2,
      0xbcc2dbcd, 0xbdf980ee, 0xbccdb7ad, 0xba30c49d, 0xbe42c10d, 0x3b83d9f7,
      0xbdcf460e, 0xbd25edf3, 0x3ccf389e, 0xbe583262, 0xbdab0f55, 0xbe7eb2ff,
      0xbe4f90b5, 0x3e3ad707, 0x3eaa53e0, 0x3bddcfca, 0xbe1775a7, 0xbe7ea785,
      0x3dfb50db, 0x3e94d66d, 0xbd37abbd, 0xbe0e311e, 0x3b99893a, 0x3d3ff63a,
      0xbe32a8f4, 0xbc20a1ff, 0xbdd2936b, 0xbca7aae4, 0x3dceaf3e, 0xbe526074,
      0xbda8f646, 0xbea1ae04, 0xbe7407c4, 0x3de07121, 0x3eab13d6, 0xbd35fd1d,
      0xbe3dd8b8, 0xbe8900d2, 0x3d99c517, 0x3e8fad14, 0xbd509783, 0xbe25d588,
      0x3d1331e9, 0x3e1c1ebc, 0xbdde9924, 0x3cc65017, 0xbdb670f0, 0x3b42adb7,
      0x3e92ec47, 0xbe359c1f, 0xbdb2ab28, 0xbea20225, 0xbe84c3cf, 0x3d93e6e6,
      0x3e9ee28e, 0xbd8f777c, 0xbe39d1e1, 0xbe997113, 0x3d0e7f2b, 0x3e6fee78,
      0xbc9bfb45, 0xbe70f085, 0x3da7354e, 0x3e8801bd, 0x3db26c6d, 0x3e37c0bd,
      0xbc2f97f7, 0x3d7034b4, 0x3e824266, 0xbde7ae61};
  uint32_t cell_out1_shape[] = {1, 2, 2, 5};
  uint32_t cell_out1_values[] = {
      0xbe64bd12, 0xbf184520, 0xbf34c5c0, 0x3e25569a, 0x3f2f7ae7,
      0xbe274afb, 0xbeabac90, 0xbf47b603, 0x3db135ad, 0x3f08b30b,
      0xbd87fabc, 0xbe95e9f6, 0xbd845920, 0xbd816afa, 0xbebb7434,
      0x3d1341a1, 0xbe8512e6, 0xbd941748, 0xbbe9b318, 0xbeb82cf6};

  zdnn_ztensor *all_ts_out1 = test_layer(
      all_ts_out0, h01_shape, (void *)h01_values, c01_shape, (void *)c01_values,
      weights1_shape, (void *)weights1_values, biases1_shape,
      (void *)biases1_values, hidden_weights1_shape,
      (void *)hidden_weights1_values, hidden_biases1_shape,
      (void *)hidden_biases1_values, all_ts_out1_shape,
      (void *)all_ts_out1_exp_values, cell_out1_shape, (void *)cell_out1_values,
      is_layer_bidir[0], is_layer_bidir[1]);

  free_ztensor_buffers(3, input0, all_ts_out0, all_ts_out1);
}

void lstm_bidir_to_fwd() {

  // num_timesteps = 5
  // num_batches = 2
  // num_features = 4
  // num_hidden = 5, 4

  bool is_layer_bidir[] = {true, false};

  // first layer

  uint32_t input0_shape[] = {5, 2, 4};
  uint32_t input0_values[] = {
      0x3f80f554, 0x3eed5744, 0x3fe9598b, 0x3fde3340, 0x3fb14cbd, 0x3f3b5a0a,
      0x3f82893d, 0x3e5414c8, 0x3f8b5bf7, 0x3f3c425a, 0x3fa6aeeb, 0x3f99290e,
      0x3ffa48dc, 0x3fd4c5a9, 0x3fb4c3ba, 0x3f768450, 0x3f1acb50, 0x3eccc9d0,
      0x3fd6c6c6, 0x3fb7bd3f, 0x3f230434, 0x3e2daec8, 0x3f9a57a9, 0x3e80dd48,
      0x3f94a1a8, 0x3f64e95e, 0x3dc195b0, 0x3ff6bde7, 0x3fd094b3, 0x3fa067b8,
      0x3fb1e4f7, 0x3e0b4360, 0x3fd2f78d, 0x3fbaec30, 0x3fd96d0d, 0x3ff7e13b,
      0x3fcab802, 0x3e0fc588, 0x3f0dc4a2, 0x3f03ec80};
  uint32_t h00_shape[] = {2, 2, 5};
  uint32_t h00_values[] = {0x3f72895c, 0x3fc19f9d, 0x3f54b050, 0x3ff7834f,
                           0x3fdc7d0d, 0x3fc1fce3, 0x3ebcf5b4, 0x3ed3cdb4,
                           0x3fb8c472, 0x3f849e59, 0x3eb88b80, 0x3bc03f00,
                           0x3f1a65ee, 0x3f5d6a8e, 0x3ea8b604, 0x3fcb5de0,
                           0x3f504bc2, 0x3fe33d36, 0x3fd8b70c, 0x3fc21f69};
  uint32_t c00_shape[] = {2, 2, 5};
  uint32_t c00_values[] = {0x3f0c2aba, 0x3f190c04, 0x3fcbd235, 0x3f32a91c,
                           0x3ee6ed24, 0x3f9027e4, 0x3f7639bc, 0x3f44af00,
                           0x3ec25e00, 0x3d230b80, 0x3fe2a3cb, 0x3faee87b,
                           0x3f1b63b4, 0x3e2f90c0, 0x3e04e860, 0x3df0eef0,
                           0x3f4d0d62, 0x3fef4e7c, 0x3f68732e, 0x3fd013d6};
  uint32_t weights0_shape[] = {2, 4, 5};
  uint32_t weights0_values[] = {
      0x3ed76812, 0xbeda1e9d, 0xbcc9dc90, 0xbe8b56d8, 0xbde3f398, 0x3e9a494e,
      0xbc03b300, 0x3d898450, 0x3ecfc37a, 0x3ca54f60, 0xbe47ad20, 0xbeac6e30,
      0xbe3b8b06, 0x3e9cea58, 0x3d85a140, 0xbde68434, 0xbeb09ec1, 0x3e87de1e,
      0xbec116de, 0x3dd939f0, 0x3d190310, 0x3e9e8ea0, 0x3e55de1c, 0x3ee46d12,
      0x3eace9cc, 0x3d64ba10, 0x3ece54c0, 0xbe5ec821, 0x3e915990, 0x3ee4e29a,
      0xbeb401c3, 0x3e86944e, 0xbebdbf60, 0xbe9e462c, 0x3ec83690, 0x3d0ff4f8,
      0x3d757710, 0xbec4e626, 0x3e332b14, 0x3eb165e2, 0xbed56486, 0x3dab6e00,
      0x3e301b34, 0x3ea3ea60, 0x3e64a8e0, 0x3ecb70ec, 0xbd9a4a9c, 0xbe879f2a,
      0x3e2b3b3c, 0x3dbfb2a0, 0x3eae1a26, 0xbd96b870, 0x3e27e118, 0xbee29f7d,
      0xbeb29e53, 0xbee46847, 0xbe51a0d6, 0x3e67965c, 0xbe9488c8, 0xbe83d8ea,
      0xbe5cf1d4, 0x3db3dd28, 0x3e354500, 0x3e1f0c64, 0xbe6f1e1b, 0x3da48928,
      0xbeaa02a3, 0xbd669538, 0xbe271822, 0x3b9084c0, 0xbe6de235, 0x3d987eb0,
      0xbebbb5a4, 0x3e2dd3f4, 0x3e3d5c4c, 0x3dd306b8, 0x3e430d88, 0xbd8d3050,
      0x3e987cda, 0x3d0f9250, 0x3e33b614, 0xbedba477, 0xbe20347e, 0xbe952b16,
      0x3c4ba340, 0x3cdb72e0, 0xbe87b6cc, 0xbebbabe9, 0xbcbc0320, 0xbd1e06b8,
      0xbe5f7204, 0xbde54684, 0xbdc2d30c, 0x3ea3e928, 0x3db95c18, 0x3e58b204,
      0xbe5f126c, 0x3dc3c730, 0xbd3c4ee0, 0x3e91971e, 0xbdd1fa20, 0xbe89ca96,
      0xbe8f83dc, 0xbda4ce70, 0xbe1b9f76, 0xbe20b66c, 0x3ecdd9da, 0x3e036284,
      0x3e3248cc, 0x3e5f481c, 0x3ebbff92, 0x3e5d42c8, 0xbe2089f6, 0x3da2aec0,
      0xbe9fee66, 0x3e804fe0, 0x3e79a49c, 0x3eb2e8ec, 0x3e42d780, 0x3dcedbf0,
      0x3ed8e4e6, 0xbecdb113, 0xbe639ee9, 0xbdb9c6e0, 0x3e8ed458, 0x3e916930,
      0xbe591a24, 0xbeaf7709, 0x3e7b2408, 0xbeb32c7f, 0x3ed963f2, 0x3e9be8be,
      0xbed64902, 0x3e7795b4, 0xbdc1a118, 0xbd047a58, 0xbb3dfe00, 0xbec14162,
      0xbedcd447, 0xbe39af54, 0xbe9aebe8, 0x3e77c3f8, 0xbe9c4230, 0xbead1b0c,
      0x3ebc5350, 0xbee1e40d, 0x3ed6d390, 0xbd93fe10, 0xbe5f6ed0, 0x3ebb7968,
      0xbeb7e0dd, 0x3eb76f78, 0xbe94b3e4, 0xbedd358e, 0x3ea3e864, 0x3eb4c482,
      0xbecc7bce, 0x3e9eff90, 0xbe81ffa5, 0x3eb0cbec};
  uint32_t biases0_shape[] = {2, 5};
  uint32_t biases0_values[] = {
      0xbe18ef8e, 0xbdfff1d8, 0x3e074a8c, 0x3ebc395c, 0x3df90638, 0x3eb037d8,
      0x3e8db48a, 0x3e7d66c4, 0x3c6b9ce0, 0xbe0d9648, 0xbe83e971, 0x3e83a64e,
      0x3e72843c, 0x3eafbdf8, 0x3e261b9c, 0xbea1a3c0, 0xbe62d375, 0xbebc9a53,
      0x3e6ef9a8, 0xbd3aa670, 0xbe58ace5, 0xbe574539, 0xbed87cba, 0x3dc5e080,
      0xbe807ef0, 0xbdc724ec, 0x3d737210, 0x3e630230, 0xbea06a9a, 0x3ec02c1a,
      0xbecc25a9, 0x3ec8c806, 0xbe94b2a7, 0xbd506310, 0x3e0332e4, 0xbd472b98,
      0xbebfde02, 0xbe77c691, 0x3eca9984, 0x3ed16702};
  uint32_t hidden_weights0_shape[] = {2, 5, 5};
  uint32_t hidden_weights0_values[] = {
      0x3e21ba5c, 0x3eb5ccc2, 0xbe7c17a9, 0x3c9a1ea0, 0x3ea04420, 0xbddc3fcc,
      0x3e9cab54, 0xbe856c8e, 0x3e9399fc, 0xbe249da6, 0xbdc09e0c, 0xbea1834a,
      0x3e1c8044, 0xbead15e7, 0x3ec122ee, 0xbebcb154, 0x3ca36520, 0xbe8872c0,
      0xbe9d3169, 0x3e2598d4, 0x3e7822c0, 0xbea16940, 0xbc1fdba0, 0xbe84f4be,
      0x3d67f3b0, 0xbeadd2f6, 0xbe8219a4, 0xbe190f96, 0x3cc42620, 0xbed14480,
      0x3edb2580, 0xbd8f889c, 0xbdf89d64, 0xbe8d6097, 0xbda3e468, 0x3eade05a,
      0x3ec26bea, 0xbdb0b3c4, 0x3cbbc620, 0xbeaa2909, 0xbe26eada, 0x3c976030,
      0x3d2f4d98, 0x3e5b9460, 0xbe436636, 0x3e80b7ea, 0xbea1ef22, 0x3ed3c2e8,
      0x3e6328f4, 0x3e24fec4, 0xbdb06a40, 0x3bbd8300, 0x3ec95d16, 0x3e90def0,
      0x3d448f50, 0xbdb3d438, 0xbe5008a0, 0xbe3acaf4, 0xbd5360c8, 0xbdbfc268,
      0xbecd7820, 0x3e2c8218, 0x3ecec37e, 0xbe4c6e38, 0x3ea38344, 0xbec5b44b,
      0xbece16e2, 0x3e25d8dc, 0x3eb7196c, 0x3dffaaa8, 0xbd8f518c, 0xbcf73e90,
      0xbecea3b7, 0xbeb90843, 0x3e5c0bbc, 0x3ecb999a, 0x3ee4e636, 0x3d527bf8,
      0x3e99b77e, 0x3e926d0a, 0xbdea86c4, 0x3e6d851c, 0xbddabb30, 0x3ea2c8c8,
      0x3d082a00, 0x3e497f5c, 0x3e8e5746, 0xbe811315, 0xbd507b08, 0xbe58367b,
      0x3ecbffd4, 0xbe20471c, 0xbe807b12, 0xbe8480f1, 0x3ed26ffe, 0x3e177f54,
      0x3e5bbc44, 0xbeb030cc, 0x3eb0f98a, 0x3e3b45e4, 0xbb5a2c80, 0x3d69b920,
      0xbd0388a0, 0x3e62db50, 0x3ba04b00, 0xbdb50e24, 0xbeb31aa8, 0x3ee28262,
      0xbea317e4, 0x3ebcdbe4, 0x3d65d3e0, 0xbe700636, 0x3e256d64, 0xbe01fea6,
      0x3d64ada0, 0xbdd123b0, 0x3eb8e4f2, 0x3c4f7420, 0xbe19a60a, 0x3ecd16fc,
      0xbb4def80, 0xbeb9d294, 0xbec66e3b, 0xbd87a1dc, 0x3e8c10dc, 0xbea299d4,
      0xbe24134e, 0xbedc1c4e, 0x3ebe51d8, 0x3ebde4ee, 0x3e2a3b28, 0x3dc7ff90,
      0xbede7c01, 0xbe37306c, 0x3e769414, 0x3ec258fa, 0xbeae8cee, 0xbdb9cd1c,
      0x3e062014, 0x3ee39938, 0x3cf049b0, 0x3e63833c, 0x3e99f81e, 0x3ca711d0,
      0x3e675c3c, 0xbe989ba1, 0xbecfe0c1, 0xbed8b7ed, 0xbece783b, 0xbe972362,
      0xbe90c29a, 0x3e1735dc, 0xbedd7037, 0xbe71302b, 0xbe295dd2, 0x3e4d6418,
      0x3eabc840, 0x3ea58b16, 0x3d721bf0, 0xbee2f252, 0xbea1dc41, 0xbe136b9a,
      0xbebd57dc, 0x3ebd57a4, 0x3e4eb6e0, 0xbe4216de, 0x3a3bae00, 0x3bc9f900,
      0xbe05dde4, 0xbe5bef69, 0x3e06b148, 0x3e6bc304, 0xbd9bb79c, 0xbe87f0ac,
      0xbe98cd9b, 0x3ebf18f2, 0xbee0d2e5, 0x3e75d434, 0x3e4904fc, 0xbee0a324,
      0x3cc4e590, 0xbe325de8, 0x3d7f2f50, 0x3ee1dbc6, 0xbec39102, 0x3e592a78,
      0x3dc59638, 0xbe9933c8, 0x3e83aa94, 0x3e55ae7c, 0xbeb798f6, 0xbeb3c7f8,
      0xbeb1870e, 0xbd4e46b0, 0xbe81b1a9, 0xbe03b7b6, 0x3e11fa20, 0xbe0d48c0,
      0x3e20904c, 0x3e5c50f0};
  uint32_t hidden_biases0_shape[] = {2, 5};
  uint32_t hidden_biases0_values[] = {
      0x3ebdf640, 0xbe1985b8, 0x3e06e404, 0xbdd1716c, 0xbca2ec60, 0xbe76cf7f,
      0x3e109bc4, 0x3e3b6830, 0xbe0751fa, 0x3e1940d8, 0x3c696e40, 0xbe9a126e,
      0xbeb9158a, 0x3eb682ba, 0x3d952498, 0xbed417e7, 0x3ea17e98, 0x3e658114,
      0x3e878898, 0x3ec1d528, 0xbe8b7474, 0x3e1ae0c4, 0x3e10c984, 0xbc172600,
      0xbdcdfc84, 0xbe03c1e6, 0x3e8d90bc, 0xbdfe1e6c, 0xbe2ab878, 0x3de69348,
      0xbe8d2560, 0x3e9e5868, 0xbeaa740b, 0x3e3b9ae0, 0xbe19fc78, 0x3e226e48,
      0xbe5fbd62, 0x3d21ed48, 0x3e84b2c0, 0xbd3641c0};
  uint32_t all_ts_out0_shape[] = {5, 2, 2, 5};
  uint32_t all_ts_out0_exp_values[] = {
      0x3dd6eff8, 0xbda76b31, 0x3c6ef7c1, 0x3a8b388f, 0x3dae9ed6, 0x3e89e8f4,
      0xbb4586cb, 0xbc9f3675, 0x3cc8b2ea, 0x3dfe784b, 0x3dbb39fe, 0x3edafae8,
      0xbd3c005a, 0xbe072b73, 0x3ef9a34c, 0xbc786331, 0x3eafda4e, 0xbdad66db,
      0xbe976dfa, 0x3ee8b575, 0x3c002849, 0xbe597b0f, 0xbcc51dc1, 0x3d32e3f7,
      0x3e51f770, 0x3e959d05, 0xbe4629d9, 0xbd0f892a, 0xbb62976b, 0x3de59bf1,
      0x3dac9b77, 0x3ee69149, 0xbd32d750, 0xbe094cd1, 0x3f0894c9, 0xbb945ae6,
      0x3ec70e1c, 0xbd0701f9, 0xbe079847, 0x3f19f4f5, 0xbdb2f5c6, 0xbe6a4e4b,
      0xbd405672, 0x3dd3f4cc, 0x3e178532, 0x3b19974b, 0xbe5f5abf, 0xbdc69fcb,
      0x3df1954a, 0x3d01bdbf, 0x3e1c36e6, 0x3ec6ffdf, 0xbbd5b4c9, 0xbd7a61de,
      0x3f084cd4, 0xbbf96418, 0x3e21a121, 0x3ca90e88, 0xbe61fed1, 0x3f0518e3,
      0xbd13ed8d, 0xbe3bded3, 0xbd99b60d, 0xbd102e12, 0x3dfeec81, 0xbe1233ea,
      0xbe61b886, 0xbd95006c, 0x3dfef1ce, 0xbd3d39b1, 0x3e843fa8, 0x3e94e115,
      0x3d84b656, 0x3cbd3390, 0x3f0ea868, 0xbc26d2c6, 0x3e28e672, 0x3d9b7799,
      0xbc5c9196, 0x3f3aef4b, 0xbd8473f6, 0xbe51d0bc, 0xbd172f10, 0xbb8eeb58,
      0x3dcde949, 0xbddf3a74, 0xbe4f4c2e, 0xbe02e647, 0xbca72b3d, 0xbdb6fbf5,
      0x3e44f9ce, 0x3ee3b9b2, 0x3d5241b9, 0x3d90a145, 0x3ee73202, 0xbc911f4e,
      0xb9c44b8a, 0x3e04e5ab, 0x3e8f7fe2, 0x3f11f168};
  uint32_t cell_out0_shape[] = {1, 2, 2, 5};
  uint32_t cell_out0_values[] = {
      0xbdb0381d, 0xbec713e6, 0xbf77a646, 0xbc19d620, 0x3e8fb6a4,
      0xbe499947, 0xbecf624a, 0xbf4b69e0, 0xbd2dd5de, 0xbe1f50a7,
      0x3e71bac2, 0x3fe517ec, 0xbdfa26c5, 0xbedb0254, 0x3f1f2b76,
      0xbd5f952c, 0x3f279d82, 0xbe9c2544, 0xbf4a2172, 0x3f1eaed8};

  zdnn_ztensor *input0 =
      alloc_ztensor_with_values(input0_shape, ZDNN_3DS, test_datatype,
                                NO_CONCAT, false, (void *)input0_values);

  zdnn_ztensor *all_ts_out0 =
      test_layer(input0, h00_shape, (void *)h00_values, c00_shape,
                 (void *)c00_values, weights0_shape, (void *)weights0_values,
                 biases0_shape, (void *)biases0_values, hidden_weights0_shape,
                 (void *)hidden_weights0_values, hidden_biases0_shape,
                 (void *)hidden_biases0_values, all_ts_out0_shape,
                 (void *)all_ts_out0_exp_values, cell_out0_shape,
                 (void *)cell_out0_values, false, is_layer_bidir[0]);

  // second layer

  uint32_t h01_shape[] = {1, 2, 4};
  uint32_t h01_values[] = {0x3f32172c, 0x3f9edf37, 0x3f2645a8, 0x3fdcb8f3,
                           0x3fcb4487, 0x3fc0f8ba, 0x3da5dda0, 0x3fa27159};
  uint32_t c01_shape[] = {1, 2, 4};
  uint32_t c01_values[] = {0x3f805978, 0x3fbe03f3, 0x3ae02000, 0x3cbf1e40,
                           0x3fe08930, 0x3fe7c408, 0x3fe105ea, 0x3f809e08};
  uint32_t weights1_shape[] = {1, 10, 4};
  uint32_t weights1_values[] = {
      0xbe2bd65c, 0x3dcefba8, 0xbedd7f20, 0x3e861f28, 0xbeed681e, 0x3e426fc0,
      0x3eed0f5e, 0xbe6cd8e8, 0x3e51e224, 0x3ecb9a06, 0x3ef28514, 0xbe7c2bdc,
      0x3ef9356c, 0x3ec6c0fc, 0x3ee721da, 0xbd332440, 0xbe11c200, 0xbd9946b0,
      0xbed7f530, 0xbd44da50, 0xbed3f95a, 0xbee1bb7e, 0xbe413088, 0xbec76b22,
      0xbee83510, 0xbee753d8, 0xbeb0114a, 0x3ed56b46, 0x3e004e90, 0x3eef0648,
      0x3ef26c56, 0xbe61dab4, 0xbd1b12f0, 0x3e625510, 0xbe5bff04, 0x3e359d30,
      0xbe8146d8, 0x3e5f2f40, 0xbe69c184, 0x3edeaa24, 0xbe0282c4, 0xbead1388,
      0xbdd7b678, 0xbde41328, 0xbea7904e, 0x3cbe3200, 0xbde4b548, 0xbdb7df18,
      0xbe37a3e0, 0xbe3ccf20, 0x3ea2ac3c, 0x3ec89742, 0x3dc17718, 0x3efb36e4,
      0x3cc3c6c0, 0x3e80aae6, 0xbe85c1f4, 0xbef2e226, 0xbef93ece, 0x3ed91e6a,
      0x3de2e688, 0xbef56ad2, 0xbe9721d8, 0x3e9414ee, 0xbdead1c8, 0x3efc4230,
      0xbe2e8a7c, 0xbe1b5134, 0xbe7818b0, 0xbea1f7f8, 0x3e80d2ca, 0xbea9d954,
      0x3d8caec8, 0x3dc45320, 0xbea5aa8e, 0xbd1860a0, 0x3ed27f84, 0xbd30c140,
      0x3ef1632e, 0x3ed3e00e, 0x3e811ae2, 0x3ee072e2, 0xbe4bac78, 0xbe94dd26,
      0x3d90fa50, 0xbda91c00, 0xbeef490e, 0x3ed28f66, 0x3ed9d1c4, 0xbee959b2,
      0xbec0dab0, 0xbecba66e, 0x3d89a708, 0xbd00be80, 0x3e5de6f0, 0xbdf65258,
      0xbe6ce154, 0x3ea0c574, 0xbe9794be, 0x3e8b418a, 0x3ef22d06, 0x3e050490,
      0x3d92e8e0, 0xbe51317c, 0x3df25c60, 0x3e21e58c, 0x3e236d10, 0x3ed70d0e,
      0xbef9c638, 0x3d3e3450, 0x3d1101a0, 0xbeb02b06, 0xbe11c318, 0x3e3ee218,
      0xbea5fa40, 0xbed6fb44, 0xbeae60fe, 0xbdf97fe8, 0x3ef4d1f0, 0xbe66dee0,
      0x3da587b0, 0xbd8cb5c8, 0xbd988fc8, 0xbda24ed0, 0x3eebb6a8, 0x3ec6c4a6,
      0xbca26d60, 0xbed4174e, 0xbe746de4, 0xbe1dac84, 0x3eaf4fca, 0x3e7db128,
      0x3e371b0c, 0x3ece1200, 0xbe0a8890, 0x3e1927cc, 0xbe005cac, 0xbef7a2f6,
      0x3ee06b10, 0x3e4dab88, 0xbdde1128, 0xbd939528, 0xbeaa72aa, 0xbe9deb0a,
      0x3ebc3cd8, 0xbdb3dca8, 0xbd5d9d20, 0x3ea4c2ae, 0xbec6657a, 0x3e25ee78,
      0xbcdc0000, 0x3ef278c0, 0x3d598660, 0x3e48df24, 0x3e6475a0, 0x3d31b530,
      0x3ef0bea8, 0x3ec48fa6, 0x3eaf0566, 0xbeef1dcc};
  uint32_t biases1_shape[] = {1, 4};
  uint32_t biases1_values[] = {0x3e382eac, 0xbeb35b5c, 0xbddc93c8, 0x3ede19b8,
                               0x3d9db078, 0x3e997152, 0xbee6ceb4, 0x3ee76a6a,
                               0xbec4697e, 0xbe15a55c, 0x3e27ed08, 0xbee0471c,
                               0x3e8c56b8, 0x3e85429c, 0x3e9ec5ca, 0xbea3364a};
  uint32_t hidden_weights1_shape[] = {1, 4, 4};
  uint32_t hidden_weights1_values[] = {
      0xbedeecba, 0x3ccc9720, 0x3ecf9ed2, 0xbe92441c, 0xbeeae27c, 0x3e0acf3c,
      0xbebdaa84, 0x3df2e668, 0x3efa0328, 0x3eae02ee, 0xbda40fe0, 0x3ef04b3c,
      0xbdff6298, 0x3eda7d48, 0xbe977c1e, 0xbecd1526, 0x3eb3b59c, 0xbe6fa27c,
      0xbea24a9c, 0xbe74491c, 0xbebdcfc0, 0x3e3246b0, 0xbd5d7530, 0x3ea400ba,
      0x3deb6398, 0xbee4f98a, 0x3d83b748, 0xbd821528, 0x3d94ce30, 0x3de939c8,
      0x3eda1908, 0xbe7329bc, 0x3e9aeeae, 0xbde79930, 0xbd845f50, 0xbecb234c,
      0xbe84ba3c, 0x3d3a7b70, 0xbebb3c68, 0x3cf98660, 0xbdc772e8, 0xbeb2f3cc,
      0x3e15eb3c, 0x3ecaf7cc, 0x3ecb3492, 0x3ed9eaec, 0xbeb6053c, 0xbe10e348,
      0x3e70fb40, 0xbd608060, 0x3ec09f96, 0xbe5da7e8, 0x3edbfc7a, 0xbe211e60,
      0x3ed7af1a, 0x3ec13d5a, 0x3ea9cb78, 0xbecddb00, 0x3d3f1470, 0xbe550c2c,
      0xbe8649a2, 0x3958a000, 0x3de892f0, 0xbeb1d4c4};
  uint32_t hidden_biases1_shape[] = {1, 4};
  uint32_t hidden_biases1_values[] = {
      0xbee4169c, 0x3e9d3bf8, 0x3d560ae0, 0x3cec4ba0, 0x3efef9ec, 0xbe97bf38,
      0x3eff933e, 0xbef5ae46, 0xbe8dc31c, 0xbe56c57c, 0xbe15b3d0, 0xbef96240,
      0x3cd4dd20, 0x3db51a80, 0xb9ddf000, 0x3e255720};
  uint32_t all_ts_out1_shape[] = {5, 1, 2, 4};
  uint32_t all_ts_out1_exp_values[] = {
      0x3d83a5ef, 0x3ec5ab29, 0xbe96f2a9, 0xbdf8dcd8, 0x3de2bd50, 0x3ed99a83,
      0x3e415f85, 0x3cfbc484, 0xbe8c59a2, 0x3e5053ab, 0x3c0308c4, 0xbe9ffe19,
      0xbe8faac0, 0x3e83c5a5, 0x3e14e22c, 0xbe1cad7e, 0xbec9f76f, 0xbd0a8fd7,
      0x3e1e0707, 0xbebd7c2d, 0xbecd1305, 0x3c73fc9e, 0x3e550bbb, 0xbe8989fa,
      0xbedd0f9f, 0xbe0144b8, 0x3e53529a, 0xbebd2a6c, 0xbef77903, 0xbe2d44ac,
      0x3e92e511, 0xbea59a45, 0xbee650d7, 0xbe2e4cba, 0x3e64bf3b, 0xbecdd76e,
      0xbefc2978, 0xbe72658c, 0x3e8fef57, 0xbeb46b98};
  uint32_t cell_out1_shape[] = {1, 1, 2, 4};
  uint32_t cell_out1_values[] = {0xbf638f5a, 0xbe9a2d53, 0x3edea444,
                                 0xbf91ec0e, 0xbf8d6301, 0xbec78653,
                                 0x3f0e8a9f, 0xbf885ceb};

  zdnn_ztensor *all_ts_out1 = test_layer(
      all_ts_out0, h01_shape, (void *)h01_values, c01_shape, (void *)c01_values,
      weights1_shape, (void *)weights1_values, biases1_shape,
      (void *)biases1_values, hidden_weights1_shape,
      (void *)hidden_weights1_values, hidden_biases1_shape,
      (void *)hidden_biases1_values, all_ts_out1_shape,
      (void *)all_ts_out1_exp_values, cell_out1_shape, (void *)cell_out1_values,
      is_layer_bidir[0], is_layer_bidir[1]);

  free_ztensor_buffers(3, input0, all_ts_out0, all_ts_out1);
}

int main() {
  UNITY_BEGIN();

#ifdef TEST_AIU
  RUN_TEST_ALL_DATATYPES(lstm_fwd_to_fwd);
  RUN_TEST_ALL_DATATYPES(lstm_fwd_to_bidir);
  RUN_TEST_ALL_DATATYPES(lstm_bidir_to_bidir);
  RUN_TEST_ALL_DATATYPES(lstm_bidir_to_fwd);
#endif
  return UNITY_END();
}
