/* -*- c++ -*-
 *
 * protocoliface.h
 *
 * Copyright (C) 2003 Sebastian Sauer <mail@dipe.org>
 * Copyright (C) 2003 Petter Stokke <ummo@hellokitty.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#ifndef __libkmldonkey_protocoliface_h__
#define __libkmldonkey_protocoliface_h__

#include <QObject>

#include "hostiface.h"
#include "infoiface.h"

//!
/**
 * Interface-class for all the different gui-protocols. Each protocol
 * have to inherit this class and overwrite the functions as needed.
 */
class ProtocolInterface : public QObject
{
    Q_OBJECT
public:
    /**
     * Constructor
     */
    ProtocolInterface(const QString& corename, QObject* parent = 0) : QObject(parent) {
        this->corename = corename;
        host = 0;
    }
    /**
     * Destructor
     */
    ~ProtocolInterface() {}

    /**
     * Each core does have an unique name. For the mlnet the name is "mldonkey"
     */
    QString getCoreName() { return corename; }

    /**
     * Each protocol does should have one HostInterface based class
     * for host-based options.
     */
    HostInterface* getHost() { return host; }
    void setHost(HostInterface* h) { host = h; }

    /**
     * Is the protocol connected with core?
     * \return results true if a connection to core is established and false if there
     *         isn't a connection right now.
     */
    virtual bool isConnected() { return false; }
    /**
     * Connect with core.
     * \return Protocol trues to connect with core (true) or wasn't able to (false).
     *         If the result is true wait for signalConnected() to wait till the
     *         connection is established.
     */
    virtual bool connectToCore() { return false; }
    /**
     * Disconnect from core.
     * \return protocol does try to disconnect now (true) or isn't able to (false).
     *         If the result is true you should evaluate signalDisconnected() to
     *         know if the disconnect was successfully.
     */
    virtual bool disconnectFromCore() { return false; }

    /**
     * Used to indicate why a connection to the core was closed.
     * Connect with signalDisconnected(int status) to handle this
     * cases.
     */
    enum enumDisconnectError {
        NoError = 0, // No error, disconnect was intentional.
        HostNotFoundError, // Could not find host, connection not possible.
        ConnectionRefusedError, // Connection to the core was refused.
        IncompatibleProtocolError, // Protocol isn't compatible.
        AuthenticationError, // Incorrect authentification (name/password).
        CommunicationError // A read error occurred when communicating with the core. The connection has been terminated.
    };
#if 0
    /**
     * Checks if a InfoDict with that key exists.
     * \param key Unique name of the InfoDict
     * \return true if such a InfoDict exists else false
     */
    bool hasInfo(const QString& key) {
        return infomap.contains(key);
    }
    /**
     * Returns the with key and id defined InfoInterface-instance.
     * \param key Unique name of the InfoDict
     * \param id Unique number of the InfoInterface-instance inside the InfoDict.
     * \return The InfoInterface-instance or 0L if there exists no such instance.
     */
    InfoInterface* getInfo(const QString& key, int id) {
        return infomap.contains(key) ? (InfoInterface*)( infomap[key].find(id) ) : 0L;
    }
    /**
     * Returns the with key defined QIntDict.
     * \param key Unique name of the InfoDict
     * \return The InfoDict or 0L if there exists no such InfoDict.
     */
    InfoDict getInfos(const QString& key) {
        return infomap[key];
    }
#endif //deprecated?
signals:
    /**
     * This signal is emitted if the connection to the core was established
     * successfully.
     */
    void signalConnected();

    /**
     * This signal is emitted if connection to core failed.
     * \param err The enumDisconnectError value that contains the reason for the disconnect.
     */
    void signalDisconnected(int err);

private:
    QString corename;
    HostInterface* host;

protected:
    //InfoMap infomap; // each protocol have to add there supported InfoInterface-classes into this map.
};

#endif
