"""Tests for terminating machines and cleaning up the environment"""

from juju import errors
from juju.lib import testing
from juju.machine import ProviderMachine
from juju.providers.openstack.machine import NovaProviderMachine
from juju.providers.openstack.tests import OpenStackTestMixin


class ShutdownMachineTests(OpenStackTestMixin, testing.TestCase):

    def test_shutdown_single(self):
        self.expect_nova_get("servers/1000",
            response={"server": {
                'name': "juju testing instance 0",
            }})
        self.expect_nova_delete("servers/1000", code=204)
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_provider().shutdown_machine(machine)
        deferred.addCallback(self.assertIs, machine)

    def test_shutdown_single_other(self):
        self.expect_nova_get("servers/1000",
            response={"server": {
                'name': "some other instance",
            }})
        self.mocker.replay()
        machine = NovaProviderMachine(1000)
        deferred = self.get_provider().shutdown_machine(machine)
        return self.assert_not_found(deferred, [1000])

    def test_shutdown_single_wrong_machine(self):
        self.mocker.reset()
        machine = ProviderMachine("i-000003E8")
        e = self.assertRaises(errors.ProviderError,
            self.get_provider().shutdown_machine, machine)
        self.assertIn("Need a NovaProviderMachine to shutdown", str(e))

    def test_shutdown_multi_none(self):
        self.mocker.reset()
        deferred = self.get_provider().shutdown_machines([])
        return deferred.addCallback(self.assertEqual, [])

    def test_shutdown_multi_some_invalid(self):
        """No machines are shutdown if some are invalid"""
        self.mocker.unorder()
        self.expect_nova_get("servers/1001",
            response={"server": {
                'name': "juju testing instance 1",
            }})
        self.expect_nova_get("servers/1002",
            response={"server": {
                'name': "some other instance",
            }})
        self.mocker.replay()
        machines = [NovaProviderMachine(1001), NovaProviderMachine(1002)]
        deferred = self.get_provider().shutdown_machines(machines)
        return self.assert_not_found(deferred, [1002])
    # XXX: dumb requirement to keep all running if some invalid, drop this
    test_shutdown_multi_some_invalid.skip = True

    # GZ 2012-06-11: Corner case difference, EC2 rechecks machine statuses and
    #                group membership on shutdown.

    def test_shutdown_multi_success(self):
        """Machines are shutdown and groups except juju group are deleted"""
        self.mocker.unorder()
        self.expect_nova_get("servers/1001",
            response={"server": {
                'name': "juju testing instance 1",
            }})
        self.expect_nova_get("servers/1002",
            response={"server": {
                'name': "juju testing instance 2",
            }})
        self.expect_nova_delete("servers/1001", code=204)
        self.expect_nova_delete("servers/1002", code=204)
        self.mocker.replay()
        machines = [NovaProviderMachine(1001), NovaProviderMachine(1002)]
        deferred = self.get_provider().shutdown_machines(machines)
        return deferred


class DestroyEnvironmentTests(OpenStackTestMixin, testing.TestCase):

    def check_machine_ids(self, machines, server_ids):
        self.assertEqual(set(m.instance_id for m in machines), set(server_ids))

    def test_destroy_environment(self):
        self.mocker.unorder()
        self.expect_swift_put("testing/provider-state", "{}\n")
        self.expect_nova_get("servers/detail", response={"servers": [
            self.make_server(1001),
            self.make_server(1002),
            ]})
        self.expect_nova_get("servers/1001",
            response={"server": {
                'name': "juju testing instance 1",
            }})
        self.expect_nova_get("servers/1002",
            response={"server": {
                'name': "juju testing instance 2",
            }})
        self.expect_nova_delete("servers/1001", code=204)
        self.expect_nova_delete("servers/1002", code=204)
        self.mocker.replay()
        deferred = self.get_provider().destroy_environment()
        return deferred.addCallback(self.check_machine_ids, [1001, 1002])

    def test_s3_failure(self):
        self.mocker.unorder()
        self.expect_swift_put("testing/provider-state", "{}\n",
            code=500, response="Server unavailable or something")
        # XXX: normal server shutdown should be expected here
        self.mocker.replay()
        deferred = self.get_provider().destroy_environment()
        return deferred.addCallback(self.assertEqual, [])
    # XXX: Need to bolster swift robustness in response to api errors
    test_s3_failure.skip = True

    # GZ 2012-06-15: Always tries removing juju group unlike EC2 currently
    def test_shutdown_no_instances(self):
        """With no instances no shutdowns are attempted"""
        self.mocker.unorder()
        self.expect_swift_put("testing/provider-state", "{}\n")
        self.expect_nova_get("servers/detail", response={"servers": []})
        self.mocker.replay()
        deferred = self.get_provider().destroy_environment()
        return deferred.addCallback(self.assertEqual, [])
