<?php
/*
  This code is part of FusionDirectory (http://www.fusiondirectory.org/)
  Copyright (C) 2003  Cajus Pollmeier
  Copyright (C) 2011-2015  FusionDirectory

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/

class password extends plugin
{
  var $proposal         = "";
  var $proposalEnabled  = FALSE;
  var $proposalSelected = FALSE;

  var $forcedHash = NULL;

  function password(&$config, $dn = NULL, $parent = NULL)
  {
    parent::__construct($config, $dn, $parent);

    // Try to generate a password proposal, if this is successfull
    //  then preselect the proposal usage.
    $this->refreshProposal();
    if ($this->proposal != "") {
      $this->proposalSelected = TRUE;
    }
  }

  function forceHash($hash)
  {
    $this->forcedHash = $hash;
  }

  function refreshProposal()
  {
    $this->proposal         = passwordMethod::getPasswordProposal($this->config);
    $this->proposalEnabled  = (!empty($this->proposal));
  }

  function execute()
  {
    plugin::execute();
    $smarty = get_smarty();
    $smarty->assign("usePrototype", "true");
    $ui = get_userinfo();

    /* Get acls */
    $password_ACLS = $ui->get_permissions($ui->dn, "user/password");
    $smarty->assign("ChangeACL", $password_ACLS);
    $smarty->assign("NotAllowed", !preg_match("/w/i", $password_ACLS));

    /* Display expiration template */
    $smarty->assign("passwordExpired", FALSE);
    if ($this->config->get_cfg_value("handleExpiredAccounts") == "TRUE") {
      $expired = $ui->expired_status();
      $smarty->assign("passwordExpired", ($expired == POSIX_FORCE_PASSWORD_CHANGE));
      if ($expired == POSIX_DISALLOW_PASSWORD_CHANGE) {
        return $smarty->fetch(get_template_path("nochange.tpl", TRUE));
      }
    }


    // Refresh proposal if requested
    if (isset($_POST['refreshProposal'])) {
      $this->refreshProposal();
    }
    if (isset($_POST['proposalSelected'])) {
      $this->proposalSelected = get_post('proposalSelected') == 1;
    }
    $smarty->assign("proposal", $this->proposal);
    $smarty->assign("proposalEnabled", $this->proposalEnabled);
    $smarty->assign("proposalSelected", $this->proposalSelected);

    /* Pwd change requested */
    if (isset($_POST['password_finish'])) {
      if ($this->proposalSelected) {
        $current_password   = $_POST['current_password'];
        $new_password       = $this->proposal;
        $repeated_password  = $this->proposal;
      } else {
        $current_password   = $_POST['current_password'];
        $new_password       = $_POST['new_password'];
        $repeated_password  = $_POST['repeated_password'];
      }

      // Perform FusionDirectory password policy checks
      $message = array();
      $problem = self::reportPasswordProblems($ui->dn, $new_password, $repeated_password, $current_password);
      if ($problem) {
        $message[] = $problem;
      } else {
        /* Call external password quality hook ?*/
        $check_hook   = $this->config->get_cfg_value("passwordHook") != "";
        $hook         = $this->config->get_cfg_value("passwordHook")." ".
            escapeshellarg($ui->username)." ".escapeshellarg($new_password)." ".escapeshellarg($current_password);
        if ($check_hook) {
          exec($hook, $resarr);
          $check_hook_output = "";
          if (count($resarr) > 0) {
            $check_hook_output = join('\n', $resarr);
          }
          if (!empty($check_hook_output)) {
            $message[] = sprintf(_("Check-hook reported a problem: %s. Password change canceled!"), $check_hook_output);
          }
        }
      }

      // Some errors/warning occured, display them and abort password change.
      if (count($message)) {
        msg_dialog::displayChecks($message);
      } else {
        /* Try to connect via current password */
        $tldap = new LDAP(
          $ui->dn,
          $current_password,
          $this->config->current['SERVER'],
          $this->config->get_cfg_value("ldapFollowReferrals") == "TRUE",
          $this->config->get_cfg_value("ldapTLS") == "TRUE"
        );

        /* connection Successfull ? */
        if (!$tldap->success()) {
          msg_dialog::display(_("Password change"),
                              _("The password you've entered as your current password doesn't match the real one."), WARNING_DIALOG);
        } else {
          /* Check FusionDirectory permissions */
          if (!preg_match("/w/i", $password_ACLS)) {
            msg_dialog::display(_("Password change"),
                                _("You have no permission to change your password."), WARNING_DIALOG);
          } else {
            if ($this->change_password($ui->dn, $new_password, $this->forcedHash)) {
              fusiondirectory_log("User/password has been changed");
              $ui->password = $new_password;
              session::set('ui', $ui);
              return $smarty->fetch(get_template_path("changed.tpl", TRUE));
            }
          }
        }
      }
    }
    return $smarty->fetch(get_template_path("password.tpl", TRUE));
  }

  function change_password($dn, $pwd, $hash)
  {
    if ($hash) {
      return change_password ($dn, $pwd, 0, $hash);
    } else {
      return change_password ($dn, $pwd);
    }
  }


  function remove_from_parent()
  {
    $this->handle_post_events("remove");
  }

  function save()
  {
  }

  static function plInfo()
  {
    return array(
      "plShortName"     => _("Password"),
      "plDescription"   => _("Change user password"),
      "plSelfModify"    => TRUE,
      "plPriority"      => 10,
      "plCategory"      => array("user"),
      "plSection"       => "personal",
      "plIcon"          => 'geticon.php?context=status&icon=dialog-password&size=48',

      "plProvidedAcls"  => array(
        "userPassword"  => _("Change user password"),
      )
    );
  }

  static function reportPasswordProblems ($user, $new_password, $repeated_password, $current_password = NULL)
  {
    global $config, $ui;

    /* Should we check different characters in new password */
    $check_differ = ($config->get_cfg_value("passwordMinDiffer") != "");
    $differ       = $config->get_cfg_value("passwordMinDiffer", 0);
    if ($current_password === NULL) {
      $check_differ = FALSE;
    }

    /* Enable length check ? */
    $check_length = ($config->get_cfg_value("passwordMinLength") != "");
    $length       = $config->get_cfg_value("passwordMinLength", 0);

    $ldap = $config->get_ldap_link();
    $ldap->cat($user, array('pwdPolicySubentry', 'pwdHistory', 'pwdChangedTime', 'userPassword'));
    $attrs = $ldap->fetch();

    $ppolicydn = '';
    if (isset($attrs['pwdPolicySubentry'][0])) {
      $ppolicydn = $attrs['pwdPolicySubentry'][0];
    } else {
      $ppolicydn = $config->get_cfg_value('ppolicyDefaultCn', '');
      if (!empty($ppolicydn)) {
        $ppolicydn = 'cn='.$ppolicydn.','.get_ou('ppolicyRDN').$config->current['BASE'];
      }
    }
    if (!empty($ppolicydn)) {
      $ldap->cat($ppolicydn, array('pwdAllowUserChange', 'pwdMinLength', 'pwdMinAge', 'pwdSafeModify'));
      $policy = $ldap->fetch();
      if (!$policy) {
        return sprintf(_('Ppolicy "%s" could not be found in the LDAP!'), $ppolicydn);
      }
      if (isset($policy['pwdAllowUserChange'][0]) && ($policy['pwdAllowUserChange'][0] == 'FALSE') && ($ui->dn == $user)) {
        return _('You are not allowed to change your own password');
      }
      if (isset($policy['pwdMinLength'][0])) {
        $check_length = TRUE;
        $length       = $policy['pwdMinLength'][0];
      }
      if (isset($policy['pwdMinAge'][0]) && isset($attrs['pwdChangedTime'][0])) {
        $date = DateTime::createFromFormat('YmdHis\Z', $attrs['pwdChangedTime'][0], timezone::utc());
        $now  = new DateTime('now', timezone::utc());
        if ($now->getTimeStamp() < $date->getTimeStamp() + $policy['pwdMinAge'][0]) {
          return sprintf(_('You must wait %d seconds before changing your password again'), $policy['pwdMinAge'][0] - ($now->getTimeStamp() - $date->getTimeStamp()));
        }
      }
      if (isset($policy['pwdSafeModify'][0]) && ($policy['pwdSafeModify'][0] == 'FALSE')) {
        if (empty($current_password)) {
          $current_password = NULL;
        }
      }
      if (isset($attrs['pwdHistory'][0])) {
        unset($attrs['pwdHistory']['count']);
        foreach ($attrs['pwdHistory'] as $pwdHistory) {
          $pwdHistory = explode('#', $pwdHistory, 4);
          $method = passwordMethod::get_method($pwdHistory[3], $user);
          if (($method !== NULL) && $method->checkPassword($new_password, $pwdHistory[3])) {
            return _('Password is in history of old passwords');
          }
        }
      }
      if (($current_password !== NULL) && ($current_password == $new_password)) {
        return _('Password is not being changed from existing value');
      } elseif (isset($attrs['userPassword'][0])) {
        $method = passwordMethod::get_method($attrs['userPassword'][0], $user);
        if (($method !== NULL) && $method->checkPassword($new_password, $attrs['userPassword'][0])) {
          return _('Password is not being changed from existing value');
        }
      }
    }

    // Perform FusionDirectory password policy checks
    if (($current_password !== NULL) && empty($current_password)) {
      return _("You need to specify your current password in order to proceed.");
    } elseif ($new_password != $repeated_password) {
      return _("The passwords you've entered as 'New password' and 'Repeated new password' do not match.");
    } elseif ($new_password == "") {
      return msgPool::required(_("New password"));
    } elseif ($check_differ && (substr($current_password, 0, $differ) == substr($new_password, 0, $differ))) {
      return _("The password used as new and current are too similar.");
    } elseif ($check_length && (strlen($new_password) < $length)) {
      return _("The password used as new is to short.");
    } elseif (!passwordMethod::is_harmless($new_password)) {
      return _("The password contains possibly problematic Unicode characters!");
    }

    return FALSE;
  }
}
?>
