/**
   @author Shin'ichiro Nakaoka
*/

#include "TimeBar.h"
#include "ExtensionManager.h"
#include "Archive.h"
#include "MessageView.h"
#include "MainWindow.h"
#include "OptionManager.h"
#include "LazyCaller.h"
#include "SpinBox.h"
#include "Slider.h"
#include "Button.h"
#include <QDialog>
#include <QTime>
#include <QCheckBox>
#include <QLayout>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
#include <limits>
#include <iostream>

#include "gettext.h"

using namespace std;
using namespace boost;
using namespace cnoid;

namespace {

    const bool TRACE_FUNCTIONS = false;
    
    inline double myNearByInt(double x)
    {
#ifdef Q_OS_WIN32
        double u = ceil(x);
        double l = floor(x);
        if(fabs(u - x) < fabs(x - l)){
            return u;
        } else {
            return l;
        }
#else
        return nearbyint(x);
#endif
    }

    class SetupDialog : public QDialog
    {
    public:
        SpinBox* frameRateSpin;
        SpinBox* playbackFrameRateSpin;
        DoubleSpinBox* playbackSpeedScaleSpin;
        QCheckBox* beatModeCheck;
        DoubleSpinBox* tempoSpin;
        SpinBox* beatcSpin;
        SpinBox* beatmSpin;
        DoubleSpinBox* beatOffsetSpin;

        SetupDialog()
            : QDialog(MainWindow::instance()) {

            setWindowTitle(_("Time Bar Setup"));

            QVBoxLayout* vbox = new QVBoxLayout();
            setLayout(vbox);

            QHBoxLayout* hbox = new QHBoxLayout();
            hbox->addWidget(new QLabel(_("Internal frame rate")));
            frameRateSpin = new SpinBox();
            frameRateSpin->setAlignment(Qt::AlignCenter);
            frameRateSpin->setRange(1, 9999);
            hbox->addWidget(frameRateSpin);
            hbox->addStretch();
            vbox->addLayout(hbox);

            hbox = new QHBoxLayout();
            hbox->addWidget(new QLabel(_("Playback frame rate")));
            playbackFrameRateSpin = new SpinBox();
            playbackFrameRateSpin->setAlignment(Qt::AlignCenter);
            playbackFrameRateSpin->setRange(0, 999);
            playbackFrameRateSpin->setValue(50);
            hbox->addWidget(playbackFrameRateSpin);
            hbox->addStretch();
            vbox->addLayout(hbox);

            hbox = new QHBoxLayout();
            hbox->addWidget(new QLabel(_("Playback speed scale")));
            playbackSpeedScaleSpin = new DoubleSpinBox();
            playbackSpeedScaleSpin->setAlignment(Qt::AlignCenter);
            playbackSpeedScaleSpin->setDecimals(1);
            playbackSpeedScaleSpin->setRange(0.1, 9.9);
            playbackSpeedScaleSpin->setSingleStep(0.1);
            playbackSpeedScaleSpin->setValue(1.0);
            hbox->addWidget(playbackSpeedScaleSpin);
            hbox->addStretch();
            vbox->addLayout(hbox);

            /*
            hbox = new QHBoxLayout();
            vbox->addLayout(hbox);
            
            beatModeCheck = new QCheckBox(_("Beat mode"));
            hbox->addWidget(beatModeCheck);

            beatcSpin = new SpinBox();
            beatcSpin->setRange(1, 99);
            hbox->addWidget(beatcSpin);

            hbox->addWidget(new QLabel("/"));

            beatmSpin = new SpinBox();
            beatmSpin->setRange(1, 99);
            hbox->addWidget(beatmSpin);

            hbox->addStretch();
            hbox = new QHBoxLayout();
            vbox->addLayout(hbox);

            hbox->addWidget(new QLabel(_("Tempo")));
            tempoSpin = new DoubleSpinBox();
            tempoSpin->setRange(1.0, 999.99);
            tempoSpin->setDecimals(2);
            hbox->addWidget(tempoSpin);

            hbox->addWidget(new QLabel(_("Offset")));
            beatOffsetSpin = new DoubleSpinBox();
            beatOffsetSpin->setRange(-9.99, 9.99);
            beatOffsetSpin->setDecimals(2);
            beatOffsetSpin->setSingleStep(0.1);
            hbox->addWidget(beatOffsetSpin);
            hbox->addWidget(new QLabel(_("[s]")));

            hbox->addStretch();
            */

            PushButton* okButton = new PushButton(_("&OK"));
            okButton->setDefault(true);

            vbox->addStretch();
        }
    };

}

namespace cnoid {

    class TimeBarImpl : public QObject
    {
    public:
        TimeBarImpl(TimeBar* self);
        ~TimeBarImpl();

        bool setTime(double time, bool calledFromPlaybackLoop, QWidget* callerWidget = 0);
        void onTimeSpinChanged(double value);
        bool onTimeSliderChangeValue(double value);

        void setTimeRange(double minTime, double maxTime);
        void setFrameRate(double rate);
        void updateTimeProperties(bool forceUpdate);
        void onPlaybackSpeedScaleChanged();
        void onPlaybackFrameRateChanged();
        void onPlayActivated();
        void onResumeActivated();
        void startPlayback();
        void stopPlayback(bool isStoppedManually);
        int startFillLevelUpdate();
        void updateFillLevel(int id, double time);
        void updateMinFillLevel();
        void stopFillLevelUpdate(int id);

        void onTimeRangeSpinsChanged();
        void onFrameRateSpinChanged();

        virtual void timerEvent(QTimerEvent* event);
        
        void onRefreshButtonClicked();

        bool storeState(Archive& archive);
        bool restoreState(const Archive& archive);

        TimeBar* self;
        ostream& os;
        SetupDialog setup;

        ToolButton* stopResumeButton;
        ToolButton* frameModeToggle;
        QIcon resumeIcon;
        QIcon stopIcon;
        
        DoubleSpinBox* timeSpin;
        Slider* timeSlider;
        DoubleSpinBox* minTimeSpin;
        DoubleSpinBox* maxTimeSpin;
        int decimals;
        double minTime;
        double maxTime;
        double playbackSpeedScale;
        double playbackFrameRate;
        double animationTimeOffset;
        int timerId;
        QTime timer;
        bool repeatMode;
        bool isDoingPlayback;
        map<int, double> fillLevelMap;
        double fillLevel;
        bool isFillLevelActive;

        boost::signal<bool(double time), TimeBar::LogicalProduct> sigPlaybackInitialized;
        boost::signal<void(double time)> sigPlaybackStarted;
        boost::signal<bool(double time), TimeBar::LogicalSum> sigTimeChanged;
        boost::signal<void(double time, bool isStoppedManually)> sigPlaybackStopped;
    };
}


static void onSigOptionsParsed(boost::program_options::variables_map& v)
{
    if(v.count("start-playback")){
        callLater(bind(&TimeBar::startPlayback, TimeBar::instance()));
    }
}


void TimeBar::initialize(ExtensionManager* ext)
{
    static bool initialized = false;
    if(!initialized){
        ext->addToolBar(TimeBar::instance());

        ext->optionManager().addOption("start-playback", "start playback automatically");
        ext->optionManager().sigOptionsParsed().connect(onSigOptionsParsed);
            
        initialized = true;
    }
}


TimeBar* TimeBar::instance()
{
    static TimeBar* timeBar = new TimeBar();
    return timeBar;
}


TimeBar::TimeBar()
    : ToolBar(N_("TimeBar"))
{
    impl = new TimeBarImpl(this);
}


TimeBarImpl::TimeBarImpl(TimeBar* self)
    : self(self),
      os(MessageView::mainInstance()->cout()),
      resumeIcon(QIcon(":/Base/icons/resume.png")),
      stopIcon(QIcon(":/Base/icons/stop.png"))
{
    self->setStretchable(true);
    
    self->time_ = 0.0;
    self->frameRate_ = 100.0;
    decimals = 2;
    minTime = 0.0;
    maxTime = 30.0;
    repeatMode = false;
    timerId = 0;
    isDoingPlayback = false;
    fillLevel = 0;
    isFillLevelActive = false;

    self->addButton(QIcon(":/Base/icons/play.png"), _("Start animation"))
        ->sigClicked().connect(bind(&TimeBarImpl::onPlayActivated, this));

    stopResumeButton = self->addButton(resumeIcon, _("Resume animation"));
    stopResumeButton->setIcon(resumeIcon);
    stopResumeButton->sigClicked().connect(bind(&TimeBarImpl::onResumeActivated, this));

    self->addButton(QIcon(":/Base/icons/refresh.png"), _("Refresh state at the current time"))
        ->sigClicked().connect(bind(&TimeBarImpl::onRefreshButtonClicked, this));
    
    timeSpin = new DoubleSpinBox();
    timeSpin->setAlignment(Qt::AlignCenter);
    timeSpin->sigValueChanged().connect(bind(&TimeBarImpl::onTimeSpinChanged, this, _1));
    self->addWidget(timeSpin);

    timeSlider = new Slider(Qt::Horizontal);
    timeSlider->sigValueChanged().connect(bind(&TimeBarImpl::onTimeSliderChangeValue, this, _1));
    self->addWidget(timeSlider);

    minTimeSpin = new DoubleSpinBox();
    minTimeSpin->setAlignment(Qt::AlignCenter);
    minTimeSpin->setRange(-999.0, 999.0);
    minTimeSpin->sigValueChanged().connect(bind(&TimeBarImpl::onTimeRangeSpinsChanged, this));
    self->addWidget(minTimeSpin);

    self->addLabel(" : ");

    maxTimeSpin = new DoubleSpinBox();
    maxTimeSpin->setAlignment(Qt::AlignCenter);
    maxTimeSpin->setRange(-999.0, 999.0);
    maxTimeSpin->sigValueChanged().connect(bind(&TimeBarImpl::onTimeRangeSpinsChanged, this));
    self->addWidget(maxTimeSpin);

    self->addButton(QIcon(":/Base/icons/setup.png"), _("Open the setup dialog"))
        ->sigClicked().connect(bind(&QDialog::show, &setup));

    setup.frameRateSpin->sigValueChanged().connect(bind(&TimeBarImpl::onFrameRateSpinChanged, this));
    setup.playbackFrameRateSpin->sigValueChanged().connect(bind(&TimeBarImpl::onPlaybackFrameRateChanged, this));
    setup.playbackSpeedScaleSpin->sigValueChanged().connect(bind(&TimeBarImpl::onPlaybackSpeedScaleChanged, this));

    playbackSpeedScale = setup.playbackSpeedScaleSpin->value();
    playbackFrameRate = setup.playbackFrameRateSpin->value();

    updateTimeProperties(true);
}


TimeBar::~TimeBar()
{
    delete impl;
}


TimeBarImpl::~TimeBarImpl()
{

}


SignalProxy< boost::signal<bool(double time), TimeBar::LogicalProduct> > TimeBar::sigPlaybackInitialized()
{
    return impl->sigPlaybackInitialized;
}


SignalProxy< boost::signal<void(double time)> > TimeBar::sigPlaybackStarted()
{
    return impl->sigPlaybackStarted;
}


/**
   Signal emitted when the TimeBar's time changes.
   
   In the function connected to this signal, please return true if the time is valid for it,
   and return false if the time is not valid. The example of the latter case is that
   the time is over the length of the data processed in the function.
*/
SignalProxy< boost::signal<bool(double time), TimeBar::LogicalSum> > TimeBar::sigTimeChanged()
{
    return impl->sigTimeChanged;
}


SignalProxy< boost::signal<void(double time, bool isStoppedManually)> > TimeBar::sigPlaybackStopped()
{
    return impl->sigPlaybackStopped;
}


bool TimeBar::setTime(double time)
{
    return impl->setTime(time, false);
}


/**
   @todo check whether block() and unblock() of sigc::connection
   decrease the performance or not.
*/
bool TimeBarImpl::setTime(double time, bool calledFromPlaybackLoop, QWidget* callerWidget)
{
    if(TRACE_FUNCTIONS){
        cout << "TimeBarImpl::setTime(" << time << ", " << calledFromPlaybackLoop << ")" << endl;
    }
    
    if(!calledFromPlaybackLoop && isDoingPlayback){
        return false;
    }

    const double newTime = myNearByInt(time * self->frameRate_) / self->frameRate_;

    if(newTime == self->time_){
        if(calledFromPlaybackLoop){
            return true;
        }
        if(callerWidget){
            return false;
        }
    }
        
    self->time_ = newTime;

    if(callerWidget != timeSpin){
        timeSpin->blockSignals(true);
        timeSpin->setValue(self->time_);
        timeSpin->blockSignals(false);
    }
    if(callerWidget != timeSlider){
        timeSlider->blockSignals(true);
        timeSlider->setValue((int)myNearByInt(self->time_ * pow(10.0, decimals)));
        timeSlider->blockSignals(false);
    }
    
    return sigTimeChanged(self->time_);
}


void TimeBarImpl::onTimeSpinChanged(double value)
{
    if(TRACE_FUNCTIONS){
        cout << "TimeBarImpl::onTimeSpinChanged()" << endl;
    }
    setTime(value, false, timeSpin);
}


bool TimeBarImpl::onTimeSliderChangeValue(double value)
{
    if(TRACE_FUNCTIONS){
        cout << "TimeBarImpl::onTimeSliderChanged(): value = " << value << endl;
    }
    setTime(value / pow(10.0, decimals), false, timeSlider);
    return true;
}


void TimeBar::setFrameRate(double rate)
{
    impl->setFrameRate(rate);
}


void TimeBarImpl::setFrameRate(double rate)
{
    if(rate > 0.0){
        self->frameRate_ = rate;
        updateTimeProperties(false);
    }
}


double TimeBar::minTime() const
{
    return impl->minTime;
}


double TimeBar::maxTime() const
{
    return impl->maxTime;
}


void TimeBar::setTimeRange(double min, double max)
{
    impl->setTimeRange(min, max);
}


void TimeBarImpl::setTimeRange(double minTime, double maxTime)
{
    this->minTime = minTime;
    this->maxTime = std::max(minTime, maxTime);
    updateTimeProperties(false);
}


void TimeBarImpl::updateTimeProperties(bool forceUpdate)
{
    timeSpin->blockSignals(true);
    timeSlider->blockSignals(true);
    minTimeSpin->blockSignals(true);
    maxTimeSpin->blockSignals(true);
    setup.frameRateSpin->blockSignals(true);
    
    double timeStep = 1.0 / self->frameRate_;
    decimals = static_cast<int>(ceil(log10(self->frameRate_)));
    double r = pow(10.0, decimals);

    if(forceUpdate || (minTime != timeSpin->minimum() || maxTime != timeSpin->maximum())){
        timeSpin->setRange(minTime, maxTime);
        timeSlider->setRange((int)myNearByInt(minTime * r), (int)myNearByInt(maxTime * r));
    }

    timeSpin->setDecimals(decimals);
    timeSpin->setSingleStep(timeStep);
    timeSlider->setSingleStep(timeStep * r);
    minTimeSpin->setValue(minTime);
    maxTimeSpin->setValue(maxTime);
    setup.frameRateSpin->setValue(self->frameRate_);

    setup.frameRateSpin->blockSignals(false);
    maxTimeSpin->blockSignals(false);
    minTimeSpin->blockSignals(false);
    timeSlider->blockSignals(false);
    timeSpin->blockSignals(false);

    setTime(self->time_, false);
}

    
void TimeBarImpl::onPlaybackSpeedScaleChanged()
{
    playbackSpeedScale = setup.playbackSpeedScaleSpin->value();
    
    if(isDoingPlayback){
        startPlayback();
    }
}


double TimeBar::playbackSpeedScale() const
{
    return impl->setup.playbackSpeedScaleSpin->value();
}


void TimeBarImpl::onPlaybackFrameRateChanged()
{
    playbackFrameRate = setup.playbackFrameRateSpin->value();
    
    if(isDoingPlayback){
        startPlayback();
    }
}


double TimeBar::playbackFrameRate() const
{
    return impl->setup.playbackFrameRateSpin->value();
}


void TimeBar::setRepeatMode(bool on)
{
    impl->repeatMode = on;
}


void TimeBarImpl::onPlayActivated()
{
    stopPlayback(true);
    setTime(minTime, false);
    startPlayback();
}


void TimeBarImpl::onResumeActivated()
{
    if(isDoingPlayback){
        stopPlayback(true);
    } else {
        stopPlayback(true);
        startPlayback();
    }
}


void TimeBar::startPlayback()
{
    impl->startPlayback();
}


void TimeBarImpl::startPlayback()
{
    stopPlayback(false);
    
    animationTimeOffset = self->time_;

    if(sigPlaybackInitialized(self->time_)){

        sigPlaybackStarted(self->time_);

        if(!setTime(self->time_, true)){
            sigPlaybackStopped(self->time_, false);

        } else {
            isDoingPlayback = true;

            const static QString tip(_("Stop animation"));
            stopResumeButton->setIcon(stopIcon);
            stopResumeButton->setToolTip(tip);

            timerId = startTimer((int)myNearByInt(1000.0 / playbackFrameRate));
            timer.start();
        }
    }
}


void TimeBar::stopPlayback(bool isStoppedManually)
{
    impl->stopPlayback(isStoppedManually);
}


void TimeBarImpl::stopPlayback(bool isStoppedManually)
{
    if(isDoingPlayback){
        killTimer(timerId);
        isDoingPlayback = false;
        sigPlaybackStopped(self->time_, isStoppedManually);

        const static QString tip(_("Resume animation"));
        stopResumeButton->setIcon(resumeIcon);
        stopResumeButton->setToolTip(tip);

        if(fillLevelMap.empty()){
            isFillLevelActive = false;
        }
    }
}


bool TimeBar::isDoingPlayback()
{
    return impl->isDoingPlayback;
}


int TimeBar::startFillLevelUpdate()
{
    return impl->startFillLevelUpdate();    
}


int TimeBarImpl::startFillLevelUpdate()
{
    int id=0;
    if(fillLevelMap.empty()){
        isFillLevelActive = true;
    } else {
        while(fillLevelMap.find(id) == fillLevelMap.end()){
            ++id;
        }
    }
    updateFillLevel(id, 0.0);
    return id;
}



void TimeBar::updateFillLevel(int id, double time)
{
    impl->updateFillLevel(id, time);
}


void TimeBarImpl::updateFillLevel(int id, double time)
{
    fillLevelMap[id] = time;
    updateMinFillLevel();
}


void TimeBarImpl::updateMinFillLevel()
{
    double minFillLevel = std::numeric_limits<double>::max();
    map<int,double>::iterator p;
    for(p = fillLevelMap.begin(); p != fillLevelMap.end(); ++p){
        minFillLevel = std::min(p->second, minFillLevel);
    }
    fillLevel = minFillLevel;
}    


void TimeBar::stopFillLevelUpdate(int id)
{
    impl->stopFillLevelUpdate(id);
}


void TimeBarImpl::stopFillLevelUpdate(int id)
{
    fillLevelMap.erase(id);

    if(!fillLevelMap.empty()){
        updateMinFillLevel();
    } else {
        if(!isDoingPlayback){
            isFillLevelActive = false;
        }
    }
}


void TimeBarImpl::timerEvent(QTimerEvent* event)
{
    double time = animationTimeOffset + playbackSpeedScale * (timer.elapsed() / 1000.0);

    bool doStopAtLastFillLevel = false;
    if(isFillLevelActive){
        if(time > fillLevel){
            animationTimeOffset -= (time - fillLevel);
            time = fillLevel;
            if(fillLevelMap.empty()){
                doStopAtLastFillLevel = true;
            }
        }
    }

    if(!setTime(time, true) || doStopAtLastFillLevel){
        stopPlayback(false);
        
        if(!doStopAtLastFillLevel && repeatMode){
            setTime(minTime, true);
            startPlayback();
        }
    }
}


void TimeBarImpl::onTimeRangeSpinsChanged()
{
    setTimeRange(minTimeSpin->value(), maxTimeSpin->value());
}


void TimeBarImpl::onFrameRateSpinChanged()
{
    setFrameRate(setup.frameRateSpin->value());
}


void TimeBarImpl::onRefreshButtonClicked()
{
    if(!isDoingPlayback){
        sigTimeChanged(self->time_);
    }
}


QSize TimeBar::minimumSizeHint () const
{
    QSize s = ToolBar::minimumSizeHint();
    s.setWidth(s.width() * 2);
    return s;
}


QSize TimeBar::sizeHint () const
{
    QSize s = ToolBar::minimumSizeHint();
    s.setWidth(s.width() * 3);
    return s;
}


bool TimeBar::storeState(Archive& archive)
{
    return impl->storeState(archive);
}


bool TimeBarImpl::storeState(Archive& archive)
{
    archive.write("minTime", minTime);
    archive.write("maxTime", maxTime);
    archive.write("frameRate", self->frameRate_);
    archive.write("currentTime", self->time_);
    return true;
}


bool TimeBar::restoreState(const Archive& archive)
{
    return impl->restoreState(archive);
}


bool TimeBarImpl::restoreState(const Archive& archive)
{
    archive.read("minTime", minTime);
    archive.read("maxTime", maxTime);
    archive.read("frameRate", self->frameRate_);
    archive.read("currentTime", self->time_);

    updateTimeProperties(false);
    
    return true;
}
