#include "../../burp.h"
#include "../../action.h"
#include "../../alloc.h"
#include "../../asfd.h"
#include "../../async.h"
#include "../../attribs.h"
#include "../../bfile.h"
#include "../../cmd.h"
#include "../../cntr.h"
#include "../../fsops.h"
#include "../../handy.h"
#include "../../log.h"
#include "../../protocol1/msg.h"
#include "../extrameta.h"
#include "../restore.h"
#include "restore.h"

static int do_restore_file_or_get_meta(struct asfd *asfd, struct BFILE *bfd,
	struct sbuf *sb, const char *fname,
	char **metadata, size_t *metalen,
	struct cntr *cntr, const char *rpath,
	const char *encryption_password)
{
	int ret=-1;
	int enccompressed=0;
	uint64_t rcvdbytes=0;
	uint64_t sentbytes=0;
	const char *encpassword=NULL;

	if(sbuf_is_encrypted(sb))
		encpassword=encryption_password;
	enccompressed=dpth_protocol1_is_compressed(sb->compression,
		sb->protocol1->datapth.buf);
/*
	printf("%s \n", fname);
	if(encpassword && !enccompressed)
		printf("encrypted and not compressed\n");
	else if(!encpassword && enccompressed)
		printf("not encrypted and compressed\n");
	else if(!encpassword && !enccompressed)
		printf("not encrypted and not compressed\n");
	else if(encpassword && enccompressed)
		printf("encrypted and compressed\n");
*/

	if(metadata)
	{
		ret=transfer_gzfile_inl(asfd,
#ifdef HAVE_WIN32
			sb,
#endif
			NULL,
			&rcvdbytes, &sentbytes, encpassword, enccompressed,
			cntr, metadata);
		*metalen=sentbytes;
		// skip setting cntr, as we do not actually
		// restore until a bit later
		goto end;
	}
	else
	{
		ret=transfer_gzfile_inl(asfd,
#ifdef HAVE_WIN32
			sb,
#endif
			bfd,
			&rcvdbytes, &sentbytes,
			encpassword, enccompressed, cntr, NULL);
#ifndef HAVE_WIN32
		if(bfd && bfd->close(bfd, asfd))
		{
			logp("error closing %s in %s\n",
				fname, __FUNCTION__);
			goto end;
		}
#endif
		if(!ret) attribs_set(asfd, rpath,
			&sb->statp, sb->winattr, cntr);
	}

	ret=0;
end:
	if(ret)
	{
		char msg[256]="";
		snprintf(msg, sizeof(msg),
			"Could not transfer file in: %s", rpath);
		if(restore_interrupt(asfd, sb, msg, cntr, PROTO_1))
			ret=-1;
	}
	return ret;
}

static int restore_file_or_get_meta(struct asfd *asfd, struct BFILE *bfd,
	struct sbuf *sb, const char *fname, enum action act,
	char **metadata, size_t *metalen, int vss_restore,
	struct cntr *cntr, const char *encyption_password)
{
	int ret=0;
	char *rpath=NULL;

	if(act==ACTION_VERIFY)
	{
		cntr_add(cntr, sb->path.cmd, 1);
		goto end;
	}

	if(build_path(fname, "", &rpath, NULL))
	{
		char msg[256]="";
		// failed - do a warning
		snprintf(msg, sizeof(msg), "build path failed: %s", fname);
		if(restore_interrupt(asfd, sb, msg, cntr, PROTO_1))
			ret=-1;
		goto end;
	}

#ifndef HAVE_WIN32
	// We always want to open the file if it is on Windows. Otherwise,
	// only open it if we are not doing metadata.
	if(!metadata)
	{
#endif
		switch(open_for_restore(asfd,
			bfd, rpath, sb, vss_restore, cntr, PROTO_1))
		{
			case OFR_OK: break;
			case OFR_CONTINUE: goto end;
			default: ret=-1; goto end;
		}
#ifndef HAVE_WIN32
	}
#endif

	if(!(ret=do_restore_file_or_get_meta(asfd, bfd, sb, fname,
		metadata, metalen, cntr, rpath, encyption_password)))
			cntr_add(cntr, sb->path.cmd, 1);
end:
	free_w(&rpath);
	if(ret) logp("restore_file error\n");
	return ret;
}

static int restore_metadata(struct asfd *asfd,
	struct BFILE *bfd, struct sbuf *sb,
	const char *fname, enum action act,
	int vss_restore, struct cntr *cntr, const char *encryption_password)
{
	int ret=-1;
	size_t metalen=0;
	char *metadata=NULL;
	
	// If it is directory metadata, try to make sure the directory
	// exists. Pass in NULL as the cntr, so no counting is done.
	// The actual directory entry will be coming after the metadata,
	// annoyingly. This is because of the way that the server is queuing
	// up directories to send after file data, so that the stat info on
	// them gets set correctly.
	if(act==ACTION_VERIFY)
	{
		cntr_add(cntr, sb->path.cmd, 1);
		ret=0;
		goto end;
	}

	if(S_ISDIR(sb->statp.st_mode)
	  && restore_dir(asfd, sb, fname, act, cntr, PROTO_1))
		goto end;

	// Read in the metadata...
	if(restore_file_or_get_meta(asfd, bfd, sb, fname, act,
		&metadata, &metalen, vss_restore, cntr, encryption_password))
			goto end;
	if(metadata)
	{
		
		if(!set_extrameta(asfd,
#ifdef HAVE_WIN32
			bfd,
#endif
			fname,
			metadata, metalen, cntr))
		{
#ifndef HAVE_WIN32
			// Set file times again, since we just diddled with the
			// file. Do not set all attributes, as it will wipe
			// out any security attributes (eg getcap /usr/bin/ping)
			if(attribs_set_file_times(asfd, fname,
				&sb->statp, cntr))
					return -1;
			cntr_add(cntr, sb->path.cmd, 1);
#endif
		}
		// Carry on if we could not set_extrameta.
	}
	ret=0;
end:
	free_w(&metadata);
	return ret;
}

int restore_switch_protocol1(struct asfd *asfd, struct sbuf *sb,
	const char *fullpath, enum action act,
	struct BFILE *bfd, int vss_restore, struct cntr *cntr,
	const char *encryption_password)
{
	switch(sb->path.cmd)
	{
		case CMD_FILE:
		case CMD_VSS_T:
		case CMD_ENC_FILE:
		case CMD_ENC_VSS_T:
		case CMD_EFS_FILE:
			if(!sb->protocol1->datapth.buf)
			{
				char msg[256];
				snprintf(msg, sizeof(msg),
				  "datapth not supplied for %c:%s in %s\n",
					sb->path.cmd, sb->path.buf,
					__FUNCTION__);
				log_and_send(asfd, msg);
				return -1;
			}
			return restore_file_or_get_meta(asfd, bfd, sb,
				fullpath, act,
				NULL, NULL, vss_restore, cntr,
				encryption_password);
		case CMD_METADATA:
		case CMD_VSS:
		case CMD_ENC_METADATA:
		case CMD_ENC_VSS:
			return restore_metadata(asfd, bfd, sb,
				fullpath, act,
				vss_restore, cntr, encryption_password);
		default:
			// Other cases (dir/links/etc) are handled in the
			// calling function.
			logp("unknown cmd: %c\n", sb->path.cmd);
			return -1;
	}
}
