/** @file
  Sample Implementation for Microsoft Authenticode Verification.

Copyright (c) 2009 - 2011, Intel Corporation. All rights reserved.<BR>
SPDX-License-Identifier: BSD-2-Clause-Patent

**/

#include "TestBaseCryptLib.h"

//
// Sample PE/COFF Image Hash Value (Digested by SHA-1).
// This value should be calculated following MSFT's Authenticode Specification
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT8 PeSha1Hash[] = {
  0x44, 0xFD, 0x4F, 0xA9, 0x17, 0xEE, 0xAC, 0xCF, 0x1F, 0x0B, 0xE3, 0xA1, 0x4D, 0x5B, 0xA6, 0x61,
  0x82, 0x97, 0xC4, 0xB6
  };

//
// Sample PE/COFF Image Hash Value (Digested by SHA-256).
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT8 PeSha256Hash[] = {
  0x61, 0x82, 0xB7, 0xF8, 0x8C, 0xFF, 0xC2, 0xEB, 0x79, 0x6E, 0x9D, 0xA9, 0xDD, 0x39, 0x52, 0xDD,
  0x36, 0xDD, 0xF1, 0x43, 0x27, 0x58, 0x8C, 0xA7, 0xCC, 0xAE, 0xDE, 0xDD, 0x3C, 0x02, 0x12, 0x49
  };

//
// Sample Authenticode Data with SHA-1 hash algorithm.
// This data should be retrieved from signed PE/COFF image according to SECURITY
// directory in PE/COFF Header.
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT8 AuthenticodeWithSha1[] = {
  0x30, 0x82, 0x1C, 0x43, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x02, 0xA0,
  0x82, 0x1C, 0x34, 0x30, 0x82, 0x1C, 0x30, 0x02, 0x01, 0x01, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x05,
  0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x68, 0x06, 0x0A, 0x2B, 0x06, 0x01, 0x04, 0x01,
  0x82, 0x37, 0x02, 0x01, 0x04, 0xA0, 0x5A, 0x30, 0x58, 0x30, 0x33, 0x06, 0x0A, 0x2B, 0x06, 0x01,
  0x04, 0x01, 0x82, 0x37, 0x02, 0x01, 0x0F, 0x30, 0x25, 0x03, 0x01, 0x00, 0xA0, 0x20, 0xA2, 0x1E,
  0x80, 0x1C, 0x00, 0x3C, 0x00, 0x3C, 0x00, 0x3C, 0x00, 0x4F, 0x00, 0x62, 0x00, 0x73, 0x00, 0x6F,
  0x00, 0x6C, 0x00, 0x65, 0x00, 0x74, 0x00, 0x65, 0x00, 0x3E, 0x00, 0x3E, 0x00, 0x3E, 0x30, 0x21,
  0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x04, 0x14, 0x44, 0xFD, 0x4F,
  0xA9, 0x17, 0xEE, 0xAC, 0xCF, 0x1F, 0x0B, 0xE3, 0xA1, 0x4D, 0x5B, 0xA6, 0x61, 0x82, 0x97, 0xC4,
  0xB6, 0xA0, 0x82, 0x17, 0x18, 0x30, 0x82, 0x04, 0xCA, 0x30, 0x82, 0x03, 0xB2, 0xA0, 0x03, 0x02,
  0x01, 0x02, 0x02, 0x0A, 0x61, 0x03, 0xDC, 0xF6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x30, 0x0D,
  0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x77, 0x31,
  0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11,
  0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0A, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6E, 0x67, 0x74, 0x6F,
  0x6E, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65, 0x64, 0x6D,
  0x6F, 0x6E, 0x64, 0x31, 0x1E, 0x30, 0x1C, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x15, 0x4D, 0x69,
  0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74,
  0x69, 0x6F, 0x6E, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x18, 0x4D, 0x69,
  0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x54, 0x69, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61,
  0x6D, 0x70, 0x20, 0x50, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x38, 0x30, 0x37, 0x32, 0x35,
  0x31, 0x39, 0x31, 0x32, 0x35, 0x30, 0x5A, 0x17, 0x0D, 0x31, 0x31, 0x30, 0x37, 0x32, 0x35, 0x31,
  0x39, 0x32, 0x32, 0x35, 0x30, 0x5A, 0x30, 0x81, 0xB3, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55,
  0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13,
  0x0A, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6E, 0x67, 0x74, 0x6F, 0x6E, 0x31, 0x10, 0x30, 0x0E, 0x06,
  0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65, 0x64, 0x6D, 0x6F, 0x6E, 0x64, 0x31, 0x1E, 0x30,
  0x1C, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x15, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66,
  0x74, 0x20, 0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x31, 0x0D, 0x30,
  0x0B, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x04, 0x4D, 0x4F, 0x50, 0x52, 0x31, 0x27, 0x30, 0x25,
  0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x1E, 0x6E, 0x43, 0x69, 0x70, 0x68, 0x65, 0x72, 0x20, 0x44,
  0x53, 0x45, 0x20, 0x45, 0x53, 0x4E, 0x3A, 0x31, 0x35, 0x39, 0x43, 0x2D, 0x41, 0x33, 0x46, 0x37,
  0x2D, 0x32, 0x35, 0x37, 0x30, 0x31, 0x25, 0x30, 0x23, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1C,
  0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x54, 0x69, 0x6D, 0x65, 0x2D, 0x53,
  0x74, 0x61, 0x6D, 0x70, 0x20, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x30, 0x82, 0x01, 0x22,
  0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03,
  0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xC0, 0xED, 0x81,
  0x14, 0xA1, 0x5E, 0x77, 0xC0, 0x5B, 0xF4, 0x76, 0x89, 0x62, 0xFA, 0xAD, 0x7C, 0x68, 0x14, 0xB4,
  0xF7, 0xBD, 0x35, 0xD8, 0x13, 0x79, 0x5A, 0x17, 0xCA, 0xD9, 0x6C, 0x51, 0x45, 0x62, 0x26, 0x7A,
  0x2F, 0x1F, 0xD8, 0xEA, 0xC1, 0x6E, 0x01, 0x17, 0xF9, 0xC3, 0xA6, 0x1F, 0x67, 0xDB, 0x51, 0xB0,
  0x2C, 0xDE, 0x8A, 0x17, 0xED, 0xFF, 0x20, 0xAD, 0x34, 0xEA, 0x98, 0xFB, 0xA5, 0xD6, 0x2A, 0xD2,
  0xF1, 0x44, 0x27, 0x07, 0x5A, 0x2D, 0x3A, 0x93, 0xFF, 0x56, 0x53, 0xB0, 0xC8, 0xF5, 0xF3, 0x03,
  0xF2, 0x49, 0xCC, 0x16, 0xD0, 0xF5, 0x00, 0x4C, 0x58, 0xF8, 0x9B, 0xF5, 0x07, 0x25, 0xB1, 0x66,
  0x17, 0xC0, 0xBD, 0xC8, 0xD2, 0x52, 0x85, 0x8D, 0xC2, 0x2B, 0x38, 0xB2, 0xC3, 0x36, 0xBE, 0xF9,
  0x87, 0xDA, 0xF4, 0x8E, 0x5D, 0x43, 0xD7, 0x06, 0xBF, 0x99, 0x05, 0x9F, 0xA4, 0xCE, 0xFE, 0xAB,
  0x8D, 0x61, 0x63, 0xE7, 0x39, 0xC5, 0xF3, 0x18, 0xF6, 0xD8, 0xFC, 0x31, 0x36, 0x69, 0x72, 0x5A,
  0xA2, 0x1A, 0x4C, 0x3E, 0xEA, 0x87, 0x25, 0x42, 0x9D, 0xD1, 0x3E, 0xF1, 0x97, 0xD2, 0x18, 0x32,
  0x93, 0x70, 0x55, 0x53, 0x81, 0x1E, 0xE3, 0x3B, 0x0D, 0xE8, 0xBE, 0x82, 0x78, 0x6D, 0xE6, 0xFA,
  0xCD, 0x98, 0xA4, 0x6F, 0xDB, 0xEE, 0x66, 0xF4, 0x95, 0xC8, 0xCD, 0x35, 0xC9, 0x9E, 0xBB, 0x36,
  0x0D, 0x83, 0x96, 0x94, 0x26, 0xA7, 0x90, 0xE0, 0xA9, 0x34, 0x3B, 0xD5, 0xC0, 0x9E, 0x3E, 0xF0,
  0xD4, 0x47, 0x8D, 0x86, 0x0C, 0x82, 0xA4, 0x58, 0x30, 0x3A, 0x1C, 0x76, 0xE3, 0xAD, 0x95, 0x66,
  0xB4, 0xB7, 0xFD, 0x09, 0x8A, 0x05, 0x60, 0x0F, 0xA3, 0x0F, 0xE2, 0x93, 0x96, 0x58, 0x22, 0x9C,
  0x9D, 0x2B, 0xDB, 0xA2, 0x94, 0x18, 0x90, 0x95, 0x02, 0xBD, 0x06, 0x40, 0x95, 0x02, 0x03, 0x01,
  0x00, 0x01, 0xA3, 0x82, 0x01, 0x19, 0x30, 0x82, 0x01, 0x15, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D,
  0x0E, 0x04, 0x16, 0x04, 0x14, 0xD2, 0xED, 0x0D, 0x1E, 0x24, 0xBB, 0x37, 0xA9, 0xD8, 0x20, 0x6A,
  0x4D, 0x1D, 0xD2, 0x16, 0xD5, 0x2E, 0xBE, 0x9E, 0xEB, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23,
  0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x23, 0x34, 0xF8, 0xD9, 0x52, 0x46, 0x70, 0x0A, 0xED, 0x40,
  0xFB, 0x76, 0xFB, 0xB3, 0x2B, 0xB0, 0xC3, 0x35, 0xB3, 0x0F, 0x30, 0x54, 0x06, 0x03, 0x55, 0x1D,
  0x1F, 0x04, 0x4D, 0x30, 0x4B, 0x30, 0x49, 0xA0, 0x47, 0xA0, 0x45, 0x86, 0x43, 0x68, 0x74, 0x74,
  0x70, 0x3A, 0x2F, 0x2F, 0x63, 0x72, 0x6C, 0x2E, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66,
  0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x2F, 0x63, 0x72, 0x6C, 0x2F, 0x70, 0x72,
  0x6F, 0x64, 0x75, 0x63, 0x74, 0x73, 0x2F, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74,
  0x54, 0x69, 0x6D, 0x65, 0x53, 0x74, 0x61, 0x6D, 0x70, 0x50, 0x43, 0x41, 0x2E, 0x63, 0x72, 0x6C,
  0x30, 0x58, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x4C, 0x30, 0x4A,
  0x30, 0x48, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x3C, 0x68, 0x74,
  0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F,
  0x66, 0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x2F, 0x63, 0x65, 0x72, 0x74, 0x73,
  0x2F, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x54, 0x69, 0x6D, 0x65, 0x53, 0x74,
  0x61, 0x6D, 0x70, 0x50, 0x43, 0x41, 0x2E, 0x63, 0x72, 0x74, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1D,
  0x25, 0x04, 0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x08, 0x30,
  0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x06, 0xC0, 0x30,
  0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82,
  0x01, 0x01, 0x00, 0x9C, 0x0A, 0x55, 0xC8, 0xCC, 0x44, 0x13, 0x34, 0x0C, 0xD8, 0x63, 0x27, 0x76,
  0x7D, 0x3E, 0xFA, 0x38, 0x32, 0x83, 0x53, 0x9D, 0xF2, 0x08, 0xF9, 0x32, 0xF5, 0xC5, 0x6E, 0x70,
  0xA1, 0xC9, 0xB1, 0x63, 0x6B, 0x19, 0x9D, 0x09, 0x67, 0xD9, 0x9D, 0xEB, 0x8A, 0x6A, 0xDB, 0x60,
  0x66, 0xE9, 0xE9, 0x52, 0x26, 0xF3, 0x3B, 0xC6, 0x6A, 0xD3, 0xC2, 0x52, 0xBE, 0xA8, 0xB9, 0xEB,
  0x6A, 0xAA, 0x78, 0x8C, 0xC9, 0x16, 0x7D, 0x90, 0x95, 0xA0, 0xCC, 0x21, 0xB3, 0x9E, 0x81, 0xBD,
  0xCD, 0xC1, 0x8B, 0x29, 0xBD, 0x62, 0x25, 0xEF, 0x09, 0x57, 0xE7, 0x86, 0x4E, 0x2A, 0xEC, 0x80,
  0xCA, 0xBB, 0xFC, 0x21, 0x16, 0xC4, 0x3F, 0x4E, 0x52, 0x19, 0xE6, 0x0E, 0xB1, 0xD8, 0xC1, 0xC2,
  0x79, 0x90, 0x64, 0xB4, 0x50, 0x73, 0x10, 0x35, 0x5E, 0x5D, 0x11, 0xC1, 0xB8, 0xBA, 0xAA, 0xCF,
  0x52, 0xF6, 0x80, 0x91, 0x00, 0xE6, 0xEF, 0x51, 0x43, 0x46, 0xE9, 0xD0, 0xE8, 0x94, 0xF6, 0x2C,
  0x24, 0x0D, 0x8A, 0xC6, 0xB2, 0x31, 0x8A, 0xA3, 0x7E, 0x36, 0x6C, 0xA4, 0x05, 0x4C, 0x67, 0x07,
  0x2A, 0xBB, 0xBB, 0x10, 0xA5, 0xA5, 0x30, 0x1A, 0x72, 0xD0, 0x06, 0x20, 0x3B, 0x24, 0x93, 0x5B,
  0x15, 0xD9, 0x39, 0x93, 0xD3, 0x73, 0x2D, 0x1A, 0xC4, 0xD4, 0x6C, 0x1E, 0xA1, 0x08, 0xEC, 0xF6,
  0x31, 0xB8, 0x6B, 0x4B, 0xEC, 0xEE, 0x5C, 0x33, 0x02, 0x14, 0x32, 0x8C, 0x7C, 0x11, 0x20, 0x2F,
  0x20, 0x03, 0x7F, 0xF9, 0x0C, 0x9D, 0xB8, 0xD3, 0x9E, 0x5F, 0xD6, 0x08, 0xFC, 0x81, 0xA0, 0x99,
  0xB8, 0xBB, 0x55, 0x6E, 0xCD, 0x42, 0x4B, 0x3A, 0x4D, 0x8C, 0x14, 0x2B, 0xCA, 0xC8, 0x12, 0xD3,
  0x62, 0x6E, 0xEA, 0x0D, 0x0A, 0x9D, 0x09, 0xA3, 0x66, 0xD9, 0x79, 0x4F, 0x8E, 0x1A, 0xA2, 0xFF,
  0xCC, 0x98, 0x04, 0x30, 0x82, 0x05, 0x96, 0x30, 0x82, 0x04, 0x7E, 0xA0, 0x03, 0x02, 0x01, 0x02,
  0x02, 0x0A, 0x61, 0x01, 0xC6, 0xC1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x30, 0x0D, 0x06, 0x09,
  0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x81, 0x81, 0x31, 0x0B,
  0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06,
  0x03, 0x55, 0x04, 0x08, 0x13, 0x0A, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6E, 0x67, 0x74, 0x6F, 0x6E,
  0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65, 0x64, 0x6D, 0x6F,
  0x6E, 0x64, 0x31, 0x1E, 0x30, 0x1C, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x15, 0x4D, 0x69, 0x63,
  0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74, 0x69,
  0x6F, 0x6E, 0x31, 0x2B, 0x30, 0x29, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x22, 0x4D, 0x69, 0x63,
  0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x57, 0x69, 0x6E, 0x64, 0x6F, 0x77, 0x73, 0x20, 0x56,
  0x65, 0x72, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x20, 0x50, 0x43, 0x41, 0x30,
  0x1E, 0x17, 0x0D, 0x30, 0x38, 0x31, 0x30, 0x32, 0x32, 0x32, 0x30, 0x33, 0x39, 0x32, 0x32, 0x5A,
  0x17, 0x0D, 0x31, 0x30, 0x30, 0x31, 0x32, 0x32, 0x32, 0x30, 0x34, 0x39, 0x32, 0x32, 0x5A, 0x30,
  0x7F, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13,
  0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0A, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6E, 0x67,
  0x74, 0x6F, 0x6E, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65,
  0x64, 0x6D, 0x6F, 0x6E, 0x64, 0x31, 0x1E, 0x30, 0x1C, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x15,
  0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72,
  0x61, 0x74, 0x69, 0x6F, 0x6E, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x04,
  0x4D, 0x4F, 0x50, 0x52, 0x31, 0x1A, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x11, 0x4D,
  0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x57, 0x69, 0x6E, 0x64, 0x6F, 0x77, 0x73,
  0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01,
  0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01,
  0x00, 0xDC, 0x3A, 0xD3, 0x44, 0xF4, 0x6E, 0x20, 0x9F, 0xDD, 0xA4, 0x0E, 0x82, 0x4E, 0xC7, 0x86,
  0x5E, 0x63, 0xCC, 0xCA, 0xE5, 0x42, 0x53, 0x4B, 0x85, 0xFA, 0x5D, 0x71, 0x6C, 0xCF, 0x76, 0x0C,
  0x18, 0x8B, 0xA6, 0x0D, 0xF5, 0x4A, 0xF7, 0xFE, 0x17, 0xF2, 0x90, 0xCC, 0x62, 0xC7, 0x24, 0xAD,
  0x9B, 0x9A, 0xE1, 0x45, 0x3B, 0x61, 0xD8, 0x0D, 0x05, 0x69, 0xC7, 0xCD, 0x88, 0x2A, 0xB8, 0xB9,
  0x18, 0x1E, 0x60, 0x10, 0x5F, 0x88, 0xC6, 0xD2, 0x82, 0x4E, 0x6D, 0x49, 0xC5, 0xBE, 0x5C, 0x12,
  0x86, 0x48, 0x85, 0x89, 0x91, 0x81, 0xCD, 0x1B, 0xAD, 0x1F, 0xB7, 0x2D, 0x67, 0x79, 0xF1, 0x7B,
  0x9F, 0x25, 0x87, 0x14, 0x76, 0x5F, 0xE3, 0x0E, 0x64, 0xA1, 0x72, 0x61, 0x25, 0xE5, 0x75, 0x69,
  0xC5, 0x14, 0xF1, 0x5F, 0x07, 0x56, 0xA4, 0x0D, 0x70, 0x06, 0x23, 0xA7, 0x6C, 0xDD, 0x82, 0xAE,
  0xD9, 0x9B, 0x47, 0xA4, 0xA5, 0x6C, 0x08, 0xB0, 0x58, 0xF1, 0x53, 0x6A, 0x4F, 0xDA, 0x85, 0x61,
  0xCB, 0x02, 0x7B, 0x49, 0xAF, 0x1F, 0xBB, 0xE0, 0xD7, 0xB9, 0x5E, 0xDB, 0x73, 0x89, 0x76, 0xC1,
  0x3A, 0xBB, 0x0D, 0xF5, 0x97, 0xF0, 0x88, 0x5D, 0x69, 0x77, 0x80, 0xCF, 0xF1, 0x7E, 0x03, 0x9F,
  0x73, 0x6D, 0xDE, 0x05, 0xB8, 0x2F, 0x77, 0xB5, 0x54, 0x55, 0x45, 0xD0, 0xD2, 0x38, 0xBD, 0x96,
  0xE3, 0xF7, 0xEA, 0x40, 0xE5, 0xAC, 0x19, 0xFC, 0x71, 0xCB, 0x28, 0x27, 0xAA, 0x71, 0xA1, 0x72,
  0xB5, 0x12, 0x27, 0xC1, 0x51, 0xF6, 0x36, 0xC5, 0xC0, 0xC7, 0x7B, 0x3A, 0x3A, 0x93, 0x37, 0x04,
  0xCC, 0xEE, 0x0B, 0x69, 0x78, 0x64, 0x75, 0x41, 0xB6, 0x78, 0x22, 0x0F, 0x77, 0x84, 0xF7, 0x4B,
  0x8D, 0x46, 0x65, 0x92, 0x5B, 0x4D, 0x56, 0x6B, 0x75, 0x04, 0x46, 0x3F, 0x0B, 0x1B, 0xB4, 0x19,
  0xBF, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x82, 0x02, 0x0F, 0x30, 0x82, 0x02, 0x0B, 0x30, 0x1F,
  0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x18, 0x30, 0x16, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05,
  0x07, 0x03, 0x03, 0x06, 0x0A, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x0A, 0x03, 0x06, 0x30,
  0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0xA1, 0xE6, 0xC3, 0x65, 0xD0, 0xE6,
  0xE8, 0x28, 0x62, 0xC2, 0xF3, 0xC2, 0x23, 0xA6, 0x1C, 0x49, 0x82, 0x0B, 0xD5, 0x53, 0x30, 0x0E,
  0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1F,
  0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x90, 0x8B, 0x11,
  0xA5, 0x70, 0xED, 0xE0, 0xF9, 0xA9, 0xC0, 0xAC, 0x08, 0xC7, 0xB5, 0xF4, 0x82, 0xB1, 0x3C, 0xC5,
  0x4A, 0x30, 0x7B, 0x06, 0x03, 0x55, 0x1D, 0x1F, 0x04, 0x74, 0x30, 0x72, 0x30, 0x70, 0xA0, 0x6E,
  0xA0, 0x6C, 0x86, 0x34, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x63, 0x72, 0x6C, 0x2E, 0x6D,
  0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69,
  0x2F, 0x63, 0x72, 0x6C, 0x2F, 0x70, 0x72, 0x6F, 0x64, 0x75, 0x63, 0x74, 0x73, 0x2F, 0x57, 0x69,
  0x6E, 0x50, 0x43, 0x41, 0x2E, 0x63, 0x72, 0x6C, 0x86, 0x34, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F,
  0x2F, 0x77, 0x77, 0x77, 0x2E, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x2E, 0x63,
  0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x2F, 0x63, 0x72, 0x6C, 0x2F, 0x70, 0x72, 0x6F, 0x64, 0x75,
  0x63, 0x74, 0x73, 0x2F, 0x57, 0x69, 0x6E, 0x50, 0x43, 0x41, 0x2E, 0x63, 0x72, 0x6C, 0x30, 0x52,
  0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x46, 0x30, 0x44, 0x30, 0x42,
  0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x36, 0x68, 0x74, 0x74, 0x70,
  0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74,
  0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x2F, 0x63, 0x65, 0x72, 0x74, 0x73, 0x2F, 0x4D,
  0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x57, 0x69, 0x6E, 0x50, 0x43, 0x41, 0x2E, 0x63,
  0x72, 0x74, 0x30, 0x81, 0xC6, 0x06, 0x03, 0x55, 0x1D, 0x20, 0x04, 0x81, 0xBE, 0x30, 0x81, 0xBB,
  0x30, 0x81, 0xB8, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x2F, 0x30, 0x81,
  0xAA, 0x30, 0x40, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x34, 0x68,
  0x74, 0x74, 0x70, 0x73, 0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x6D, 0x69, 0x63, 0x72, 0x6F,
  0x73, 0x6F, 0x66, 0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x2F, 0x73, 0x73, 0x6C,
  0x2F, 0x63, 0x70, 0x73, 0x2F, 0x57, 0x69, 0x6E, 0x64, 0x6F, 0x77, 0x73, 0x50, 0x43, 0x41, 0x2E,
  0x68, 0x74, 0x6D, 0x30, 0x66, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x02, 0x30,
  0x5A, 0x1E, 0x58, 0x00, 0x43, 0x00, 0x6F, 0x00, 0x70, 0x00, 0x79, 0x00, 0x72, 0x00, 0x69, 0x00,
  0x67, 0x00, 0x68, 0x00, 0x74, 0x00, 0x20, 0x00, 0xA9, 0x00, 0x20, 0x00, 0x31, 0x00, 0x39, 0x00,
  0x39, 0x00, 0x39, 0x00, 0x2D, 0x00, 0x32, 0x00, 0x30, 0x00, 0x30, 0x00, 0x35, 0x00, 0x20, 0x00,
  0x4D, 0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x6F, 0x00, 0x66, 0x00,
  0x74, 0x00, 0x20, 0x00, 0x43, 0x00, 0x6F, 0x00, 0x72, 0x00, 0x70, 0x00, 0x6F, 0x00, 0x72, 0x00,
  0x61, 0x00, 0x74, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x2E, 0x30, 0x0D, 0x06, 0x09, 0x2A,
  0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x73,
  0x5F, 0xCA, 0x80, 0x1C, 0x60, 0x46, 0x6F, 0xB9, 0x34, 0x9D, 0x88, 0xE3, 0xBE, 0x22, 0x8C, 0xFA,
  0xE6, 0x58, 0x9A, 0xAB, 0x7B, 0x1A, 0x97, 0xFD, 0xED, 0x2E, 0x39, 0xCC, 0x59, 0x5B, 0x1D, 0x7A,
  0x06, 0x8A, 0xBB, 0x43, 0x93, 0x7B, 0x1E, 0xA1, 0x88, 0x53, 0xDF, 0x44, 0xF8, 0x53, 0xA9, 0xEA,
  0xF6, 0x67, 0x1B, 0x3A, 0x78, 0x84, 0x11, 0x6A, 0x6F, 0x29, 0x47, 0x90, 0x0A, 0x0C, 0x7B, 0x22,
  0x77, 0x4E, 0x6F, 0xB8, 0x64, 0x29, 0xDF, 0x06, 0xC7, 0xC8, 0x73, 0x84, 0xD6, 0x66, 0xA0, 0xCA,
  0xD9, 0x5A, 0x26, 0x82, 0x57, 0xF9, 0xE3, 0x4F, 0x39, 0xAF, 0x2E, 0x8E, 0xB1, 0x06, 0x5B, 0x72,
  0xF2, 0x37, 0x32, 0xAE, 0x4E, 0xCE, 0x3C, 0x7D, 0xB0, 0x12, 0x2B, 0x9E, 0xA5, 0x75, 0xE3, 0x43,
  0xA6, 0x12, 0x8B, 0x06, 0x14, 0x98, 0x77, 0xE3, 0x58, 0x32, 0x25, 0x60, 0x07, 0x8C, 0x59, 0x71,
  0xA7, 0x71, 0x41, 0xB3, 0x06, 0x8D, 0x5C, 0xEF, 0x9C, 0x7F, 0x5A, 0x22, 0x6D, 0xB7, 0xD3, 0xD9,
  0xF5, 0xA6, 0x1B, 0x52, 0xDE, 0xF5, 0x7E, 0x76, 0x7C, 0xFE, 0xF4, 0xC8, 0x23, 0x1A, 0x4B, 0x25,
  0xEB, 0xE4, 0xEE, 0xAF, 0x10, 0x0B, 0x55, 0xC3, 0xD8, 0xC1, 0x17, 0x85, 0x61, 0x6F, 0xD3, 0x3F,
  0xB6, 0xE9, 0xEC, 0x84, 0xA5, 0xEE, 0x6D, 0xB2, 0xFF, 0xE8, 0x6C, 0x95, 0xAB, 0x2B, 0x5E, 0xC8,
  0x85, 0xC3, 0x11, 0x60, 0xAC, 0xFA, 0x02, 0x05, 0xF1, 0x7B, 0xDA, 0xC3, 0x69, 0x49, 0x96, 0xA5,
  0x70, 0xF9, 0x65, 0x66, 0x46, 0x10, 0x8D, 0x34, 0xE9, 0x21, 0x94, 0x3C, 0x0F, 0x71, 0x4A, 0x1C,
  0xEA, 0x1F, 0xF7, 0x23, 0xA6, 0x87, 0x60, 0x34, 0xE9, 0x14, 0xE1, 0xDE, 0x03, 0x59, 0xB4, 0x02,
  0x1D, 0x3A, 0xAF, 0xE3, 0x55, 0x05, 0xF5, 0xED, 0xC1, 0xF4, 0xE4, 0x5D, 0x0E, 0xD3, 0x97, 0x30,
  0x82, 0x06, 0x07, 0x30, 0x82, 0x03, 0xEF, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x0A, 0x61, 0x16,
  0x68, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
  0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x5F, 0x31, 0x13, 0x30, 0x11, 0x06, 0x0A, 0x09,
  0x92, 0x26, 0x89, 0x93, 0xF2, 0x2C, 0x64, 0x01, 0x19, 0x16, 0x03, 0x63, 0x6F, 0x6D, 0x31, 0x19,
  0x30, 0x17, 0x06, 0x0A, 0x09, 0x92, 0x26, 0x89, 0x93, 0xF2, 0x2C, 0x64, 0x01, 0x19, 0x16, 0x09,
  0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x31, 0x2D, 0x30, 0x2B, 0x06, 0x03, 0x55,
  0x04, 0x03, 0x13, 0x24, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x52, 0x6F,
  0x6F, 0x74, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x41,
  0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x37, 0x30, 0x34,
  0x30, 0x33, 0x31, 0x32, 0x35, 0x33, 0x30, 0x39, 0x5A, 0x17, 0x0D, 0x32, 0x31, 0x30, 0x34, 0x30,
  0x33, 0x31, 0x33, 0x30, 0x33, 0x30, 0x39, 0x5A, 0x30, 0x77, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03,
  0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08,
  0x13, 0x0A, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6E, 0x67, 0x74, 0x6F, 0x6E, 0x31, 0x10, 0x30, 0x0E,
  0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65, 0x64, 0x6D, 0x6F, 0x6E, 0x64, 0x31, 0x1E,
  0x30, 0x1C, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x15, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F,
  0x66, 0x74, 0x20, 0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x31, 0x21,
  0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x18, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F,
  0x66, 0x74, 0x20, 0x54, 0x69, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61, 0x6D, 0x70, 0x20, 0x50, 0x43,
  0x41, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01,
  0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01,
  0x01, 0x00, 0x9F, 0xA1, 0x6C, 0xB1, 0xDF, 0xDB, 0x48, 0x92, 0x2A, 0x7C, 0x6B, 0x2E, 0x19, 0xE1,
  0xBD, 0xE2, 0xE3, 0xC5, 0x99, 0x51, 0x23, 0x50, 0xAD, 0xCE, 0xDD, 0x18, 0x4E, 0x24, 0x0F, 0xEE,
  0xD1, 0xA7, 0xD1, 0x4C, 0xAD, 0x74, 0x30, 0x20, 0x11, 0xEB, 0x07, 0xD5, 0x54, 0x95, 0x15, 0x49,
  0x94, 0x1B, 0x42, 0x92, 0xAE, 0x98, 0x5C, 0x30, 0x26, 0xDA, 0x00, 0x6B, 0xE8, 0x7B, 0xBD, 0xEC,
  0x89, 0x07, 0x0F, 0xF7, 0x0E, 0x04, 0x98, 0xF0, 0x89, 0xCC, 0x1F, 0xCB, 0x33, 0x24, 0x87, 0x9D,
  0xF2, 0xF4, 0x67, 0x1C, 0x2C, 0xFC, 0x7B, 0xE7, 0x88, 0x1D, 0xEA, 0xE7, 0x4E, 0xA3, 0xA1, 0xC1,
  0x23, 0x53, 0xCA, 0x8D, 0xFA, 0x45, 0xCF, 0x09, 0xD0, 0x5E, 0xAF, 0xD0, 0xB0, 0x42, 0x04, 0xA2,
  0xF9, 0xA6, 0x6C, 0x93, 0x67, 0xD7, 0x28, 0xDC, 0x46, 0x53, 0xB0, 0x86, 0xD0, 0xE5, 0x28, 0x46,
  0x2E, 0x27, 0xAC, 0x86, 0x4F, 0x55, 0x52, 0x0C, 0xE4, 0x03, 0x2C, 0xFB, 0x6A, 0x90, 0x90, 0x30,
  0x6E, 0x87, 0xF3, 0x59, 0x30, 0x9D, 0xFA, 0x7E, 0xD6, 0x97, 0xB3, 0xE8, 0x21, 0x97, 0x7E, 0xF8,
  0xD2, 0x13, 0xF3, 0x08, 0xB7, 0x53, 0x6D, 0x52, 0xB4, 0x45, 0x90, 0x9F, 0x48, 0x00, 0x4A, 0x47,
  0x66, 0x11, 0x27, 0x29, 0x66, 0xA8, 0x97, 0xE4, 0xD3, 0x06, 0x81, 0x4A, 0xA2, 0xF9, 0x84, 0xA7,
  0x11, 0x47, 0x14, 0x09, 0x82, 0x9F, 0x84, 0xED, 0x55, 0x78, 0xFE, 0x01, 0x9A, 0x1D, 0x50, 0x08,
  0x85, 0x00, 0x10, 0x30, 0x46, 0xED, 0xB7, 0xDE, 0x23, 0x46, 0xBB, 0xC4, 0x2D, 0x54, 0x9F, 0xAF,
  0x1E, 0x78, 0x41, 0x31, 0x77, 0xCC, 0x9B, 0xDF, 0x3B, 0x83, 0x93, 0xA1, 0x61, 0x02, 0xB5, 0x1D,
  0x0D, 0xB1, 0xFC, 0xF7, 0x9B, 0xB2, 0x01, 0xCE, 0x22, 0x4B, 0x54, 0xFF, 0xF9, 0x05, 0xC3, 0xC2,
  0x20, 0x0B, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x82, 0x01, 0xAB, 0x30, 0x82, 0x01, 0xA7, 0x30,
  0x0F, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF,
  0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0x23, 0x34, 0xF8, 0xD9, 0x52,
  0x46, 0x70, 0x0A, 0xED, 0x40, 0xFB, 0x76, 0xFB, 0xB3, 0x2B, 0xB0, 0xC3, 0x35, 0xB3, 0x0F, 0x30,
  0x0B, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x04, 0x04, 0x03, 0x02, 0x01, 0x86, 0x30, 0x10, 0x06, 0x09,
  0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x01, 0x04, 0x03, 0x02, 0x01, 0x00, 0x30, 0x81,
  0x98, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x81, 0x90, 0x30, 0x81, 0x8D, 0x80, 0x14, 0x0E, 0xAC,
  0x82, 0x60, 0x40, 0x56, 0x27, 0x97, 0xE5, 0x25, 0x13, 0xFC, 0x2A, 0xE1, 0x0A, 0x53, 0x95, 0x59,
  0xE4, 0xA4, 0xA1, 0x63, 0xA4, 0x61, 0x30, 0x5F, 0x31, 0x13, 0x30, 0x11, 0x06, 0x0A, 0x09, 0x92,
  0x26, 0x89, 0x93, 0xF2, 0x2C, 0x64, 0x01, 0x19, 0x16, 0x03, 0x63, 0x6F, 0x6D, 0x31, 0x19, 0x30,
  0x17, 0x06, 0x0A, 0x09, 0x92, 0x26, 0x89, 0x93, 0xF2, 0x2C, 0x64, 0x01, 0x19, 0x16, 0x09, 0x6D,
  0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x31, 0x2D, 0x30, 0x2B, 0x06, 0x03, 0x55, 0x04,
  0x03, 0x13, 0x24, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x52, 0x6F, 0x6F,
  0x74, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75,
  0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x82, 0x10, 0x79, 0xAD, 0x16, 0xA1, 0x4A, 0xA0, 0xA5,
  0xAD, 0x4C, 0x73, 0x58, 0xF4, 0x07, 0x13, 0x2E, 0x65, 0x30, 0x50, 0x06, 0x03, 0x55, 0x1D, 0x1F,
  0x04, 0x49, 0x30, 0x47, 0x30, 0x45, 0xA0, 0x43, 0xA0, 0x41, 0x86, 0x3F, 0x68, 0x74, 0x74, 0x70,
  0x3A, 0x2F, 0x2F, 0x63, 0x72, 0x6C, 0x2E, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74,
  0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x2F, 0x63, 0x72, 0x6C, 0x2F, 0x70, 0x72, 0x6F,
  0x64, 0x75, 0x63, 0x74, 0x73, 0x2F, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x72,
  0x6F, 0x6F, 0x74, 0x63, 0x65, 0x72, 0x74, 0x2E, 0x63, 0x72, 0x6C, 0x30, 0x54, 0x06, 0x08, 0x2B,
  0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x48, 0x30, 0x46, 0x30, 0x44, 0x06, 0x08, 0x2B,
  0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x38, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F,
  0x77, 0x77, 0x77, 0x2E, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x2E, 0x63, 0x6F,
  0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x2F, 0x63, 0x65, 0x72, 0x74, 0x73, 0x2F, 0x4D, 0x69, 0x63, 0x72,
  0x6F, 0x73, 0x6F, 0x66, 0x74, 0x52, 0x6F, 0x6F, 0x74, 0x43, 0x65, 0x72, 0x74, 0x2E, 0x63, 0x72,
  0x74, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 0x06,
  0x01, 0x05, 0x05, 0x07, 0x03, 0x08, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
  0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x02, 0x01, 0x00, 0x10, 0x97, 0x8A, 0xC3, 0x5C, 0x03,
  0x44, 0x36, 0xDD, 0xE9, 0xB4, 0xAD, 0x77, 0xDB, 0xCE, 0x79, 0x51, 0x4D, 0x01, 0xB1, 0x2E, 0x74,
  0x71, 0x5B, 0x6D, 0x0C, 0x13, 0xAB, 0xCE, 0xBE, 0x7B, 0x8F, 0xB8, 0x2E, 0xD4, 0x12, 0xA2, 0x8C,
  0x6D, 0x62, 0xB8, 0x57, 0x02, 0xCB, 0x4E, 0x20, 0x13, 0x50, 0x99, 0xDD, 0x7A, 0x40, 0xE2, 0x57,
  0xBB, 0xAF, 0x58, 0x9A, 0x1C, 0xE1, 0x1D, 0x01, 0x86, 0xAC, 0xBB, 0x78, 0xF2, 0x8B, 0xD0, 0xEC,
  0x3B, 0x01, 0xEE, 0xE2, 0xBE, 0x8F, 0x0A, 0x05, 0xC8, 0x8D, 0x48, 0xE2, 0xF0, 0x53, 0x15, 0xDD,
  0x4F, 0xAB, 0x92, 0xE4, 0xE7, 0x8D, 0x6A, 0xD5, 0x80, 0xC1, 0xE6, 0x94, 0xF2, 0x06, 0x2F, 0x85,
  0x03, 0xE9, 0x91, 0x2A, 0x24, 0x22, 0x70, 0xFB, 0xF6, 0xFC, 0xE4, 0x78, 0x99, 0x2E, 0x0D, 0xF7,
  0x07, 0xE2, 0x70, 0xBC, 0x18, 0x4E, 0x9D, 0x8E, 0x6B, 0x0A, 0x72, 0x95, 0xB8, 0xA1, 0x39, 0x9C,
  0x67, 0x2D, 0xC5, 0x51, 0x0E, 0xEA, 0x62, 0x5C, 0x3F, 0x16, 0x98, 0x8B, 0x20, 0x3F, 0xE2, 0x07,
  0x1A, 0x32, 0xF9, 0xCC, 0x31, 0x4A, 0x76, 0x31, 0x3D, 0x2B, 0x72, 0x0B, 0xC8, 0xEA, 0x70, 0x3D,
  0xFF, 0x85, 0x0A, 0x13, 0xDF, 0xC2, 0x0A, 0x61, 0x8E, 0xF0, 0xD7, 0xB8, 0x17, 0xEB, 0x4E, 0x8B,
  0x7F, 0xC5, 0x35, 0x2B, 0x5E, 0xA3, 0xBF, 0xEB, 0xBC, 0x7D, 0x0B, 0x42, 0x7B, 0xD4, 0x53, 0x72,
  0x21, 0xEE, 0x30, 0xCA, 0xBB, 0x78, 0x65, 0x5C, 0x5B, 0x01, 0x17, 0x0A, 0x14, 0x0E, 0xD2, 0xDA,
  0x14, 0x98, 0xF5, 0x3C, 0xB9, 0x66, 0x58, 0xB3, 0x2D, 0x2F, 0xE7, 0xF9, 0x85, 0x86, 0xCC, 0x51,
  0x56, 0xE8, 0x9D, 0x70, 0x94, 0x6C, 0xAC, 0x39, 0x4C, 0xD4, 0xF6, 0x79, 0xBF, 0xAA, 0x18, 0x7A,
  0x62, 0x29, 0xEF, 0xA2, 0x9B, 0x29, 0x34, 0x06, 0x77, 0x1A, 0x62, 0xC9, 0x3D, 0x1E, 0x6D, 0x1F,
  0x82, 0xF0, 0x0B, 0xC7, 0x2C, 0xBB, 0xCF, 0x43, 0xB3, 0xE5, 0xF9, 0xEC, 0x7D, 0xB5, 0xE3, 0xA4,
  0xA8, 0x74, 0x35, 0xB8, 0x4E, 0xC5, 0x71, 0x23, 0x12, 0x26, 0x76, 0x0B, 0x3C, 0x52, 0x8C, 0x71,
  0x5A, 0x46, 0x43, 0x14, 0xBC, 0xB3, 0xB3, 0xB0, 0x4D, 0x67, 0xC8, 0x9F, 0x42, 0xFF, 0x80, 0x79,
  0x21, 0x80, 0x9E, 0x15, 0x30, 0x66, 0xE8, 0x42, 0x12, 0x5E, 0x1A, 0xC8, 0x9E, 0x22, 0x21, 0xD0,
  0x43, 0xE9, 0x2B, 0xE9, 0xBB, 0xF4, 0x48, 0xCC, 0x2C, 0xD4, 0xD8, 0x32, 0x80, 0x4C, 0x26, 0x2A,
  0x48, 0x24, 0x5F, 0x5A, 0xEA, 0x56, 0xEF, 0xA6, 0xDE, 0x99, 0x9D, 0xCA, 0x3A, 0x6F, 0xBD, 0x81,
  0x27, 0x74, 0x06, 0x11, 0xEE, 0x76, 0x21, 0xBF, 0x9B, 0x82, 0xC1, 0x27, 0x54, 0xB6, 0xB1, 0x6A,
  0x3D, 0x89, 0xA1, 0x76, 0x61, 0xB4, 0x6E, 0xA1, 0x13, 0xA6, 0xBF, 0xAA, 0x47, 0xF0, 0x12, 0x6F,
  0xFD, 0x8A, 0x32, 0x6C, 0xB2, 0xFE, 0xDF, 0x51, 0xC8, 0x8C, 0x23, 0xC9, 0x66, 0xBD, 0x9D, 0x1D,
  0x87, 0x12, 0x64, 0x02, 0x3D, 0x2D, 0xAF, 0x59, 0x8F, 0xB8, 0xE4, 0x21, 0xE5, 0xB5, 0xB0, 0xCA,
  0x63, 0xB4, 0x78, 0x54, 0x05, 0xD4, 0x41, 0x2E, 0x50, 0xAC, 0x94, 0xB0, 0xA5, 0x78, 0xAB, 0xB3,
  0xA0, 0x96, 0x75, 0x1A, 0xD9, 0x92, 0x87, 0x13, 0x75, 0x22, 0x2F, 0x32, 0xA8, 0x08, 0x6E, 0xA0,
  0x5B, 0x8C, 0x25, 0xBF, 0xA0, 0xEF, 0x84, 0xCA, 0x21, 0xD6, 0xEB, 0x1E, 0x4F, 0xC9, 0x9A, 0xEE,
  0x49, 0xE0, 0xF7, 0x01, 0x65, 0x6F, 0x89, 0x0B, 0x7D, 0xC8, 0x69, 0xC8, 0xE6, 0x6E, 0xEA, 0xA7,
  0x97, 0xCE, 0x31, 0x29, 0xFF, 0x0E, 0xC5, 0x5B, 0x5C, 0xD8, 0x4D, 0x1B, 0xA1, 0xD8, 0xFA, 0x2F,
  0x9E, 0x3F, 0x2E, 0x55, 0x16, 0x6B, 0xC9, 0x13, 0xA3, 0xFD, 0x30, 0x82, 0x06, 0xA1, 0x30, 0x82,
  0x04, 0x89, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x0A, 0x61, 0x07, 0x02, 0xDC, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0B, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05,
  0x05, 0x00, 0x30, 0x5F, 0x31, 0x13, 0x30, 0x11, 0x06, 0x0A, 0x09, 0x92, 0x26, 0x89, 0x93, 0xF2,
  0x2C, 0x64, 0x01, 0x19, 0x16, 0x03, 0x63, 0x6F, 0x6D, 0x31, 0x19, 0x30, 0x17, 0x06, 0x0A, 0x09,
  0x92, 0x26, 0x89, 0x93, 0xF2, 0x2C, 0x64, 0x01, 0x19, 0x16, 0x09, 0x6D, 0x69, 0x63, 0x72, 0x6F,
  0x73, 0x6F, 0x66, 0x74, 0x31, 0x2D, 0x30, 0x2B, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x24, 0x4D,
  0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x65,
  0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72,
  0x69, 0x74, 0x79, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x35, 0x30, 0x39, 0x31, 0x35, 0x32, 0x31, 0x35,
  0x35, 0x34, 0x31, 0x5A, 0x17, 0x0D, 0x31, 0x36, 0x30, 0x33, 0x31, 0x35, 0x32, 0x32, 0x30, 0x35,
  0x34, 0x31, 0x5A, 0x30, 0x81, 0x81, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
  0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0A, 0x57, 0x61,
  0x73, 0x68, 0x69, 0x6E, 0x67, 0x74, 0x6F, 0x6E, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04,
  0x07, 0x13, 0x07, 0x52, 0x65, 0x64, 0x6D, 0x6F, 0x6E, 0x64, 0x31, 0x1E, 0x30, 0x1C, 0x06, 0x03,
  0x55, 0x04, 0x0A, 0x13, 0x15, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x43,
  0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x31, 0x2B, 0x30, 0x29, 0x06, 0x03,
  0x55, 0x04, 0x03, 0x13, 0x22, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x57,
  0x69, 0x6E, 0x64, 0x6F, 0x77, 0x73, 0x20, 0x56, 0x65, 0x72, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74,
  0x69, 0x6F, 0x6E, 0x20, 0x50, 0x43, 0x41, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A,
  0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30,
  0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xC5, 0xAC, 0x93, 0x4E, 0xE6, 0x4A, 0x11, 0x9E,
  0x37, 0xD0, 0x35, 0xD2, 0xB0, 0x06, 0x5C, 0x83, 0x14, 0xA5, 0x61, 0x92, 0xFC, 0xFE, 0xB3, 0x89,
  0xC1, 0xE6, 0xCD, 0xC8, 0x1F, 0x31, 0x84, 0x09, 0x91, 0x34, 0x46, 0x92, 0x2C, 0xB8, 0xEE, 0x2C,
  0xC5, 0x20, 0x73, 0xC4, 0xE8, 0x1A, 0xD8, 0x8D, 0xCE, 0x7A, 0x68, 0xB5, 0x66, 0x78, 0x8B, 0xE5,
  0x60, 0x15, 0xA2, 0xF6, 0x29, 0x5A, 0x1D, 0x5E, 0x5D, 0xFC, 0x62, 0xD3, 0xC1, 0x5C, 0x29, 0x89,
  0xFB, 0x33, 0xF8, 0x16, 0x95, 0x36, 0xDD, 0xB1, 0x46, 0x74, 0x69, 0x72, 0x53, 0xD5, 0xAA, 0xE8,
  0x8A, 0x99, 0x1A, 0xD8, 0xF7, 0x67, 0x09, 0xD9, 0x09, 0x20, 0x22, 0x38, 0x7A, 0xD6, 0x03, 0x23,
  0xD7, 0x89, 0x9F, 0x1C, 0x01, 0xB7, 0x51, 0xDF, 0x98, 0x66, 0x2A, 0x02, 0x8B, 0x06, 0xEC, 0xE4,
  0x29, 0xEF, 0x5B, 0x42, 0x17, 0x15, 0x97, 0x51, 0x8D, 0x7D, 0x25, 0x0A, 0xAF, 0x25, 0xE8, 0xDE,
  0xDC, 0x9B, 0x67, 0x16, 0x0A, 0x56, 0x73, 0x9D, 0xB3, 0x1D, 0x85, 0x83, 0x0B, 0x7E, 0x33, 0x2B,
  0x62, 0x33, 0xCE, 0x1C, 0x20, 0x81, 0x4B, 0x5E, 0xD3, 0xC6, 0x49, 0xB8, 0xF6, 0x2E, 0xD3, 0x4E,
  0xB0, 0x71, 0x44, 0x46, 0x64, 0x3E, 0xDE, 0x43, 0x66, 0x04, 0xB9, 0xCC, 0x83, 0x48, 0x3A, 0xC5,
  0x36, 0x7A, 0x04, 0x48, 0x0B, 0x89, 0x02, 0x3D, 0x63, 0xA2, 0x01, 0x76, 0x29, 0x97, 0x75, 0xE9,
  0x01, 0xE6, 0x00, 0x97, 0x09, 0x92, 0xF8, 0xE2, 0x27, 0xF0, 0x29, 0x67, 0x43, 0x77, 0xC3, 0x50,
  0x96, 0x53, 0x31, 0xE1, 0xB6, 0x71, 0x8B, 0xEC, 0x77, 0xC7, 0x7C, 0x31, 0x48, 0xD5, 0xB8, 0x25,
  0x22, 0x8C, 0x00, 0xF7, 0x28, 0x38, 0x7A, 0xBD, 0x7C, 0xC7, 0x3F, 0xCD, 0x40, 0x26, 0x77, 0xDD,
  0x00, 0x00, 0x11, 0x9A, 0x95, 0xBE, 0x1F, 0xDB, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x82, 0x02,
  0x3A, 0x30, 0x82, 0x02, 0x36, 0x30, 0x10, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37,
  0x15, 0x01, 0x04, 0x03, 0x02, 0x01, 0x00, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16,
  0x04, 0x14, 0x90, 0x8B, 0x11, 0xA5, 0x70, 0xED, 0xE0, 0xF9, 0xA9, 0xC0, 0xAC, 0x08, 0xC7, 0xB5,
  0xF4, 0x82, 0xB1, 0x3C, 0xC5, 0x4A, 0x30, 0x0B, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x04, 0x04, 0x03,
  0x02, 0x01, 0xC6, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x05, 0x30,
  0x03, 0x01, 0x01, 0xFF, 0x30, 0x81, 0x98, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x81, 0x90, 0x30,
  0x81, 0x8D, 0x80, 0x14, 0x0E, 0xAC, 0x82, 0x60, 0x40, 0x56, 0x27, 0x97, 0xE5, 0x25, 0x13, 0xFC,
  0x2A, 0xE1, 0x0A, 0x53, 0x95, 0x59, 0xE4, 0xA4, 0xA1, 0x63, 0xA4, 0x61, 0x30, 0x5F, 0x31, 0x13,
  0x30, 0x11, 0x06, 0x0A, 0x09, 0x92, 0x26, 0x89, 0x93, 0xF2, 0x2C, 0x64, 0x01, 0x19, 0x16, 0x03,
  0x63, 0x6F, 0x6D, 0x31, 0x19, 0x30, 0x17, 0x06, 0x0A, 0x09, 0x92, 0x26, 0x89, 0x93, 0xF2, 0x2C,
  0x64, 0x01, 0x19, 0x16, 0x09, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x31, 0x2D,
  0x30, 0x2B, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x24, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F,
  0x66, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63,
  0x61, 0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x82, 0x10, 0x79,
  0xAD, 0x16, 0xA1, 0x4A, 0xA0, 0xA5, 0xAD, 0x4C, 0x73, 0x58, 0xF4, 0x07, 0x13, 0x2E, 0x65, 0x30,
  0x50, 0x06, 0x03, 0x55, 0x1D, 0x1F, 0x04, 0x49, 0x30, 0x47, 0x30, 0x45, 0xA0, 0x43, 0xA0, 0x41,
  0x86, 0x3F, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x63, 0x72, 0x6C, 0x2E, 0x6D, 0x69, 0x63,
  0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x2F, 0x63,
  0x72, 0x6C, 0x2F, 0x70, 0x72, 0x6F, 0x64, 0x75, 0x63, 0x74, 0x73, 0x2F, 0x6D, 0x69, 0x63, 0x72,
  0x6F, 0x73, 0x6F, 0x66, 0x74, 0x72, 0x6F, 0x6F, 0x74, 0x63, 0x65, 0x72, 0x74, 0x2E, 0x63, 0x72,
  0x6C, 0x30, 0x54, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x48, 0x30,
  0x46, 0x30, 0x44, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x38, 0x68,
  0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73,
  0x6F, 0x66, 0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x2F, 0x63, 0x65, 0x72, 0x74,
  0x73, 0x2F, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x52, 0x6F, 0x6F, 0x74, 0x43,
  0x65, 0x72, 0x74, 0x2E, 0x63, 0x72, 0x74, 0x30, 0x81, 0x80, 0x06, 0x03, 0x55, 0x1D, 0x20, 0x04,
  0x79, 0x30, 0x77, 0x30, 0x75, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x2F,
  0x30, 0x68, 0x30, 0x66, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x02, 0x30, 0x5A,
  0x1E, 0x58, 0x00, 0x43, 0x00, 0x6F, 0x00, 0x70, 0x00, 0x79, 0x00, 0x72, 0x00, 0x69, 0x00, 0x67,
  0x00, 0x68, 0x00, 0x74, 0x00, 0x20, 0x00, 0xA9, 0x00, 0x20, 0x00, 0x31, 0x00, 0x39, 0x00, 0x39,
  0x00, 0x39, 0x00, 0x2D, 0x00, 0x32, 0x00, 0x30, 0x00, 0x30, 0x00, 0x35, 0x00, 0x20, 0x00, 0x4D,
  0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x6F, 0x00, 0x66, 0x00, 0x74,
  0x00, 0x20, 0x00, 0x43, 0x00, 0x6F, 0x00, 0x72, 0x00, 0x70, 0x00, 0x6F, 0x00, 0x72, 0x00, 0x61,
  0x00, 0x74, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x2E, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D,
  0x25, 0x04, 0x18, 0x30, 0x16, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x03, 0x06,
  0x0A, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x0A, 0x03, 0x06, 0x30, 0x0D, 0x06, 0x09, 0x2A,
  0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x02, 0x01, 0x00, 0x25,
  0x31, 0xA1, 0x58, 0xEA, 0x52, 0xE5, 0xE1, 0x17, 0x0C, 0xE6, 0xF1, 0x3F, 0x78, 0xA3, 0x3F, 0x72,
  0xAF, 0xA7, 0x57, 0x10, 0x53, 0x89, 0x10, 0x5E, 0x32, 0x9C, 0xB6, 0x70, 0xC3, 0x15, 0x2B, 0x4D,
  0x15, 0x30, 0x34, 0xE8, 0xC0, 0x6A, 0xE4, 0x1C, 0xD3, 0x2E, 0x20, 0x65, 0x48, 0xD7, 0x1B, 0x98,
  0x62, 0x21, 0xBA, 0x45, 0x9F, 0x4A, 0xEC, 0xDB, 0x2F, 0x09, 0x19, 0x51, 0xE5, 0xED, 0x32, 0x19,
  0x51, 0x2F, 0xE1, 0xDD, 0xFB, 0xC6, 0x52, 0xFD, 0xEB, 0xC6, 0x82, 0x25, 0x42, 0x03, 0x09, 0xA6,
  0x38, 0xB6, 0x36, 0x1F, 0xCC, 0xC9, 0x80, 0xBB, 0x5A, 0x69, 0x18, 0x31, 0xC3, 0xB3, 0xA0, 0xB3,
  0x67, 0x47, 0xBE, 0x9D, 0xC7, 0xE2, 0x3F, 0x96, 0xB3, 0x88, 0xF8, 0x19, 0xBE, 0x39, 0xB9, 0xE9,
  0x95, 0xCE, 0xFC, 0x7C, 0xAF, 0xA8, 0xCD, 0xD0, 0x41, 0x90, 0xE0, 0xD5, 0xB3, 0x1C, 0x2F, 0x68,
  0xBB, 0xDB, 0x0F, 0x6C, 0x6A, 0xDD, 0xF2, 0xAF, 0xDE, 0xF2, 0xB5, 0xDE, 0x0D, 0xB6, 0xA6, 0x5A,
  0xF0, 0x86, 0x0A, 0xB9, 0x6D, 0x99, 0x4B, 0x3F, 0x7B, 0x2D, 0x01, 0x84, 0x6C, 0x8F, 0x87, 0xDC,
  0x7F, 0x8F, 0xAB, 0x14, 0x88, 0xD0, 0x06, 0x91, 0x34, 0xBE, 0x1B, 0x82, 0x22, 0xA4, 0xBC, 0x55,
  0x8A, 0xAD, 0x9B, 0xFC, 0x73, 0x14, 0x10, 0xC4, 0xC9, 0x19, 0x1E, 0x07, 0x7D, 0x9B, 0x0E, 0xC0,
  0x95, 0x26, 0x5D, 0xC6, 0x1F, 0xAC, 0xB4, 0xF2, 0x7E, 0xBA, 0x25, 0x70, 0x4A, 0x7B, 0xD7, 0x8E,
  0xD1, 0x9D, 0xA0, 0x13, 0x49, 0x7A, 0xB0, 0x02, 0x52, 0x52, 0x24, 0xF4, 0xAF, 0xDD, 0x40, 0x2D,
  0xE5, 0x3E, 0x32, 0x58, 0xB3, 0x4A, 0x6A, 0xDD, 0x11, 0x59, 0xAA, 0x2D, 0xBC, 0xA4, 0xA0, 0x73,
  0x38, 0xF9, 0x40, 0x77, 0x6B, 0x34, 0x19, 0x57, 0xCD, 0x38, 0x68, 0x27, 0x82, 0xF8, 0xD1, 0x6F,
  0xEB, 0x23, 0xC0, 0x3F, 0x52, 0xF3, 0x4E, 0xD5, 0x02, 0x3E, 0x6A, 0x9A, 0x2B, 0xC1, 0xF5, 0x31,
  0x71, 0xDB, 0x41, 0x4D, 0x3B, 0xDE, 0xEF, 0xAD, 0xAF, 0x1F, 0x88, 0x65, 0x43, 0x1B, 0x51, 0xB7,
  0x9A, 0x75, 0xCA, 0x8E, 0x69, 0x49, 0x10, 0x8F, 0x78, 0x8A, 0x74, 0x45, 0xB9, 0x09, 0x8E, 0x73,
  0x77, 0x07, 0x32, 0x4A, 0x4B, 0xD7, 0x68, 0x2B, 0x98, 0xC5, 0xBA, 0x54, 0xEA, 0x3F, 0xCB, 0xA2,
  0x00, 0x8C, 0xBB, 0xD8, 0x10, 0x58, 0xF2, 0xDB, 0xDC, 0x9B, 0xCD, 0xD8, 0xEA, 0x48, 0x43, 0xE2,
  0x4A, 0x7E, 0x65, 0xB2, 0xDC, 0xF5, 0x2D, 0x4E, 0x25, 0x67, 0xA8, 0xE0, 0xB5, 0xBA, 0xA7, 0xDD,
  0x7E, 0x5E, 0xC1, 0x4C, 0x02, 0x74, 0xC9, 0xB3, 0x6E, 0xE3, 0xF8, 0xF0, 0x0B, 0xED, 0xFC, 0xB9,
  0x29, 0xC5, 0x5B, 0xC9, 0x36, 0x51, 0x90, 0xDB, 0x78, 0x7D, 0xB9, 0x32, 0x0F, 0x5E, 0x76, 0xD2,
  0x15, 0x5C, 0x3B, 0x37, 0x21, 0xC6, 0xDB, 0xC9, 0x19, 0x6E, 0xED, 0x74, 0x2A, 0x5C, 0x2C, 0x0B,
  0x51, 0x49, 0x45, 0x53, 0xB0, 0xB2, 0xB3, 0x23, 0xD4, 0xA1, 0xB0, 0x5F, 0x0D, 0x19, 0xCD, 0x14,
  0xA7, 0xE3, 0x3C, 0x9B, 0x97, 0x72, 0x94, 0x14, 0xDF, 0xFF, 0xC1, 0x90, 0x1B, 0xA5, 0xDF, 0xF5,
  0xA9, 0xF3, 0x1B, 0x17, 0xDA, 0xB5, 0xFC, 0x44, 0xE0, 0xE8, 0xE2, 0x3C, 0xA2, 0x7A, 0xBB, 0xBB,
  0x65, 0xE6, 0x4D, 0xB1, 0xB5, 0x15, 0xA1, 0xD9, 0x67, 0x3B, 0xB0, 0x0C, 0x7D, 0x3B, 0xE9, 0xEE,
  0x51, 0x2A, 0x47, 0xF5, 0x15, 0x0F, 0x8C, 0xAD, 0x5D, 0x2E, 0x35, 0xDF, 0xF4, 0xA4, 0x2E, 0xF6,
  0x13, 0x37, 0x5A, 0x2B, 0xE8, 0x55, 0x9A, 0x49, 0x2C, 0x97, 0xCE, 0x9D, 0x01, 0x9E, 0x97, 0x46,
  0x5C, 0xD9, 0x2D, 0xBC, 0x24, 0x5A, 0x95, 0x59, 0x6F, 0x4D, 0xCA, 0x9D, 0xD6, 0x57, 0x26, 0x31,
  0x82, 0x04, 0x96, 0x30, 0x82, 0x04, 0x92, 0x02, 0x01, 0x01, 0x30, 0x81, 0x90, 0x30, 0x81, 0x81,
  0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30,
  0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0A, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6E, 0x67, 0x74,
  0x6F, 0x6E, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65, 0x64,
  0x6D, 0x6F, 0x6E, 0x64, 0x31, 0x1E, 0x30, 0x1C, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x15, 0x4D,
  0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61,
  0x74, 0x69, 0x6F, 0x6E, 0x31, 0x2B, 0x30, 0x29, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x22, 0x4D,
  0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x57, 0x69, 0x6E, 0x64, 0x6F, 0x77, 0x73,
  0x20, 0x56, 0x65, 0x72, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x20, 0x50, 0x43,
  0x41, 0x02, 0x0A, 0x61, 0x01, 0xC6, 0xC1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x30, 0x09, 0x06,
  0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0xA0, 0x81, 0xBA, 0x30, 0x19, 0x06, 0x09, 0x2A,
  0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x03, 0x31, 0x0C, 0x06, 0x0A, 0x2B, 0x06, 0x01, 0x04,
  0x01, 0x82, 0x37, 0x02, 0x01, 0x04, 0x30, 0x1C, 0x06, 0x0A, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82,
  0x37, 0x02, 0x01, 0x0B, 0x31, 0x0E, 0x30, 0x0C, 0x06, 0x0A, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82,
  0x37, 0x02, 0x01, 0x15, 0x30, 0x23, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09,
  0x04, 0x31, 0x16, 0x04, 0x14, 0xC5, 0xC5, 0xC1, 0xC3, 0x46, 0xC4, 0x12, 0xB9, 0x38, 0xED, 0xE5,
  0x7E, 0x66, 0x78, 0x4E, 0x8F, 0x55, 0x10, 0x7D, 0xAC, 0x30, 0x5A, 0x06, 0x0A, 0x2B, 0x06, 0x01,
  0x04, 0x01, 0x82, 0x37, 0x02, 0x01, 0x0C, 0x31, 0x4C, 0x30, 0x4A, 0xA0, 0x24, 0x80, 0x22, 0x00,
  0x4D, 0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x6F, 0x00, 0x66, 0x00,
  0x74, 0x00, 0x20, 0x00, 0x57, 0x00, 0x69, 0x00, 0x6E, 0x00, 0x64, 0x00, 0x6F, 0x00, 0x77, 0x00,
  0x73, 0xA1, 0x22, 0x80, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E,
  0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x77, 0x69,
  0x6E, 0x64, 0x6F, 0x77, 0x73, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01,
  0x01, 0x01, 0x05, 0x00, 0x04, 0x82, 0x01, 0x00, 0xC0, 0x53, 0x19, 0xE7, 0x27, 0xE6, 0xD6, 0xD7,
  0xA8, 0xAE, 0xA0, 0xE6, 0x4C, 0xEF, 0x9A, 0xBC, 0x1D, 0x19, 0xD4, 0xC5, 0x7F, 0x7C, 0x71, 0xFF,
  0x14, 0x7C, 0xAD, 0x7A, 0x8F, 0x2D, 0x0F, 0x40, 0x38, 0x9B, 0x83, 0x09, 0xEE, 0xF5, 0x34, 0x92,
  0xB4, 0x85, 0x34, 0x7C, 0xBA, 0x9B, 0x58, 0x5D, 0xA6, 0x15, 0xDE, 0x99, 0xBF, 0x25, 0x89, 0xDB,
  0xC0, 0x3A, 0x98, 0x21, 0x1C, 0x25, 0x09, 0xBD, 0x41, 0xAA, 0x0F, 0x90, 0x6C, 0x62, 0xE9, 0xBD,
  0x1D, 0xDD, 0x7C, 0xA4, 0x1F, 0x19, 0x3F, 0xC8, 0x11, 0xF5, 0x1C, 0x9E, 0xEA, 0x2E, 0x1A, 0xCE,
  0x5A, 0x61, 0x8A, 0x77, 0xFA, 0x65, 0x1F, 0x03, 0xB4, 0x45, 0x02, 0x0F, 0xB4, 0x0E, 0xB9, 0x44,
  0xF6, 0x75, 0xF6, 0x67, 0x91, 0xAD, 0xFA, 0xF9, 0xC3, 0xFA, 0x63, 0x91, 0x58, 0xFE, 0x40, 0xDF,
  0x92, 0x6B, 0x59, 0x42, 0x44, 0x66, 0xD2, 0x88, 0x49, 0x59, 0x29, 0x1D, 0x6A, 0xF9, 0x93, 0x9B,
  0xD5, 0x38, 0x84, 0x96, 0xD7, 0xAF, 0xBA, 0xEC, 0x29, 0x8D, 0x06, 0x8C, 0x8B, 0xF4, 0xA1, 0x56,
  0x0F, 0x78, 0x3F, 0x35, 0xCB, 0x12, 0x8A, 0x85, 0xD5, 0x7C, 0xFC, 0x63, 0xDD, 0x72, 0xA1, 0x13,
  0xD3, 0x25, 0x4A, 0x59, 0x21, 0xD0, 0xC6, 0x47, 0x9F, 0x22, 0xF7, 0xF8, 0x32, 0xA2, 0x05, 0x2B,
  0x53, 0xC5, 0x96, 0x98, 0x0D, 0x39, 0x02, 0xAD, 0x85, 0xF7, 0x18, 0x73, 0x34, 0x12, 0x7F, 0x5A,
  0xCB, 0xE1, 0xA2, 0xFB, 0x4E, 0x25, 0x81, 0xF0, 0x9F, 0x7A, 0x79, 0xDB, 0x4F, 0x4B, 0x66, 0x44,
  0x30, 0x8E, 0x0D, 0x57, 0x23, 0xD3, 0x7F, 0x60, 0x87, 0xDE, 0xD0, 0x4C, 0x90, 0x90, 0x84, 0x64,
  0x70, 0xE8, 0x15, 0x65, 0xCA, 0x18, 0xF8, 0x9A, 0xB6, 0x0C, 0x24, 0xF5, 0xDF, 0xC3, 0x47, 0xE2,
  0x7C, 0xAD, 0x91, 0x70, 0x6D, 0xF8, 0x43, 0xE9, 0xA1, 0x82, 0x02, 0x1D, 0x30, 0x82, 0x02, 0x19,
  0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x06, 0x31, 0x82, 0x02, 0x0A, 0x30,
  0x82, 0x02, 0x06, 0x02, 0x01, 0x01, 0x30, 0x81, 0x85, 0x30, 0x77, 0x31, 0x0B, 0x30, 0x09, 0x06,
  0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04,
  0x08, 0x13, 0x0A, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6E, 0x67, 0x74, 0x6F, 0x6E, 0x31, 0x10, 0x30,
  0x0E, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65, 0x64, 0x6D, 0x6F, 0x6E, 0x64, 0x31,
  0x1E, 0x30, 0x1C, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x15, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73,
  0x6F, 0x66, 0x74, 0x20, 0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x31,
  0x21, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x18, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73,
  0x6F, 0x66, 0x74, 0x20, 0x54, 0x69, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61, 0x6D, 0x70, 0x20, 0x50,
  0x43, 0x41, 0x02, 0x0A, 0x61, 0x03, 0xDC, 0xF6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x30, 0x07,
  0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0xA0, 0x5D, 0x30, 0x18, 0x06, 0x09, 0x2A, 0x86, 0x48,
  0x86, 0xF7, 0x0D, 0x01, 0x09, 0x03, 0x31, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
  0x01, 0x07, 0x01, 0x30, 0x1C, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x05,
  0x31, 0x0F, 0x17, 0x0D, 0x30, 0x39, 0x30, 0x34, 0x32, 0x32, 0x30, 0x35, 0x35, 0x34, 0x33, 0x34,
  0x5A, 0x30, 0x23, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x04, 0x31, 0x16,
  0x04, 0x14, 0x74, 0xD7, 0x54, 0xA0, 0xC0, 0x9E, 0x62, 0xF0, 0xDD, 0x92, 0xDC, 0xD6, 0x96, 0x34,
  0xA1, 0x0B, 0xF0, 0xD5, 0xB5, 0x3A, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
  0x01, 0x01, 0x05, 0x05, 0x00, 0x04, 0x82, 0x01, 0x00, 0x8D, 0x16, 0xB5, 0xC8, 0x3D, 0xBC, 0x8A,
  0xB5, 0xC3, 0x23, 0xF3, 0x33, 0xB0, 0xFA, 0x75, 0xAD, 0xAC, 0x53, 0x0D, 0x5C, 0xF4, 0xD9, 0xBE,
  0x63, 0xEB, 0x0D, 0x6F, 0x69, 0x63, 0x91, 0xF8, 0x8E, 0x67, 0x03, 0xE1, 0xC6, 0x2E, 0xCD, 0x9C,
  0xEB, 0xDD, 0x9C, 0xCA, 0x65, 0x4A, 0xBB, 0xF2, 0xF4, 0x1C, 0x85, 0x44, 0xD6, 0x58, 0x70, 0x2E,
  0x3B, 0xFE, 0x97, 0xD0, 0xEC, 0x72, 0x77, 0x20, 0xA8, 0x20, 0xAC, 0xF0, 0xED, 0x86, 0xC4, 0xFF,
  0x3A, 0x3D, 0x40, 0xD9, 0xF6, 0xD5, 0x99, 0xBE, 0xCC, 0xD2, 0xD8, 0xBA, 0x71, 0x51, 0x60, 0xAA,
  0x72, 0x36, 0x13, 0x8B, 0x0A, 0xA9, 0xD6, 0xF5, 0x36, 0xE0, 0x44, 0x6B, 0xC1, 0x44, 0x42, 0x43,
  0x56, 0x1A, 0xD6, 0xF1, 0x5A, 0x98, 0xCC, 0xBA, 0xA9, 0x4C, 0xCF, 0xC9, 0x86, 0xC3, 0x5B, 0x1C,
  0xDF, 0x75, 0x22, 0x54, 0x6A, 0xF0, 0x18, 0x34, 0x0B, 0x29, 0xB0, 0x26, 0x07, 0x59, 0xB6, 0x52,
  0x7B, 0x69, 0x74, 0x11, 0x1B, 0xB8, 0x29, 0x19, 0x36, 0x0E, 0xB7, 0x65, 0xCB, 0x87, 0x09, 0x9E,
  0x03, 0x28, 0x6C, 0xDB, 0xA7, 0xFB, 0xDF, 0xC7, 0xF7, 0xCE, 0x0D, 0xC2, 0xBA, 0xEA, 0x3C, 0xE8,
  0x43, 0x1E, 0x7F, 0xF2, 0x64, 0xDA, 0x87, 0x89, 0xC0, 0xCD, 0x67, 0xE4, 0x84, 0x9B, 0xA7, 0x77,
  0xFA, 0x6C, 0x69, 0x19, 0x5B, 0xD1, 0xF7, 0xD9, 0x07, 0x91, 0x50, 0xDD, 0x98, 0x26, 0x10, 0xE3,
  0x2E, 0x39, 0x62, 0xF0, 0x56, 0xE9, 0xD9, 0x5C, 0x85, 0x67, 0x9B, 0x2F, 0xDE, 0xD1, 0x00, 0x45,
  0x0E, 0x99, 0x1F, 0xDF, 0x7E, 0xC9, 0x10, 0x56, 0xC7, 0x00, 0x6D, 0x5F, 0x23, 0x57, 0x12, 0x84,
  0xCD, 0xAC, 0x82, 0xAE, 0x39, 0x52, 0xA5, 0x19, 0x23, 0xA3, 0x6B, 0xE7, 0x49, 0x8F, 0x86, 0x74,
  0x46, 0x41, 0x2A, 0x0F, 0x3D, 0x29, 0xB7, 0xAE, 0x8C, 0x00
  };

//
// Sample Authenticode Data with SHA-256 hash algorithm.
// This data should be retrieved from signed PE/COFF image according to SECURITY
// directory in PE/COFF Header.
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT8 AuthenticodeWithSha256[] = {
  0x30, 0x82, 0x0e, 0xd1, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x07, 0x02, 0xa0,
  0x82, 0x0e, 0xc2, 0x30, 0x82, 0x0e, 0xbe, 0x02, 0x01, 0x01, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x09,
  0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x5c, 0x06, 0x0a, 0x2b,
  0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x02, 0x01, 0x04, 0xa0, 0x4e, 0x30, 0x4c, 0x30, 0x17, 0x06,
  0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x02, 0x01, 0x0f, 0x30, 0x09, 0x03, 0x01, 0x00,
  0xa0, 0x04, 0xa2, 0x02, 0x80, 0x00, 0x30, 0x31, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01,
  0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x04, 0x20, 0x61, 0x82, 0xb7, 0xf8, 0x8c, 0xff, 0xc2,
  0xeb, 0x79, 0x6e, 0x9d, 0xa9, 0xdd, 0x39, 0x52, 0xdd, 0x36, 0xdd, 0xf1, 0x43, 0x27, 0x58, 0x8c,
  0xa7, 0xcc, 0xae, 0xde, 0xdd, 0x3c, 0x02, 0x12, 0x49, 0xa0, 0x82, 0x0c, 0x0a, 0x30, 0x82, 0x05,
  0xe5, 0x30, 0x82, 0x03, 0xcd, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x0a, 0x61, 0x03, 0x5f, 0x09,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
  0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x81, 0x90, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
  0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a,
  0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03,
  0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65, 0x64, 0x6d, 0x6f, 0x6e, 0x64, 0x31, 0x1e, 0x30, 0x1c,
  0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x15, 0x4d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74,
  0x20, 0x43, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x3a, 0x30, 0x38,
  0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x31, 0x4d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74,
  0x20, 0x54, 0x65, 0x73, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x65,
  0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72,
  0x69, 0x74, 0x79, 0x20, 0x32, 0x30, 0x31, 0x30, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30, 0x30, 0x36,
  0x32, 0x31, 0x32, 0x32, 0x35, 0x35, 0x30, 0x31, 0x5a, 0x17, 0x0d, 0x31, 0x34, 0x30, 0x36, 0x32,
  0x31, 0x32, 0x33, 0x30, 0x35, 0x30, 0x31, 0x5a, 0x30, 0x81, 0x81, 0x31, 0x13, 0x30, 0x11, 0x06,
  0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x03, 0x63, 0x6f, 0x6d,
  0x31, 0x19, 0x30, 0x17, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19,
  0x16, 0x09, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x31, 0x14, 0x30, 0x12, 0x06,
  0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x04, 0x63, 0x6f, 0x72,
  0x70, 0x31, 0x17, 0x30, 0x15, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01,
  0x19, 0x16, 0x07, 0x72, 0x65, 0x64, 0x6d, 0x6f, 0x6e, 0x64, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x03,
  0x55, 0x04, 0x03, 0x13, 0x17, 0x4d, 0x53, 0x49, 0x54, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x43,
  0x6f, 0x64, 0x65, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x41, 0x20, 0x33, 0x30, 0x82, 0x01, 0x22,
  0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03,
  0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0x9b, 0x71, 0xdd,
  0x0d, 0x32, 0x2d, 0x08, 0xe8, 0x2d, 0x01, 0x90, 0x9c, 0xdf, 0x9f, 0x32, 0x58, 0x8b, 0x47, 0xd0,
  0xfe, 0x9a, 0x6b, 0x40, 0x5e, 0x76, 0xa2, 0x10, 0x00, 0x0e, 0xd2, 0x8f, 0xce, 0xa1, 0x5f, 0x42,
  0xa7, 0x0b, 0x5e, 0x83, 0xc2, 0xe9, 0x06, 0xa6, 0x0a, 0xb4, 0x3d, 0x30, 0x50, 0xdb, 0x35, 0x43,
  0x2c, 0xd3, 0xc3, 0xc6, 0x3c, 0xc9, 0xb5, 0x88, 0xfc, 0x0b, 0xe6, 0xa3, 0x71, 0x47, 0xce, 0xfd,
  0x88, 0x62, 0xa4, 0x25, 0xe2, 0x4e, 0x1b, 0x79, 0x3d, 0xe3, 0xbf, 0x8b, 0x84, 0x2d, 0x64, 0x25,
  0xa5, 0xfa, 0x8e, 0x30, 0xae, 0xe3, 0x34, 0x7d, 0x27, 0xf3, 0x5e, 0xf4, 0x05, 0x60, 0xc6, 0x0f,
  0x1a, 0x1b, 0x9e, 0xaf, 0x21, 0x37, 0xa9, 0xd9, 0xb6, 0xb7, 0x6a, 0x54, 0x4b, 0x5e, 0xa0, 0x75,
  0xb5, 0x9a, 0xc4, 0xb6, 0xbe, 0x28, 0xe3, 0x3b, 0xf7, 0x5a, 0x08, 0x77, 0xd9, 0x39, 0x4a, 0x2a,
  0x7a, 0xf8, 0xf4, 0xfe, 0xe8, 0x12, 0x54, 0x41, 0x11, 0x55, 0xc8, 0xc7, 0x98, 0x3e, 0x64, 0xa5,
  0xbb, 0xda, 0x6b, 0xa4, 0xaa, 0x29, 0x8c, 0x9f, 0x27, 0x55, 0x25, 0xe7, 0xfd, 0x31, 0x70, 0x60,
  0x1d, 0xd3, 0xc4, 0xf2, 0xb9, 0xef, 0x92, 0x4a, 0x9f, 0xc8, 0x4c, 0x93, 0xe7, 0x19, 0xa8, 0x4e,
  0xbc, 0xe3, 0x62, 0xf4, 0xe0, 0x6e, 0xec, 0x45, 0x6e, 0x61, 0x4e, 0xad, 0x58, 0xe0, 0xf0, 0xea,
  0xf7, 0x74, 0x2c, 0xdf, 0x1b, 0xf6, 0x43, 0x3c, 0x84, 0xc2, 0x8c, 0x0b, 0xb4, 0xbd, 0x4a, 0x57,
  0xa6, 0x0c, 0x4e, 0x9e, 0x73, 0x43, 0x5f, 0x2d, 0xaf, 0x22, 0xf4, 0xa6, 0xfd, 0x5a, 0x87, 0xb9,
  0xbe, 0x6f, 0x9e, 0x1d, 0x99, 0xc8, 0xdf, 0x95, 0xc2, 0x02, 0xad, 0x7d, 0x4a, 0xf3, 0xba, 0x5d,
  0xe8, 0x7b, 0x6e, 0x57, 0xe5, 0x3a, 0x02, 0xf8, 0xd3, 0x01, 0x15, 0xae, 0x4f, 0x02, 0x03, 0x01,
  0x00, 0x01, 0xa3, 0x82, 0x01, 0x4c, 0x30, 0x82, 0x01, 0x48, 0x30, 0x10, 0x06, 0x09, 0x2b, 0x06,
  0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x01, 0x04, 0x03, 0x02, 0x01, 0x00, 0x30, 0x1d, 0x06, 0x03,
  0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x4e, 0xfe, 0x61, 0x39, 0x6f, 0xca, 0xc5, 0x64, 0xe0,
  0x20, 0xb6, 0xda, 0xc1, 0xb9, 0x52, 0x0e, 0xca, 0xdf, 0x8e, 0xfe, 0x30, 0x19, 0x06, 0x09, 0x2b,
  0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x14, 0x02, 0x04, 0x0c, 0x1e, 0x0a, 0x00, 0x53, 0x00, 0x75,
  0x00, 0x62, 0x00, 0x43, 0x00, 0x41, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03,
  0x02, 0x01, 0x86, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30,
  0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18,
  0x30, 0x16, 0x80, 0x14, 0xa3, 0x01, 0x04, 0x7e, 0x30, 0x88, 0x33, 0xeb, 0xb9, 0x31, 0x9c, 0xca,
  0xeb, 0x85, 0x76, 0x67, 0xfc, 0x65, 0xb4, 0xd1, 0x30, 0x59, 0x06, 0x03, 0x55, 0x1d, 0x1f, 0x04,
  0x52, 0x30, 0x50, 0x30, 0x4e, 0xa0, 0x4c, 0xa0, 0x4a, 0x86, 0x48, 0x68, 0x74, 0x74, 0x70, 0x3a,
  0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x2e,
  0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x69, 0x2f, 0x63, 0x72, 0x6c, 0x2f, 0x70, 0x72, 0x6f, 0x64,
  0x75, 0x63, 0x74, 0x73, 0x2f, 0x4d, 0x69, 0x63, 0x54, 0x65, 0x73, 0x52, 0x6f, 0x6f, 0x43, 0x65,
  0x72, 0x41, 0x75, 0x74, 0x5f, 0x32, 0x30, 0x31, 0x30, 0x2d, 0x30, 0x36, 0x2d, 0x31, 0x37, 0x2e,
  0x63, 0x72, 0x6c, 0x30, 0x5d, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04,
  0x51, 0x30, 0x4f, 0x30, 0x4d, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86,
  0x41, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x6d, 0x69, 0x63, 0x72,
  0x6f, 0x73, 0x6f, 0x66, 0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x69, 0x2f, 0x63, 0x65,
  0x72, 0x74, 0x73, 0x2f, 0x4d, 0x69, 0x63, 0x54, 0x65, 0x73, 0x52, 0x6f, 0x6f, 0x43, 0x65, 0x72,
  0x41, 0x75, 0x74, 0x5f, 0x32, 0x30, 0x31, 0x30, 0x2d, 0x30, 0x36, 0x2d, 0x31, 0x37, 0x2e, 0x63,
  0x72, 0x74, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05,
  0x00, 0x03, 0x82, 0x02, 0x01, 0x00, 0x91, 0x2d, 0x11, 0x36, 0xd1, 0x2c, 0x5e, 0x21, 0x77, 0x62,
  0x1a, 0xa1, 0xcf, 0x9a, 0x7e, 0x62, 0x2d, 0xc9, 0x90, 0xb9, 0x0e, 0x3d, 0x71, 0x0d, 0xa0, 0x4a,
  0x48, 0xad, 0xe1, 0xd6, 0xb0, 0x73, 0x4a, 0x51, 0x4b, 0x4c, 0xb2, 0x6b, 0x88, 0x0a, 0x5a, 0xb7,
  0xe9, 0x55, 0x67, 0x43, 0x8b, 0xf1, 0x9d, 0xcc, 0x52, 0x03, 0xaf, 0xac, 0x85, 0xe9, 0xd3, 0x75,
  0xce, 0xe5, 0x6f, 0x8b, 0x3c, 0x80, 0x81, 0xab, 0xd1, 0xd1, 0x73, 0x0d, 0x6b, 0xb0, 0x51, 0xb8,
  0xa2, 0x5e, 0x0c, 0x09, 0xe5, 0xa6, 0xbc, 0x54, 0x49, 0xe5, 0x54, 0x38, 0x5a, 0xb7, 0x43, 0xd2,
  0x75, 0x34, 0x52, 0xdd, 0x35, 0x79, 0x95, 0x98, 0xd5, 0x3c, 0xc6, 0xda, 0x46, 0xb2, 0xd8, 0xa0,
  0xa7, 0xcc, 0x30, 0x31, 0x0f, 0x4b, 0xeb, 0xb7, 0x02, 0xab, 0x2b, 0x38, 0xd2, 0xb7, 0x27, 0xc2,
  0xc7, 0x0d, 0x62, 0xa4, 0xc0, 0x48, 0xf4, 0x1a, 0x48, 0x79, 0x7d, 0xe1, 0x31, 0xa4, 0x5d, 0x19,
  0xd0, 0x18, 0x17, 0x00, 0xca, 0xe3, 0x5d, 0x0b, 0xc1, 0x85, 0x7f, 0xf1, 0xbe, 0xd3, 0xcb, 0x97,
  0xf4, 0x1b, 0x62, 0x0f, 0xd7, 0x14, 0x38, 0x8d, 0xb2, 0x2e, 0x98, 0xd4, 0x41, 0x6a, 0xb0, 0xb5,
  0x84, 0xee, 0xd8, 0x2c, 0x2d, 0x13, 0xfc, 0x50, 0x1d, 0x9a, 0x6f, 0x31, 0x87, 0x1c, 0xe0, 0xd0,
  0xd0, 0xd3, 0x24, 0x52, 0xdf, 0x12, 0x8d, 0xfb, 0xf7, 0x7f, 0x33, 0x04, 0x17, 0x7a, 0x8b, 0xe3,
  0xbd, 0x7c, 0x1e, 0xa2, 0x2a, 0x91, 0x76, 0xd9, 0xfc, 0x43, 0x17, 0xa7, 0x0a, 0xa1, 0xd9, 0xc6,
  0x3c, 0x3a, 0x42, 0xed, 0xb7, 0x12, 0xcd, 0x9e, 0xdd, 0xfe, 0x91, 0x9d, 0x98, 0xb5, 0x35, 0xe8,
  0x94, 0xdd, 0x9a, 0xce, 0xd4, 0xf4, 0xac, 0xca, 0xa6, 0xcd, 0x59, 0x3d, 0xde, 0x44, 0xbf, 0x0e,
  0x82, 0xe5, 0xdf, 0x5c, 0xe3, 0x49, 0xcb, 0x01, 0xa0, 0x7f, 0x7b, 0xe2, 0x25, 0x21, 0xab, 0x95,
  0x50, 0x1e, 0x9f, 0xea, 0x77, 0x56, 0x47, 0x74, 0x73, 0xf2, 0x25, 0x1b, 0x0e, 0x36, 0xb9, 0xa2,
  0x57, 0xbf, 0x2c, 0x57, 0xbc, 0xe4, 0xfb, 0x88, 0xb1, 0x83, 0x56, 0xd8, 0x3c, 0xe0, 0x7c, 0xfd,
  0x11, 0x9f, 0xde, 0xe1, 0xb7, 0xd4, 0xc6, 0x23, 0xaf, 0xae, 0xc7, 0x5e, 0x4f, 0xc5, 0x23, 0x74,
  0x98, 0x38, 0x74, 0x9d, 0x0d, 0x0d, 0x6d, 0xf4, 0xde, 0x28, 0x44, 0x41, 0xd8, 0x22, 0x68, 0xe7,
  0x57, 0xa7, 0x66, 0x74, 0xb5, 0xc3, 0x31, 0x24, 0xe2, 0x7c, 0xd5, 0x52, 0x5d, 0x8a, 0xd2, 0x4e,
  0x14, 0xbb, 0x08, 0xd3, 0x1a, 0x31, 0x12, 0x04, 0x58, 0x9b, 0x3a, 0x99, 0x78, 0x93, 0x51, 0x3f,
  0x0b, 0x85, 0xce, 0xde, 0x09, 0x58, 0xf9, 0x96, 0x1b, 0x91, 0x11, 0xa0, 0x68, 0x53, 0x33, 0xcb,
  0xbe, 0x08, 0x89, 0xcc, 0x95, 0x55, 0xae, 0x06, 0x78, 0x27, 0x99, 0xa2, 0x56, 0xef, 0xae, 0x95,
  0x0a, 0x7f, 0x0d, 0xeb, 0x43, 0x0c, 0xc6, 0x9e, 0x3f, 0x80, 0xbb, 0xc4, 0x74, 0x03, 0x9d, 0xb8,
  0x1a, 0xaa, 0xbc, 0xcb, 0x1e, 0x03, 0xfb, 0xd8, 0x49, 0x89, 0x9d, 0x20, 0xaa, 0x25, 0x4b, 0x47,
  0x67, 0xf4, 0xa7, 0x0d, 0x8d, 0x76, 0x30, 0xc6, 0x48, 0xce, 0xdd, 0x8d, 0xfa, 0xbe, 0xda, 0xb4,
  0xeb, 0x3d, 0xd8, 0x0c, 0x3b, 0x15, 0x9a, 0xd1, 0x83, 0x85, 0xe5, 0xe3, 0x92, 0xaa, 0x3b, 0xad,
  0x34, 0x0c, 0x39, 0xf1, 0x7f, 0x75, 0xae, 0xe8, 0x70, 0xb0, 0xfd, 0xf2, 0x8e, 0x29, 0xda, 0xdc,
  0x09, 0x4f, 0x8a, 0x7b, 0x63, 0xbe, 0x1e, 0x8a, 0xc3, 0x58, 0xd1, 0x03, 0xbe, 0x37, 0xe9, 0xed,
  0x32, 0x26, 0xa8, 0x0c, 0xfe, 0xed, 0x12, 0xa7, 0x63, 0x04, 0xb3, 0x92, 0x97, 0xd9, 0x46, 0xa3,
  0x33, 0x33, 0x9c, 0x5b, 0xbc, 0x54, 0x30, 0x82, 0x06, 0x1d, 0x30, 0x82, 0x05, 0x05, 0xa0, 0x03,
  0x02, 0x01, 0x02, 0x02, 0x0a, 0x4e, 0x30, 0xa5, 0xbb, 0x00, 0x00, 0x00, 0x12, 0x7b, 0xf2, 0x30,
  0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x81,
  0x81, 0x31, 0x13, 0x30, 0x11, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01,
  0x19, 0x16, 0x03, 0x63, 0x6f, 0x6d, 0x31, 0x19, 0x30, 0x17, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89,
  0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x09, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66,
  0x74, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01,
  0x19, 0x16, 0x04, 0x63, 0x6f, 0x72, 0x70, 0x31, 0x17, 0x30, 0x15, 0x06, 0x0a, 0x09, 0x92, 0x26,
  0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x07, 0x72, 0x65, 0x64, 0x6d, 0x6f, 0x6e, 0x64,
  0x31, 0x20, 0x30, 0x1e, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x17, 0x4d, 0x53, 0x49, 0x54, 0x20,
  0x54, 0x65, 0x73, 0x74, 0x20, 0x43, 0x6f, 0x64, 0x65, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x41,
  0x20, 0x33, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x31, 0x30, 0x34, 0x32, 0x31, 0x30, 0x33, 0x33, 0x33,
  0x34, 0x39, 0x5a, 0x17, 0x0d, 0x31, 0x33, 0x30, 0x34, 0x32, 0x30, 0x30, 0x33, 0x33, 0x33, 0x34,
  0x39, 0x5a, 0x30, 0x70, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
  0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x57, 0x61, 0x73, 0x68,
  0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13,
  0x07, 0x52, 0x65, 0x64, 0x6d, 0x6f, 0x6e, 0x64, 0x31, 0x1e, 0x30, 0x1c, 0x06, 0x03, 0x55, 0x04,
  0x0a, 0x13, 0x15, 0x4d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x20, 0x43, 0x6f, 0x72,
  0x70, 0x6f, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04,
  0x03, 0x13, 0x11, 0x4d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x20, 0x57, 0x69, 0x6e,
  0x64, 0x6f, 0x77, 0x73, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
  0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a,
  0x02, 0x82, 0x01, 0x01, 0x00, 0xca, 0x9f, 0xcb, 0x8c, 0x5e, 0x61, 0x19, 0xb5, 0x37, 0x72, 0xd1,
  0xe2, 0x88, 0x1f, 0x37, 0x56, 0xce, 0xa8, 0x88, 0x05, 0x3e, 0xe8, 0xa6, 0x8d, 0xfd, 0x0b, 0x30,
  0x63, 0x54, 0x89, 0x1d, 0xda, 0x11, 0x70, 0x5a, 0x06, 0x00, 0x86, 0xd7, 0x7f, 0xe6, 0x21, 0x70,
  0x41, 0x81, 0x26, 0x4b, 0x6e, 0xc7, 0x02, 0xe2, 0xbf, 0x61, 0x45, 0x04, 0x2f, 0x45, 0xbc, 0xd2,
  0x0d, 0xf6, 0xac, 0xda, 0x91, 0x6d, 0xc6, 0x60, 0xad, 0xb3, 0x0b, 0x32, 0xe5, 0x13, 0x95, 0x96,
  0x02, 0x4e, 0x10, 0xad, 0x2e, 0x6c, 0xb0, 0xc7, 0x65, 0x6d, 0xea, 0x89, 0x91, 0x67, 0x73, 0x5d,
  0x05, 0x12, 0x9f, 0x95, 0x74, 0x6a, 0x6d, 0xb0, 0x6a, 0xd8, 0x94, 0x9b, 0xb0, 0x70, 0x79, 0xb0,
  0x85, 0x42, 0xe7, 0xd7, 0x19, 0xc4, 0x33, 0xfd, 0xac, 0xbe, 0x2b, 0xd5, 0x7e, 0x5a, 0xa9, 0xc7,
  0xa7, 0x68, 0x53, 0xed, 0xaa, 0x51, 0x2a, 0x6b, 0xac, 0x5a, 0xd7, 0x6e, 0xcf, 0x68, 0x60, 0x90,
  0xc7, 0xda, 0xd1, 0xe7, 0xf2, 0x53, 0x4a, 0x01, 0xaf, 0xe8, 0xb1, 0x29, 0x91, 0xab, 0x4c, 0x58,
  0x64, 0xe6, 0xb6, 0xcc, 0xdc, 0x0f, 0x43, 0xf2, 0x87, 0x15, 0xf8, 0x73, 0x06, 0xb8, 0x1f, 0xaa,
  0xf2, 0x66, 0x49, 0x82, 0xca, 0xf0, 0xdd, 0x4c, 0x7c, 0xd5, 0xe7, 0x59, 0xb7, 0xc6, 0xd3, 0x81,
  0x29, 0x5c, 0xce, 0xe4, 0xfd, 0xf7, 0xeb, 0x58, 0x6b, 0x50, 0x79, 0x72, 0x83, 0x4d, 0x59, 0xec,
  0xd1, 0xd2, 0xfc, 0x3a, 0x10, 0xc5, 0x7a, 0xba, 0x90, 0xea, 0xfd, 0x54, 0xc7, 0xc8, 0x58, 0x2d,
  0x84, 0xe5, 0x5b, 0x51, 0x98, 0x2c, 0x36, 0x7c, 0x42, 0x2b, 0xb1, 0x62, 0x4b, 0x0f, 0xf9, 0x72,
  0xd3, 0x3f, 0x99, 0xc8, 0xb4, 0x02, 0x17, 0x39, 0xe5, 0x0b, 0xf2, 0x5d, 0x2d, 0x7f, 0x23, 0x41,
  0xcd, 0x58, 0x3a, 0x25, 0x75, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x02, 0xa5, 0x30, 0x82,
  0x02, 0xa1, 0x30, 0x3d, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x07, 0x04,
  0x30, 0x30, 0x2e, 0x06, 0x26, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x08, 0x83, 0xcf,
  0x89, 0x4d, 0xad, 0xf2, 0x02, 0x85, 0xa1, 0x9f, 0x0c, 0x82, 0xfa, 0xca, 0x7d, 0x81, 0xe1, 0xfb,
  0x74, 0x81, 0x4f, 0x84, 0x9a, 0xf1, 0x69, 0x85, 0xa1, 0xad, 0x1d, 0x02, 0x01, 0x64, 0x02, 0x01,
  0x0d, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x29,
  0x06, 0x09, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x0a, 0x04, 0x1c, 0x30, 0x1a, 0x30,
  0x0c, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x0a, 0x03, 0x06, 0x30, 0x0a, 0x06,
  0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x03, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x25,
  0x04, 0x18, 0x30, 0x16, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x0a, 0x03, 0x06,
  0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x03, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d,
  0x0e, 0x04, 0x16, 0x04, 0x14, 0xc9, 0x5a, 0xb3, 0x11, 0x02, 0x66, 0x71, 0x8c, 0x5b, 0x06, 0x80,
  0x8a, 0x6d, 0x4f, 0x5b, 0xf2, 0xb5, 0x67, 0x45, 0xb5, 0x30, 0x30, 0x06, 0x03, 0x55, 0x1d, 0x11,
  0x04, 0x29, 0x30, 0x27, 0xa0, 0x25, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x14,
  0x02, 0x03, 0xa0, 0x17, 0x0c, 0x15, 0x77, 0x69, 0x6e, 0x63, 0x62, 0x6c, 0x64, 0x40, 0x6d, 0x69,
  0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1f, 0x06, 0x03, 0x55,
  0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x4e, 0xfe, 0x61, 0x39, 0x6f, 0xca, 0xc5, 0x64,
  0xe0, 0x20, 0xb6, 0xda, 0xc1, 0xb9, 0x52, 0x0e, 0xca, 0xdf, 0x8e, 0xfe, 0x30, 0x81, 0xe8, 0x06,
  0x03, 0x55, 0x1d, 0x1f, 0x04, 0x81, 0xe0, 0x30, 0x81, 0xdd, 0x30, 0x81, 0xda, 0xa0, 0x81, 0xd7,
  0xa0, 0x81, 0xd4, 0x86, 0x36, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x6f, 0x72, 0x70,
  0x70, 0x6b, 0x69, 0x2f, 0x63, 0x72, 0x6c, 0x2f, 0x4d, 0x53, 0x49, 0x54, 0x25, 0x32, 0x30, 0x54,
  0x65, 0x73, 0x74, 0x25, 0x32, 0x30, 0x43, 0x6f, 0x64, 0x65, 0x53, 0x69, 0x67, 0x6e, 0x25, 0x32,
  0x30, 0x43, 0x41, 0x25, 0x32, 0x30, 0x33, 0x2e, 0x63, 0x72, 0x6c, 0x86, 0x4d, 0x68, 0x74, 0x74,
  0x70, 0x3a, 0x2f, 0x2f, 0x6d, 0x73, 0x63, 0x72, 0x6c, 0x2e, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73,
  0x6f, 0x66, 0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x69, 0x2f, 0x6d, 0x73, 0x63, 0x6f,
  0x72, 0x70, 0x2f, 0x63, 0x72, 0x6c, 0x2f, 0x4d, 0x53, 0x49, 0x54, 0x25, 0x32, 0x30, 0x54, 0x65,
  0x73, 0x74, 0x25, 0x32, 0x30, 0x43, 0x6f, 0x64, 0x65, 0x53, 0x69, 0x67, 0x6e, 0x25, 0x32, 0x30,
  0x43, 0x41, 0x25, 0x32, 0x30, 0x33, 0x2e, 0x63, 0x72, 0x6c, 0x86, 0x4b, 0x68, 0x74, 0x74, 0x70,
  0x3a, 0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74,
  0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x69, 0x2f, 0x6d, 0x73, 0x63, 0x6f, 0x72, 0x70, 0x2f,
  0x63, 0x72, 0x6c, 0x2f, 0x4d, 0x53, 0x49, 0x54, 0x25, 0x32, 0x30, 0x54, 0x65, 0x73, 0x74, 0x25,
  0x32, 0x30, 0x43, 0x6f, 0x64, 0x65, 0x53, 0x69, 0x67, 0x6e, 0x25, 0x32, 0x30, 0x43, 0x41, 0x25,
  0x32, 0x30, 0x33, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x81, 0xa9, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05,
  0x05, 0x07, 0x01, 0x01, 0x04, 0x81, 0x9c, 0x30, 0x81, 0x99, 0x30, 0x42, 0x06, 0x08, 0x2b, 0x06,
  0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x36, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63,
  0x6f, 0x72, 0x70, 0x70, 0x6b, 0x69, 0x2f, 0x61, 0x69, 0x61, 0x2f, 0x4d, 0x53, 0x49, 0x54, 0x25,
  0x32, 0x30, 0x54, 0x65, 0x73, 0x74, 0x25, 0x32, 0x30, 0x43, 0x6f, 0x64, 0x65, 0x53, 0x69, 0x67,
  0x6e, 0x25, 0x32, 0x30, 0x43, 0x41, 0x25, 0x32, 0x30, 0x33, 0x2e, 0x63, 0x72, 0x74, 0x30, 0x53,
  0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x47, 0x68, 0x74, 0x74, 0x70,
  0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74,
  0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x70, 0x6b, 0x69, 0x2f, 0x6d, 0x73, 0x63, 0x6f, 0x72, 0x70, 0x2f,
  0x4d, 0x53, 0x49, 0x54, 0x25, 0x32, 0x30, 0x54, 0x65, 0x73, 0x74, 0x25, 0x32, 0x30, 0x43, 0x6f,
  0x64, 0x65, 0x53, 0x69, 0x67, 0x6e, 0x25, 0x32, 0x30, 0x43, 0x41, 0x25, 0x32, 0x30, 0x33, 0x2e,
  0x63, 0x72, 0x74, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b,
  0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x97, 0x6d, 0x52, 0x46, 0x65, 0x08, 0x60, 0x01, 0x12,
  0x7a, 0x26, 0xaa, 0xd9, 0xc3, 0x15, 0xec, 0xc6, 0xae, 0x0b, 0x75, 0xe4, 0xd5, 0xe0, 0x31, 0xf6,
  0xda, 0xc0, 0xb1, 0x1b, 0x7b, 0x8a, 0x07, 0xda, 0x2f, 0x82, 0x4a, 0x21, 0x89, 0xe6, 0x61, 0x97,
  0xe4, 0x3a, 0x38, 0xd8, 0x0a, 0x26, 0x4e, 0x83, 0x54, 0xbf, 0x32, 0xe3, 0x64, 0x36, 0x61, 0x40,
  0x75, 0x55, 0xf1, 0xaa, 0x34, 0xb2, 0xd3, 0x45, 0x8d, 0x79, 0x9a, 0x93, 0x0b, 0x9f, 0x5a, 0xb3,
  0x5b, 0xcf, 0x0d, 0x28, 0x00, 0x46, 0x68, 0xe7, 0x36, 0xc9, 0xe6, 0xd2, 0x03, 0x2d, 0x1c, 0x2b,
  0x24, 0x79, 0x32, 0xfb, 0xd8, 0x43, 0xb4, 0x0e, 0xd0, 0x37, 0x3b, 0x5e, 0x15, 0xbf, 0x8a, 0x00,
  0x56, 0x00, 0x95, 0x00, 0xd4, 0x71, 0x09, 0xe1, 0x84, 0xfa, 0x6e, 0xc5, 0x75, 0x2f, 0xe1, 0x83,
  0x7b, 0x66, 0xe2, 0x22, 0x75, 0x23, 0x6a, 0xf3, 0x83, 0xfc, 0x94, 0xf8, 0x51, 0x6d, 0x40, 0x6d,
  0x20, 0x7e, 0xb0, 0xe9, 0xac, 0xa8, 0x28, 0x04, 0xa0, 0x1a, 0x3c, 0x88, 0x7e, 0x7e, 0x3b, 0x96,
  0xe9, 0xd0, 0x4b, 0xfa, 0xfa, 0x34, 0x4a, 0x4f, 0xe7, 0xe8, 0x76, 0x16, 0x20, 0xc0, 0x10, 0xca,
  0x1f, 0x7a, 0xde, 0x54, 0xfd, 0xb7, 0xf0, 0xea, 0x8f, 0xe2, 0xcb, 0xa5, 0x1c, 0x58, 0x18, 0x57,
  0x6c, 0x7d, 0x06, 0xde, 0x8c, 0xbd, 0x45, 0xd7, 0x28, 0x86, 0x13, 0x42, 0xaa, 0x4a, 0x9a, 0x50,
  0xc5, 0x76, 0x7f, 0x63, 0x99, 0xd0, 0x90, 0x92, 0xd6, 0xc9, 0x21, 0xf2, 0xb8, 0x66, 0x9c, 0x96,
  0x53, 0xd2, 0xe5, 0xc4, 0xdd, 0x3a, 0x7c, 0x06, 0xae, 0xb1, 0x7a, 0xb4, 0x36, 0x04, 0x9d, 0xe6,
  0x2a, 0xe4, 0x6a, 0x56, 0xd0, 0x9a, 0x98, 0xa2, 0xbf, 0x6f, 0x43, 0xf2, 0xb0, 0x5b, 0xfd, 0xd9,
  0xfa, 0x08, 0x17, 0xc8, 0x3c, 0x61, 0xf1, 0x31, 0x82, 0x02, 0x3a, 0x30, 0x82, 0x02, 0x36, 0x02,
  0x01, 0x01, 0x30, 0x81, 0x90, 0x30, 0x81, 0x81, 0x31, 0x13, 0x30, 0x11, 0x06, 0x0a, 0x09, 0x92,
  0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x03, 0x63, 0x6f, 0x6d, 0x31, 0x19, 0x30,
  0x17, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x09, 0x6d,
  0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x09, 0x92,
  0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x04, 0x63, 0x6f, 0x72, 0x70, 0x31, 0x17,
  0x30, 0x15, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x07,
  0x72, 0x65, 0x64, 0x6d, 0x6f, 0x6e, 0x64, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x03, 0x55, 0x04, 0x03,
  0x13, 0x17, 0x4d, 0x53, 0x49, 0x54, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x43, 0x6f, 0x64, 0x65,
  0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x41, 0x20, 0x33, 0x02, 0x0a, 0x4e, 0x30, 0xa5, 0xbb, 0x00,
  0x00, 0x00, 0x12, 0x7b, 0xf2, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04,
  0x02, 0x01, 0x05, 0x00, 0xa0, 0x7c, 0x30, 0x10, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82,
  0x37, 0x02, 0x01, 0x0c, 0x31, 0x02, 0x30, 0x00, 0x30, 0x19, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
  0xf7, 0x0d, 0x01, 0x09, 0x03, 0x31, 0x0c, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37,
  0x02, 0x01, 0x04, 0x30, 0x1c, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x02, 0x01,
  0x0b, 0x31, 0x0e, 0x30, 0x0c, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x02, 0x01,
  0x15, 0x30, 0x2f, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x04, 0x31, 0x22,
  0x04, 0x20, 0x25, 0x60, 0x6e, 0xc8, 0x66, 0x72, 0x73, 0xb0, 0x71, 0x7b, 0x82, 0x09, 0x8c, 0x27,
  0xe8, 0xb1, 0x4a, 0x36, 0x1c, 0x81, 0x36, 0xb4, 0x1b, 0x44, 0xdf, 0x09, 0x82, 0x8b, 0xf9, 0xb6,
  0x1c, 0x65, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05,
  0x00, 0x04, 0x82, 0x01, 0x00, 0x01, 0xb7, 0x8e, 0x46, 0x9b, 0xb4, 0x18, 0xcd, 0xf6, 0x39, 0x8f,
  0x82, 0x79, 0xbc, 0x96, 0x55, 0x7e, 0xd7, 0x84, 0x8f, 0x18, 0x37, 0x48, 0x4e, 0x8f, 0x25, 0x6f,
  0x1a, 0x6c, 0x60, 0xe9, 0xc1, 0x5f, 0x65, 0xec, 0xb0, 0x4b, 0x35, 0x98, 0x8a, 0x31, 0x6b, 0xfd,
  0xa9, 0xd2, 0x28, 0xa7, 0x00, 0xd0, 0x5c, 0x97, 0x2e, 0x3d, 0x19, 0x96, 0x75, 0xc2, 0x9a, 0xa7,
  0x4e, 0x2d, 0x3c, 0x6a, 0xd7, 0xd1, 0x81, 0x96, 0x2e, 0x06, 0x54, 0x82, 0xe9, 0x87, 0xa2, 0x53,
  0x63, 0x6a, 0xda, 0x72, 0xa3, 0x41, 0x4f, 0x14, 0xc5, 0x3d, 0x41, 0x7d, 0xbb, 0xd7, 0x29, 0x11,
  0x96, 0x14, 0x60, 0x5f, 0xb9, 0xd3, 0x47, 0xb6, 0x23, 0x97, 0x64, 0xf6, 0xe1, 0xac, 0x6c, 0x4d,
  0xa9, 0x58, 0xc2, 0x35, 0xbf, 0x02, 0xee, 0x29, 0xd4, 0x52, 0x80, 0x25, 0xa1, 0x7e, 0xbe, 0x21,
  0xcd, 0x04, 0x3c, 0x4d, 0x7d, 0xdb, 0x9b, 0x13, 0x71, 0x35, 0x13, 0x8d, 0x16, 0x0a, 0x40, 0x7b,
  0x63, 0x45, 0x00, 0x11, 0xc5, 0x3e, 0xc8, 0x21, 0x37, 0xc9, 0xbc, 0xd2, 0x0f, 0x82, 0x21, 0xf9,
  0x2f, 0xde, 0xb3, 0x1c, 0x9b, 0x67, 0xb6, 0x5a, 0xb8, 0x74, 0xa0, 0x30, 0xd8, 0x86, 0x1f, 0xd9,
  0x08, 0x45, 0xf9, 0x6b, 0xe8, 0x57, 0xda, 0xb7, 0x14, 0xf7, 0x63, 0x97, 0xfd, 0x87, 0x61, 0x1e,
  0x95, 0x6f, 0xba, 0x53, 0x40, 0x5e, 0x9f, 0x49, 0xf6, 0xe7, 0x3c, 0x59, 0xd9, 0x78, 0x08, 0x4c,
  0xa6, 0x4a, 0xa2, 0x05, 0x42, 0xa7, 0x77, 0xbc, 0x44, 0x7b, 0xed, 0xdb, 0x77, 0x63, 0x4c, 0x63,
  0x22, 0xb6, 0x17, 0x3f, 0x4f, 0x33, 0xed, 0xa6, 0xd9, 0x0f, 0x74, 0xfa, 0x97, 0x8d, 0x98, 0x55,
  0x42, 0x14, 0x76, 0xd4, 0x89, 0x07, 0xf2, 0x30, 0x5a, 0x27, 0x96, 0x1b, 0x92, 0x70, 0x5a, 0x83,
  0x89, 0x0f, 0xdb, 0x47, 0x7a, 0x00, 0x00, 0x00, 0x0a
  };

//
// Sample root certificate for code signing.
// A trusted certificate with self-signing, will be used to construct the certificate chains for
// authority checking.
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT8 TestRootCert1[] = {
  0x30, 0x82, 0x05, 0x99, 0x30, 0x82, 0x03, 0x81, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x79,
  0xAD, 0x16, 0xA1, 0x4A, 0xA0, 0xA5, 0xAD, 0x4C, 0x73, 0x58, 0xF4, 0x07, 0x13, 0x2E, 0x65, 0x30,
  0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x5F,
  0x31, 0x13, 0x30, 0x11, 0x06, 0x0A, 0x09, 0x92, 0x26, 0x89, 0x93, 0xF2, 0x2C, 0x64, 0x01, 0x19,
  0x16, 0x03, 0x63, 0x6F, 0x6D, 0x31, 0x19, 0x30, 0x17, 0x06, 0x0A, 0x09, 0x92, 0x26, 0x89, 0x93,
  0xF2, 0x2C, 0x64, 0x01, 0x19, 0x16, 0x09, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74,
  0x31, 0x2D, 0x30, 0x2B, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x24, 0x4D, 0x69, 0x63, 0x72, 0x6F,
  0x73, 0x6F, 0x66, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66,
  0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x30,
  0x1E, 0x17, 0x0D, 0x30, 0x31, 0x30, 0x35, 0x30, 0x39, 0x32, 0x33, 0x31, 0x39, 0x32, 0x32, 0x5A,
  0x17, 0x0D, 0x32, 0x31, 0x30, 0x35, 0x30, 0x39, 0x32, 0x33, 0x32, 0x38, 0x31, 0x33, 0x5A, 0x30,
  0x5F, 0x31, 0x13, 0x30, 0x11, 0x06, 0x0A, 0x09, 0x92, 0x26, 0x89, 0x93, 0xF2, 0x2C, 0x64, 0x01,
  0x19, 0x16, 0x03, 0x63, 0x6F, 0x6D, 0x31, 0x19, 0x30, 0x17, 0x06, 0x0A, 0x09, 0x92, 0x26, 0x89,
  0x93, 0xF2, 0x2C, 0x64, 0x01, 0x19, 0x16, 0x09, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66,
  0x74, 0x31, 0x2D, 0x30, 0x2B, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x24, 0x4D, 0x69, 0x63, 0x72,
  0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69,
  0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79,
  0x30, 0x82, 0x02, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01,
  0x01, 0x05, 0x00, 0x03, 0x82, 0x02, 0x0F, 0x00, 0x30, 0x82, 0x02, 0x0A, 0x02, 0x82, 0x02, 0x01,
  0x00, 0xF3, 0x5D, 0xFA, 0x80, 0x67, 0xD4, 0x5A, 0xA7, 0xA9, 0x0C, 0x2C, 0x90, 0x20, 0xD0, 0x35,
  0x08, 0x3C, 0x75, 0x84, 0xCD, 0xB7, 0x07, 0x89, 0x9C, 0x89, 0xDA, 0xDE, 0xCE, 0xC3, 0x60, 0xFA,
  0x91, 0x68, 0x5A, 0x9E, 0x94, 0x71, 0x29, 0x18, 0x76, 0x7C, 0xC2, 0xE0, 0xC8, 0x25, 0x76, 0x94,
  0x0E, 0x58, 0xFA, 0x04, 0x34, 0x36, 0xE6, 0xDF, 0xAF, 0xF7, 0x80, 0xBA, 0xE9, 0x58, 0x0B, 0x2B,
  0x93, 0xE5, 0x9D, 0x05, 0xE3, 0x77, 0x22, 0x91, 0xF7, 0x34, 0x64, 0x3C, 0x22, 0x91, 0x1D, 0x5E,
  0xE1, 0x09, 0x90, 0xBC, 0x14, 0xFE, 0xFC, 0x75, 0x58, 0x19, 0xE1, 0x79, 0xB7, 0x07, 0x92, 0xA3,
  0xAE, 0x88, 0x59, 0x08, 0xD8, 0x9F, 0x07, 0xCA, 0x03, 0x58, 0xFC, 0x68, 0x29, 0x6D, 0x32, 0xD7,
  0xD2, 0xA8, 0xCB, 0x4B, 0xFC, 0xE1, 0x0B, 0x48, 0x32, 0x4F, 0xE6, 0xEB, 0xB8, 0xAD, 0x4F, 0xE4,
  0x5C, 0x6F, 0x13, 0x94, 0x99, 0xDB, 0x95, 0xD5, 0x75, 0xDB, 0xA8, 0x1A, 0xB7, 0x94, 0x91, 0xB4,
  0x77, 0x5B, 0xF5, 0x48, 0x0C, 0x8F, 0x6A, 0x79, 0x7D, 0x14, 0x70, 0x04, 0x7D, 0x6D, 0xAF, 0x90,
  0xF5, 0xDA, 0x70, 0xD8, 0x47, 0xB7, 0xBF, 0x9B, 0x2F, 0x6C, 0xE7, 0x05, 0xB7, 0xE1, 0x11, 0x60,
  0xAC, 0x79, 0x91, 0x14, 0x7C, 0xC5, 0xD6, 0xA6, 0xE4, 0xE1, 0x7E, 0xD5, 0xC3, 0x7E, 0xE5, 0x92,
  0xD2, 0x3C, 0x00, 0xB5, 0x36, 0x82, 0xDE, 0x79, 0xE1, 0x6D, 0xF3, 0xB5, 0x6E, 0xF8, 0x9F, 0x33,
  0xC9, 0xCB, 0x52, 0x7D, 0x73, 0x98, 0x36, 0xDB, 0x8B, 0xA1, 0x6B, 0xA2, 0x95, 0x97, 0x9B, 0xA3,
  0xDE, 0xC2, 0x4D, 0x26, 0xFF, 0x06, 0x96, 0x67, 0x25, 0x06, 0xC8, 0xE7, 0xAC, 0xE4, 0xEE, 0x12,
  0x33, 0x95, 0x31, 0x99, 0xC8, 0x35, 0x08, 0x4E, 0x34, 0xCA, 0x79, 0x53, 0xD5, 0xB5, 0xBE, 0x63,
  0x32, 0x59, 0x40, 0x36, 0xC0, 0xA5, 0x4E, 0x04, 0x4D, 0x3D, 0xDB, 0x5B, 0x07, 0x33, 0xE4, 0x58,
  0xBF, 0xEF, 0x3F, 0x53, 0x64, 0xD8, 0x42, 0x59, 0x35, 0x57, 0xFD, 0x0F, 0x45, 0x7C, 0x24, 0x04,
  0x4D, 0x9E, 0xD6, 0x38, 0x74, 0x11, 0x97, 0x22, 0x90, 0xCE, 0x68, 0x44, 0x74, 0x92, 0x6F, 0xD5,
  0x4B, 0x6F, 0xB0, 0x86, 0xE3, 0xC7, 0x36, 0x42, 0xA0, 0xD0, 0xFC, 0xC1, 0xC0, 0x5A, 0xF9, 0xA3,
  0x61, 0xB9, 0x30, 0x47, 0x71, 0x96, 0x0A, 0x16, 0xB0, 0x91, 0xC0, 0x42, 0x95, 0xEF, 0x10, 0x7F,
  0x28, 0x6A, 0xE3, 0x2A, 0x1F, 0xB1, 0xE4, 0xCD, 0x03, 0x3F, 0x77, 0x71, 0x04, 0xC7, 0x20, 0xFC,
  0x49, 0x0F, 0x1D, 0x45, 0x88, 0xA4, 0xD7, 0xCB, 0x7E, 0x88, 0xAD, 0x8E, 0x2D, 0xEC, 0x45, 0xDB,
  0xC4, 0x51, 0x04, 0xC9, 0x2A, 0xFC, 0xEC, 0x86, 0x9E, 0x9A, 0x11, 0x97, 0x5B, 0xDE, 0xCE, 0x53,
  0x88, 0xE6, 0xE2, 0xB7, 0xFD, 0xAC, 0x95, 0xC2, 0x28, 0x40, 0xDB, 0xEF, 0x04, 0x90, 0xDF, 0x81,
  0x33, 0x39, 0xD9, 0xB2, 0x45, 0xA5, 0x23, 0x87, 0x06, 0xA5, 0x55, 0x89, 0x31, 0xBB, 0x06, 0x2D,
  0x60, 0x0E, 0x41, 0x18, 0x7D, 0x1F, 0x2E, 0xB5, 0x97, 0xCB, 0x11, 0xEB, 0x15, 0xD5, 0x24, 0xA5,
  0x94, 0xEF, 0x15, 0x14, 0x89, 0xFD, 0x4B, 0x73, 0xFA, 0x32, 0x5B, 0xFC, 0xD1, 0x33, 0x00, 0xF9,
  0x59, 0x62, 0x70, 0x07, 0x32, 0xEA, 0x2E, 0xAB, 0x40, 0x2D, 0x7B, 0xCA, 0xDD, 0x21, 0x67, 0x1B,
  0x30, 0x99, 0x8F, 0x16, 0xAA, 0x23, 0xA8, 0x41, 0xD1, 0xB0, 0x6E, 0x11, 0x9B, 0x36, 0xC4, 0xDE,
  0x40, 0x74, 0x9C, 0xE1, 0x58, 0x65, 0xC1, 0x60, 0x1E, 0x7A, 0x5B, 0x38, 0xC8, 0x8F, 0xBB, 0x04,
  0x26, 0x7C, 0xD4, 0x16, 0x40, 0xE5, 0xB6, 0x6B, 0x6C, 0xAA, 0x86, 0xFD, 0x00, 0xBF, 0xCE, 0xC1,
  0x35, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x51, 0x30, 0x4F, 0x30, 0x0B, 0x06, 0x03, 0x55, 0x1D,
  0x0F, 0x04, 0x04, 0x03, 0x02, 0x01, 0xC6, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01,
  0xFF, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04,
  0x16, 0x04, 0x14, 0x0E, 0xAC, 0x82, 0x60, 0x40, 0x56, 0x27, 0x97, 0xE5, 0x25, 0x13, 0xFC, 0x2A,
  0xE1, 0x0A, 0x53, 0x95, 0x59, 0xE4, 0xA4, 0x30, 0x10, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01,
  0x82, 0x37, 0x15, 0x01, 0x04, 0x03, 0x02, 0x01, 0x00, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48,
  0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x02, 0x01, 0x00, 0xC5, 0x11, 0x4D,
  0x03, 0x3A, 0x60, 0xDD, 0x5D, 0x52, 0x11, 0x77, 0x8F, 0xB2, 0xBB, 0x36, 0xC8, 0xB2, 0x05, 0xBF,
  0xB4, 0xB7, 0xA8, 0xD8, 0x20, 0x9D, 0x5C, 0x13, 0x03, 0xB6, 0x1C, 0x22, 0xFA, 0x06, 0x13, 0x35,
  0xB6, 0xC8, 0x63, 0xD4, 0x9A, 0x47, 0x6F, 0x26, 0x57, 0xD2, 0x55, 0xF1, 0x04, 0xB1, 0x26, 0x5F,
  0xD6, 0xA9, 0x50, 0x68, 0xA0, 0xBC, 0xD2, 0xB8, 0x6E, 0xCC, 0xC3, 0xE9, 0xAC, 0xDF, 0x19, 0xCD,
  0x78, 0xAC, 0x59, 0x74, 0xAC, 0x66, 0x34, 0x36, 0xC4, 0x1B, 0x3E, 0x6C, 0x38, 0x4C, 0x33, 0x0E,
  0x30, 0x12, 0x0D, 0xA3, 0x26, 0xFE, 0x51, 0x53, 0x00, 0xFF, 0xAF, 0x5A, 0x4E, 0x84, 0x0D, 0x0F,
  0x1F, 0xE4, 0x6D, 0x05, 0x2E, 0x4E, 0x85, 0x4B, 0x8D, 0x6C, 0x33, 0x6F, 0x54, 0xD2, 0x64, 0xAB,
  0xBF, 0x50, 0xAF, 0x7D, 0x7A, 0x39, 0xA0, 0x37, 0xED, 0x63, 0x03, 0x0F, 0xFC, 0x13, 0x06, 0xCE,
  0x16, 0x36, 0xD4, 0x54, 0x3B, 0x95, 0x1B, 0x51, 0x62, 0x3A, 0xE5, 0x4D, 0x17, 0xD4, 0x05, 0x39,
  0x92, 0x9A, 0x27, 0xA8, 0x5B, 0xAA, 0xBD, 0xEC, 0xBB, 0xBE, 0xE3, 0x20, 0x89, 0x60, 0x71, 0x6C,
  0x56, 0xB3, 0xA5, 0x13, 0xD0, 0x6D, 0x0E, 0x23, 0x7E, 0x95, 0x03, 0xED, 0x68, 0x3D, 0xF2, 0xD8,
  0x63, 0xB8, 0x6B, 0x4D, 0xB6, 0xE8, 0x30, 0xB5, 0xE1, 0xCA, 0x94, 0x4B, 0xF7, 0xA2, 0xAA, 0x5D,
  0x99, 0x30, 0xB2, 0x3D, 0xA7, 0xC2, 0x51, 0x6C, 0x28, 0x20, 0x01, 0x24, 0x27, 0x2B, 0x4B, 0x00,
  0xB7, 0x9D, 0x11, 0x6B, 0x70, 0xBE, 0xB2, 0x10, 0x82, 0xBC, 0x0C, 0x9B, 0x68, 0xD0, 0x8D, 0x3B,
  0x24, 0x87, 0xAA, 0x99, 0x28, 0x72, 0x9D, 0x33, 0x5F, 0x59, 0x90, 0xBD, 0xF5, 0xDE, 0x93, 0x9E,
  0x3A, 0x62, 0x5A, 0x34, 0x39, 0xE2, 0x88, 0x55, 0x1D, 0xB9, 0x06, 0xB0, 0xC1, 0x89, 0x6B, 0x2D,
  0xD7, 0x69, 0xC3, 0x19, 0x12, 0x36, 0x84, 0xD0, 0xC9, 0xA0, 0xDA, 0xFF, 0x2F, 0x69, 0x78, 0xB2,
  0xE5, 0x7A, 0xDA, 0xEB, 0xD7, 0x0C, 0xC0, 0xF7, 0xBD, 0x63, 0x17, 0xB8, 0x39, 0x13, 0x38, 0xA2,
  0x36, 0x5B, 0x7B, 0xF2, 0x85, 0x56, 0x6A, 0x1D, 0x64, 0x62, 0xC1, 0x38, 0xE2, 0xAA, 0xBF, 0x51,
  0x66, 0xA2, 0x94, 0xF5, 0x12, 0x9C, 0x66, 0x22, 0x10, 0x6B, 0xF2, 0xB7, 0x30, 0x92, 0x2D, 0xF2,
  0x29, 0xF0, 0x3D, 0x3B, 0x14, 0x43, 0x68, 0xA2, 0xF1, 0x9C, 0x29, 0x37, 0xCB, 0xCE, 0x38, 0x20,
  0x25, 0x6D, 0x7C, 0x67, 0xF3, 0x7E, 0x24, 0x12, 0x24, 0x03, 0x08, 0x81, 0x47, 0xEC, 0xA5, 0x9E,
  0x97, 0xF5, 0x18, 0xD7, 0xCF, 0xBB, 0xD5, 0xEF, 0x76, 0x96, 0xEF, 0xFD, 0xCE, 0xDB, 0x56, 0x9D,
  0x95, 0xA0, 0x42, 0xF9, 0x97, 0x58, 0xE1, 0xD7, 0x31, 0x22, 0xD3, 0x5F, 0x59, 0xE6, 0x3E, 0x6E,
  0x22, 0x00, 0xEA, 0x43, 0x84, 0xB6, 0x25, 0xDB, 0xD9, 0xF3, 0x08, 0x56, 0x68, 0xC0, 0x64, 0x6B,
  0x1D, 0x7C, 0xEC, 0xB6, 0x93, 0xA2, 0x62, 0x57, 0x6E, 0x2E, 0xD8, 0xE7, 0x58, 0x8F, 0xC4, 0x31,
  0x49, 0x26, 0xDD, 0xDE, 0x29, 0x35, 0x87, 0xF5, 0x30, 0x71, 0x70, 0x5B, 0x14, 0x3C, 0x69, 0xBD,
  0x89, 0x12, 0x7D, 0xEB, 0x2E, 0xA3, 0xFE, 0xD8, 0x7F, 0x9E, 0x82, 0x5A, 0x52, 0x0A, 0x2B, 0xC1,
  0x43, 0x2B, 0xD9, 0x30, 0x88, 0x9F, 0xC8, 0x10, 0xFB, 0x89, 0x8D, 0xE6, 0xA1, 0x85, 0x75, 0x33,
  0x7E, 0x6C, 0x9E, 0xDB, 0x73, 0x13, 0x64, 0x62, 0x69, 0xA5, 0x2F, 0x7D, 0xCA, 0x96, 0x6D, 0x9F,
  0xF8, 0x04, 0x4D, 0x30, 0x92, 0x3D, 0x6E, 0x21, 0x14, 0x21, 0xC9, 0x3D, 0xE0, 0xC3, 0xFD, 0x8A,
  0x6B, 0x9D, 0x4A, 0xFD, 0xD1, 0xA1, 0x9D, 0x99, 0x43, 0x77, 0x3F, 0xB0, 0xDA
};

GLOBAL_REMOVE_IF_UNREFERENCED UINT8 TestRootCert2[] = {
  0x30, 0x82, 0x05, 0xfd, 0x30, 0x82, 0x03, 0xe5, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x74,
  0x45, 0xc8, 0x78, 0x4e, 0x0c, 0xc9, 0x96, 0x4a, 0xb4, 0x2f, 0xbc, 0xda, 0x29, 0xe1, 0xbc, 0x30,
  0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x81,
  0x90, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13,
  0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67,
  0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65,
  0x64, 0x6d, 0x6f, 0x6e, 0x64, 0x31, 0x1e, 0x30, 0x1c, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x15,
  0x4d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x20, 0x43, 0x6f, 0x72, 0x70, 0x6f, 0x72,
  0x61, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x3a, 0x30, 0x38, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x31,
  0x4d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x20, 0x54, 0x65, 0x73, 0x74, 0x69, 0x6e,
  0x67, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61,
  0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x74, 0x79, 0x20, 0x32, 0x30, 0x31,
  0x30, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30, 0x30, 0x36, 0x31, 0x37, 0x32, 0x30, 0x35, 0x38, 0x30,
  0x32, 0x5a, 0x17, 0x0d, 0x33, 0x35, 0x30, 0x36, 0x31, 0x37, 0x32, 0x31, 0x30, 0x34, 0x31, 0x31,
  0x5a, 0x30, 0x81, 0x90, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
  0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x57, 0x61, 0x73, 0x68,
  0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13,
  0x07, 0x52, 0x65, 0x64, 0x6d, 0x6f, 0x6e, 0x64, 0x31, 0x1e, 0x30, 0x1c, 0x06, 0x03, 0x55, 0x04,
  0x0a, 0x13, 0x15, 0x4d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x20, 0x43, 0x6f, 0x72,
  0x70, 0x6f, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x3a, 0x30, 0x38, 0x06, 0x03, 0x55, 0x04,
  0x03, 0x13, 0x31, 0x4d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x20, 0x54, 0x65, 0x73,
  0x74, 0x69, 0x6e, 0x67, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66,
  0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x74, 0x79, 0x20,
  0x32, 0x30, 0x31, 0x30, 0x30, 0x82, 0x02, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
  0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x02, 0x0f, 0x00, 0x30, 0x82, 0x02, 0x0a,
  0x02, 0x82, 0x02, 0x01, 0x00, 0x95, 0xe3, 0xa8, 0xc1, 0xb9, 0x9c, 0x26, 0x54, 0xb0, 0x99, 0xef,
  0x26, 0x1f, 0xac, 0x1e, 0xc7, 0x30, 0x80, 0xbb, 0xf5, 0x3f, 0xf2, 0xe4, 0xbb, 0xf8, 0xfe, 0x06,
  0x6a, 0x0a, 0xa6, 0x88, 0xbc, 0xb4, 0x8c, 0x45, 0xe0, 0x70, 0x55, 0x19, 0x88, 0xb4, 0x05, 0xcb,
  0xb5, 0xc1, 0xa1, 0xfa, 0xd4, 0x7c, 0xc2, 0x42, 0x53, 0x07, 0x9c, 0x54, 0x56, 0xa8, 0x97, 0xe0,
  0x94, 0x69, 0xbe, 0x13, 0x24, 0xef, 0xe5, 0x8a, 0x29, 0x9c, 0xa6, 0xd0, 0x2b, 0x2f, 0x8a, 0xa6,
  0xe8, 0x79, 0x44, 0x2e, 0x8b, 0xea, 0xc9, 0xbe, 0xb8, 0x54, 0x86, 0x53, 0xbe, 0x07, 0x24, 0x34,
  0x54, 0x15, 0x22, 0x20, 0x01, 0x7b, 0x8a, 0x46, 0xfb, 0xd2, 0x91, 0x07, 0x95, 0x09, 0xb0, 0x56,
  0x11, 0xcc, 0x76, 0xb2, 0xd0, 0x1f, 0x44, 0x79, 0x52, 0x34, 0x28, 0xec, 0x4f, 0x49, 0xc2, 0xcb,
  0x61, 0xd3, 0x86, 0xdc, 0xe4, 0xa3, 0x7e, 0x55, 0x9e, 0x9f, 0xee, 0x10, 0x6f, 0xcf, 0xe1, 0x3d,
  0xf8, 0xb7, 0x84, 0x79, 0xa2, 0x3b, 0x8d, 0x1c, 0xb0, 0x81, 0x7c, 0xe4, 0x44, 0x07, 0xe4, 0xce,
  0x46, 0xb0, 0x98, 0x83, 0x8d, 0x87, 0x8f, 0xe5, 0xf5, 0xae, 0x40, 0x7a, 0xf1, 0xed, 0x3d, 0x9b,
  0x9a, 0x7c, 0x4a, 0xd1, 0xb9, 0xc3, 0x94, 0x05, 0x7b, 0xdc, 0xda, 0xb8, 0xce, 0xdc, 0x1e, 0x6c,
  0xcf, 0xd9, 0x9e, 0x37, 0xef, 0xc3, 0x5a, 0x36, 0x7b, 0x90, 0x86, 0x45, 0xdc, 0xf6, 0x2e, 0xca,
  0xdd, 0xee, 0xde, 0x27, 0xd9, 0x74, 0x9a, 0x69, 0xf5, 0xd9, 0x5d, 0x09, 0x2d, 0x45, 0x41, 0xcc,
  0xb7, 0xc2, 0x82, 0xd4, 0x2a, 0x8c, 0x16, 0x25, 0x92, 0x97, 0x3d, 0x94, 0x4e, 0x89, 0x33, 0x7e,
  0x5b, 0x03, 0x54, 0xcd, 0xb0, 0x83, 0xa0, 0x8e, 0x41, 0xb7, 0x87, 0x8d, 0xd9, 0x05, 0x63, 0x52,
  0xf6, 0xee, 0xe6, 0x4e, 0x13, 0x9d, 0x54, 0xcd, 0x49, 0xfe, 0xe3, 0x8b, 0x3b, 0x50, 0x9b, 0x48,
  0xbb, 0xb2, 0xe5, 0x92, 0xd4, 0xab, 0xa0, 0xc5, 0x10, 0xaf, 0x3e, 0xb1, 0x45, 0x21, 0x34, 0x90,
  0xdc, 0xad, 0xb9, 0xf7, 0xfe, 0x21, 0xae, 0xee, 0x50, 0x58, 0x7a, 0x3a, 0xe5, 0xaa, 0xd8, 0xe3,
  0x82, 0xd6, 0xcf, 0x6d, 0x4d, 0xc9, 0x15, 0xac, 0x9c, 0x31, 0x17, 0xa5, 0x16, 0xa7, 0x42, 0xf6,
  0xda, 0x12, 0x78, 0xa7, 0x66, 0x90, 0xec, 0xfc, 0xcd, 0x01, 0x63, 0xff, 0xf0, 0x0e, 0xba, 0xe1,
  0xcd, 0xf0, 0xdb, 0x6b, 0x9a, 0x0f, 0xf6, 0x0f, 0x04, 0x01, 0x09, 0xbc, 0x9f, 0xce, 0xb7, 0x6c,
  0x51, 0x70, 0x57, 0x08, 0x1b, 0xff, 0x79, 0x9a, 0x52, 0x5d, 0xba, 0xac, 0x14, 0xe5, 0x3b, 0x67,
  0xcf, 0x2c, 0x52, 0xde, 0x27, 0x9a, 0x34, 0x03, 0x6e, 0x25, 0x48, 0xb0, 0x19, 0x74, 0xfc, 0x4d,
  0x98, 0xc2, 0x4b, 0x8c, 0x92, 0xe1, 0x88, 0xae, 0x48, 0x2a, 0xab, 0xab, 0xcd, 0x14, 0x4d, 0xb6,
  0x61, 0x0e, 0xa1, 0x09, 0x8f, 0x2c, 0xdb, 0x45, 0xaf, 0x7d, 0x3b, 0x81, 0x56, 0x08, 0xc9, 0x3b,
  0x41, 0xb7, 0x64, 0x9f, 0x5d, 0x2e, 0x12, 0x7f, 0xb9, 0x69, 0x29, 0x1f, 0x52, 0x45, 0x4a, 0x23,
  0xc6, 0xaf, 0xb6, 0xb2, 0x38, 0x72, 0x9d, 0x08, 0x33, 0xff, 0xd0, 0xcf, 0x89, 0xb6, 0xea, 0x6e,
  0x85, 0x44, 0x94, 0x3e, 0x91, 0x59, 0xeb, 0xef, 0x9e, 0xbd, 0x9b, 0x9c, 0x1a, 0x47, 0x03, 0x4e,
  0xa2, 0x17, 0x96, 0xfa, 0x62, 0x0b, 0xe8, 0x53, 0xb6, 0x4e, 0xe3, 0xe8, 0x2a, 0x73, 0x59, 0xe2,
  0x13, 0xb8, 0xf8, 0x5a, 0x7e, 0xc6, 0xe2, 0x0a, 0xdd, 0x4a, 0x43, 0xcc, 0xc3, 0x77, 0x3b, 0x7a,
  0x31, 0x04, 0x0a, 0xc1, 0x84, 0x96, 0x3a, 0x63, 0x6e, 0x1a, 0x3e, 0x0a, 0x0c, 0x25, 0xb8, 0x7e,
  0xb5, 0x52, 0x0c, 0xb9, 0xab, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x51, 0x30, 0x4f, 0x30, 0x0b,
  0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02, 0x01, 0x86, 0x30, 0x0f, 0x06, 0x03, 0x55,
  0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x1d, 0x06, 0x03,
  0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xa3, 0x01, 0x04, 0x7e, 0x30, 0x88, 0x33, 0xeb, 0xb9,
  0x31, 0x9c, 0xca, 0xeb, 0x85, 0x76, 0x67, 0xfc, 0x65, 0xb4, 0xd1, 0x30, 0x10, 0x06, 0x09, 0x2b,
  0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x01, 0x04, 0x03, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06,
  0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x02, 0x01,
  0x00, 0x49, 0x8b, 0xc1, 0xfc, 0x4f, 0xe8, 0xe4, 0x2d, 0x67, 0x92, 0x9a, 0x76, 0x05, 0xba, 0xd1,
  0xbc, 0x98, 0xe4, 0x2b, 0xba, 0x1f, 0x66, 0x5f, 0x66, 0x23, 0xcf, 0x1c, 0x27, 0xeb, 0xb4, 0xaa,
  0xdd, 0xa0, 0x17, 0x20, 0x55, 0x72, 0x33, 0xb1, 0x76, 0xde, 0xc9, 0x6d, 0x0d, 0x3c, 0x2d, 0x0a,
  0x08, 0x24, 0x2d, 0xec, 0x38, 0x96, 0x7a, 0x83, 0xf1, 0x27, 0x50, 0x3c, 0x86, 0x09, 0xdd, 0x0d,
  0x41, 0xce, 0xaa, 0x5e, 0xf3, 0x8f, 0x7a, 0x3e, 0x3e, 0xf1, 0xf0, 0xba, 0x8b, 0x72, 0xdd, 0x36,
  0xa1, 0x69, 0x05, 0x5b, 0x7c, 0xec, 0xe7, 0x70, 0x63, 0x8d, 0x1d, 0x6e, 0xc0, 0xfd, 0x3a, 0x03,
  0xf1, 0x10, 0x3e, 0x90, 0xd7, 0x7b, 0x7a, 0xdc, 0xea, 0x60, 0xec, 0x2f, 0x53, 0xfd, 0x19, 0x1d,
  0x3a, 0xa1, 0x74, 0x08, 0xc2, 0x7b, 0x3c, 0xe0, 0x50, 0xac, 0x21, 0xd7, 0xb6, 0xdd, 0xdd, 0x3c,
  0x44, 0x1b, 0xf7, 0xf3, 0x44, 0x3e, 0x6c, 0x96, 0xe0, 0xc0, 0x9f, 0xe6, 0xef, 0xdd, 0xdd, 0xb1,
  0xa6, 0x68, 0x61, 0x6c, 0x5e, 0x9e, 0xf9, 0xff, 0x9a, 0x06, 0xa4, 0x6a, 0xcd, 0x9e, 0x75, 0x43,
  0x89, 0x9b, 0xcb, 0x85, 0xf6, 0xdc, 0x0c, 0x46, 0x4a, 0x8c, 0x9b, 0xac, 0x11, 0xa6, 0x63, 0x45,
  0xfb, 0xfc, 0xde, 0x20, 0xee, 0xce, 0x67, 0x9f, 0x3d, 0xd0, 0x93, 0xdb, 0x39, 0xfb, 0xea, 0x5e,
  0x4b, 0xfc, 0xd6, 0x20, 0xf1, 0x95, 0x36, 0x08, 0x8c, 0xb2, 0xb3, 0xa1, 0x97, 0x1b, 0x41, 0x19,
  0xb0, 0xac, 0xfe, 0xe2, 0xd5, 0xab, 0x7d, 0xd9, 0x26, 0xd4, 0xdc, 0xbd, 0x1f, 0x38, 0xc0, 0xe3,
  0x86, 0xdf, 0x24, 0xe7, 0xf5, 0x3e, 0x09, 0xca, 0x4d, 0xa1, 0xba, 0x16, 0xc3, 0x4a, 0xb1, 0xfc,
  0x72, 0x98, 0xcf, 0x0e, 0x92, 0xfa, 0x57, 0x45, 0xe9, 0x48, 0x4d, 0xc6, 0xa2, 0x7c, 0x3b, 0x72,
  0x63, 0xac, 0x4e, 0xf4, 0x74, 0xe9, 0x2b, 0x57, 0xac, 0xab, 0x32, 0x88, 0x0b, 0xa9, 0x10, 0x67,
  0x53, 0x7e, 0xd2, 0x62, 0xd2, 0xfa, 0x68, 0xe8, 0x9d, 0x5b, 0xae, 0xcd, 0xe0, 0xe5, 0xe2, 0x06,
  0x96, 0x0c, 0x34, 0x32, 0xf6, 0xbc, 0x25, 0xad, 0x98, 0xf3, 0x32, 0x60, 0xbe, 0x14, 0xd3, 0x78,
  0xd1, 0x10, 0x6f, 0xff, 0x32, 0xe3, 0x9e, 0x3d, 0x88, 0xda, 0xb3, 0x32, 0x0a, 0xcf, 0x20, 0x65,
  0x47, 0x78, 0xaa, 0xa5, 0x4b, 0x87, 0x6a, 0x83, 0xdc, 0x1a, 0x5a, 0x2a, 0xdf, 0x70, 0x61, 0xaf,
  0x35, 0x32, 0xe0, 0x59, 0xa1, 0x9f, 0x0b, 0x14, 0x7a, 0xaa, 0xab, 0x42, 0x0b, 0x6b, 0xff, 0xfb,
  0x34, 0xcb, 0x9d, 0x96, 0xd7, 0x26, 0x2a, 0x13, 0x3b, 0xe3, 0xdf, 0x11, 0xe6, 0x86, 0x7d, 0x0d,
  0x09, 0x11, 0x93, 0x4b, 0xa4, 0xf6, 0xd2, 0x07, 0xc2, 0xcd, 0xc8, 0xbe, 0xf5, 0x67, 0xf7, 0xae,
  0x05, 0xce, 0x16, 0xfe, 0x90, 0xc9, 0x4a, 0x98, 0x1b, 0x24, 0x69, 0x78, 0x90, 0xf9, 0x34, 0x8e,
  0x37, 0xe8, 0x6e, 0x1d, 0xdc, 0xcf, 0x4f, 0xe7, 0xd2, 0x64, 0x40, 0x1d, 0xc4, 0x30, 0xba, 0xd5,
  0x08, 0x88, 0x67, 0x4b, 0x0f, 0xb8, 0xe5, 0x59, 0xe9, 0x18, 0xd8, 0x0c, 0x60, 0x68, 0xae, 0x7f,
  0xea, 0x91, 0x55, 0xbe, 0xeb, 0xf1, 0xa7, 0x8e, 0xd8, 0x5d, 0x50, 0x3e, 0xbf, 0xd5, 0x69, 0x57,
  0x95, 0x8f, 0xa7, 0xff, 0xe4, 0x09, 0x3f, 0x08, 0x80, 0x97, 0x32, 0x42, 0xb8, 0x82, 0x43, 0x82,
  0x6f, 0x8b, 0x0b, 0x93, 0xda, 0x19, 0xbf, 0x63, 0x4e, 0x5f, 0x9f, 0xed, 0x2c, 0x22, 0xb6, 0x20,
  0x5f, 0x70, 0x44, 0xfa, 0x89, 0x59, 0x93, 0xb0, 0x7b, 0x12, 0x0f, 0x5e, 0x62, 0x62, 0x51, 0x11,
  0xbd, 0xba, 0x5a, 0xd0, 0xce, 0xa1, 0xb6, 0xef, 0x80, 0x20, 0xe6, 0x73, 0x4b, 0x11, 0x06, 0x56,
  0xe2, 0x0a
  };

//
// OID ASN.1 Value for Hash Algorithms
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT8 HashOidValue[] = {
  0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x02, 0x05,         // OBJ_md5
  0x2B, 0x0E, 0x03, 0x02, 0x1A,                           // OBJ_sha1
  0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x04,   // OBJ_sha224
  0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01,   // OBJ_sha256
  0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x02,   // OBJ_sha384
  0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x03,   // OBJ_sha512
  };

UNIT_TEST_STATUS
EFIAPI
TestVerifyAuthenticodeVerify (
  IN UNIT_TEST_CONTEXT           Context
  )
{
  BOOLEAN  Status;

  //
  // Sample Code: Demonstrate how to check the Hash algorithm in PE/COFF Authenticode.
  //    According to PKCS#7 Definition:
  //        SignedData ::= SEQUENCE {
  //            version Version,
  //            digestAlgorithms DigestAlgorithmIdentifiers,
  //            contentInfo ContentInfo,
  //            .... }
  //    The DigestAlgorithmIdentifiers can be used to determine the hash algorithm in PE/COFF hashing
  //    This field has the fixed offset (+32) in final Authenticode ASN.1 data.
  //
  if (CompareMem (AuthenticodeWithSha1 + 32, &HashOidValue[8], 5) == 0) {
    //
    // Verify Authenticode signature.
    // The PeHash value should be calculated following MSFT's Authenticode Specification.
    //
    Status = AuthenticodeVerify (
               AuthenticodeWithSha1,
               sizeof (AuthenticodeWithSha1),
               TestRootCert1,
               sizeof (TestRootCert1),
               PeSha1Hash,
               SHA1_DIGEST_SIZE
               );
    UT_ASSERT_TRUE (Status);
  }

  if (CompareMem (AuthenticodeWithSha256 + 32, &HashOidValue[22], 9) == 0) {
    Status = AuthenticodeVerify (
               AuthenticodeWithSha256,
               sizeof (AuthenticodeWithSha256),
               TestRootCert2,
               sizeof (TestRootCert2),
               PeSha256Hash,
               SHA256_DIGEST_SIZE
               );
    UT_ASSERT_TRUE (Status);
  }

  return UNIT_TEST_PASSED;
}

TEST_DESC mAuthenticodeTest[] = {
    //
    // -----Description--------------------------------------Class----------------------Function-----------------Pre---Post--Context
    //
    {"TestVerifyAuthenticodeVerify()",        "CryptoPkg.BaseCryptLib.Authenticode",   TestVerifyAuthenticodeVerify,        NULL, NULL, NULL},
};

UINTN mAuthenticodeTestNum = ARRAY_SIZE(mAuthenticodeTest);
