;;;; -*- Mode: Lisp; Syntax: ANSI-Common-Lisp; Base: 10; Package: umlisp -*-
;;;; *************************************************************************
;;;; FILE IDENTIFICATION
;;;;
;;;; Name:     classes.lisp
;;;; Purpose:  Class defintions for UMLisp
;;;; Author:   Kevin M. Rosenberg
;;;; Created:  Apr 2000
;;;;
;;;; $Id$
;;;;
;;;; This file, part of UMLisp, is
;;;;    Copyright (c) 2000-2006 by Kevin M. Rosenberg, M.D.
;;;;
;;;; UMLisp users are granted the rights to distribute and use this software
;;;; as governed by the terms of the GNU General Public License.
;;;; *************************************************************************

(in-package #:umlisp)

(defclass umlsclass (hyperobject)
  ()
  (:metaclass hyperobject-class)
  (:description "Parent class of all UMLS objects. It is based on the HYPEROBJECT-CLASS metaclass that provides object printing functions."))


(defclass usrl (umlsclass)
  ((sab :value-type string :initarg :sab :reader sab)
   (srl :value-type fixnum :initarg :srl :reader srl))
  (:metaclass hyperobject-class)
  (:user-name "Source Restriction Level")
  (:default-print-slots sab srl)
  (:description "Custom Table: Source Restriction Level"))


(defclass urank (umlsclass)
  ((rank :value-type fixnum :initarg :rank :reader rank)
   (sab :value-type string :initarg :sab :reader sab)
   (tty :value-type string :initarg :tty :reader tty)
   (suppress :value-type string :initarg :suppress :reader suppress))
  (:metaclass hyperobject-class)
  (:user-name "Rank")
  (:default-print-slots rank sab tty suppres))

(defclass udef (umlsclass)
  ((def :value-type cdata :initarg :def :reader def)
   (sab :value-type string :initarg :sab :reader sab :hyperlink find-usab-rsab)
   (suppress :value-type string :initarg :suppress :reader suppress))
  (:metaclass hyperobject-class)
  (:user-name "Definition")
  (:default-print-slots sab def))

(defclass usat (umlsclass)
  ((sab :value-type string :initarg :sab :reader sab :hyperlink find-usab-rsab)
   (code :value-type string :initarg :code :reader code)
   (atn :value-type string :initarg :atn :reader atn)
   (atv :value-type cdata :initarg :atv :reader atv))
  (:metaclass hyperobject-class)
  (:user-name "Simple Attribute")
  (:default-print-slots sab code atn atv))

(defclass usab (umlsclass)
  ((vcui :value-type fixnum :initarg :vcui :reader vcui :print-formatter fmt-cui)
   (rcui :value-type fixnum :initarg :rcui :reader rcui :print-formatter fmt-cui)
   (vsab :value-type string :initarg :vsab :reader vsab)
   (rsab :value-type string :initarg :rsab :reader rsab :hyperlink find-ustr-sab
         :hyperlink-parameters (("subobjects" . "no")))
   (son :value-type string :initarg :son :reader son)
   (sf :value-type string :initarg :sf :reader sf)
   (sver :value-type string :initarg :sver :reader sver)
   (vstart :value-type string :initarg :vstart :reader vstart)
   (vend :value-type string :initarg :vend :reader vend)
   (imeta :value-type string :initarg :imeta :reader imeta)
   (rmeta :value-type string :initarg :rmeta :reader rmeta)
   (slc :value-type cdata :initarg :slc :reader slc)
   (scc :value-type cdata :initarg :scc :reader scc)
   (srl :value-type fixnum :initarg :srl :reader srl)
   (tfr :value-type fixnum :initarg :tfr :reader tfr :print-formatter fmt-comma-integer)
   (cfr :value-type fixnum :initarg :cfr :reader cfr :print-formatter fmt-comma-integer)
   (cxty :value-type string :initarg :cxty :reader cxty)
   (ttyl :value-type string :initarg :ttyl :reader ttyl)
   (atnl :value-type string :initarg :atnl :reader atnl)
   (lat :value-type string :initarg :lat :reader lat)
   (cenc :value-type string :initarg :cenc :reader cenc)
   (curver :value-type string :initarg :curver :reader curver)
   (sabin :value-type string :initarg :sabin :reader sabin)
   (ssn :value-type string :initarg :ssn :reader ssn)
   (scit :value-type string :initarg :scit :reader scit))
  (:metaclass hyperobject-class)
  (:user-name "Source Abbreviation")
  (:default-print-slots vcui rcui vsab rsab son sf sver vstart vend imeta
                rmeta slc scc srl tfr cfr cxty ttyl atnl lat cenc
                curver sabin ssn scit))

(defclass uhier (umlsclass)
  ((cui :value-type fixnum :initarg :cui :reader cui :hyperlink find-ucon-cui
        :print-formatter fmt-cui)
   (aui :value-type fixnum :initarg :aui :reader aui :hyperlink find-ucon-aui
         :print-formatter fmt-aui)
   (cxn :value-type fixnum :initarg :cxn :reader cxn)
   (paui :value-type fixnum :initarg :paui :reader paui
         :print-formatter fmt-aui)
   (sab :value-type string :initarg :sab :reader sab :hyperlink find-usab-rsab)
   (rela :value-type string :initarg :rela :reader rela)
   (ptr :value-type string :initarg :ptr :reader ptr)
   (hcd :value-type string :initarg :hcd :reader hcd)
   (cvf :value-type string  :initarg :cvf :reader cvf))
  (:metaclass hyperobject-class)
  (:user-name "Context")
  (:default-print-slots cxn paui sab rela ptr hcd))

(defclass ustr (umlsclass)
  ((sui :value-type fixnum :initarg :sui :reader sui :print-formatter fmt-sui
        :hyperlink find-ustr-sui)
   (cui :value-type fixnum :initarg :cui :reader cui :print-formatter fmt-cui
        :hyperlink find-ucon-cui)
   (lui :value-type fixnum :initarg :lui :reader lui :print-formatter fmt-lui
        :hyperlink find-uterm-lui)
   (cuisui :value-type integer :initarg :cuisui :reader cuisui )
   (str :value-type cdata :initarg :str :reader str)
   (lrl :value-type fixnum :initarg :lrl :reader lrl)
   (stt :value-type string :initarg :stt :reader stt)
   (suppress :value-type string :initarg :suppress :reader suppress)
   (s#so :reader s#so :subobject (find-uso-cuisui cui sui))
   (s#sat :reader s#sat :subobject (find-usat-ui cui lui sui)))
  (:metaclass hyperobject-class)
  (:user-name "String")
  (:default-print-slots sui stt lrl str suppress))


(defclass uso (umlsclass)
  ((aui :value-type fixnum :initarg :aui :reader aui :print-formatter fmt-aui
        :hyperlink find-ucon-aui)
   (cui :value-type fixnum :initarg :cui :reader cui :print-formatter fmt-cui
        :hyperlink find-ucon-cui)
   (sui :value-type fixnum :initarg :sui :reader sui :print-formatter fmt-sui
        :hyperlink find-ucon-sui)
   (saui :value-type string :initarg :saui :reader saui)
   (sdui :value-type string :initarg :sdui :reader sdui)
   (scui :value-type string :initarg :scui :reader scui)
   (tty :value-type string :initarg :tty :reader tty)
   (code :value-type string :initarg :code :reader code)
   (sab :value-type string :initarg :sab :reader sab :hyperlink find-usab-rsab)
   (lat :value-type string :initarg :lat :reader lat)
   (str :value-type cdata :initarg :str :reader str)
   (srl :value-type fixnum :initarg :srl :reader srl))
  (:metaclass hyperobject-class)
  (:user-name "Source")
  (:default-print-slots aui sab code saui sdui scui tty srl))

(defclass uterm (umlsclass)
  ((lui :value-type fixnum :initarg :lui :reader lui :print-formatter fmt-lui
        :hyperlink find-uterm-lui)
   (cui :value-type fixnum :initarg :cui :reader cui :print-formatter fmt-cui
        :hyperlink find-ucon-cui)
   (lat :value-type string :initarg :lat :reader lat)
   (ts :value-type string  :initarg :ts :reader ts)
   (lrl :value-type fixnum :initarg :lrl :reader lrl)
   (s#str :reader s#str :subobject (find-ustr-cuilui cui lui))
   (s#sat :reader s#sat :subobject (find-usat-ui cui lui)))
  (:metaclass hyperobject-class)
  (:user-name "Term")
  (:default-print-slots lui lat ts lrl))

(defclass usty (umlsclass)
  ((tui :value-type fixnum :initarg :tui :reader tui :print-formatter fmt-tui
        :hyperlink find-ucon-tui
        :hyperlink-parameters (("subobjects" . "no")))
   (sty :value-type string :initarg :sty :reader sty))
  (:metaclass hyperobject-class)
  (:user-name "Semantic Type")
  (:default-print-slots tui sty))

(defclass urel (umlsclass)
  ((rel :value-type string :initarg :rel :reader rel :hyperlink find-brel-rel)
   (cui1 :value-type fixnum :initarg :cui1 :reader cui1 :print-formatter fmt-cui)
   (aui1 :value-type fixnum :initarg :aui1 :reader aui1 :print-formatter fmt-aui)
   (stype1 :value-type string  :initarg :stype1 :reader stype1)
   (cui2 :value-type fixnum :initarg :cui2 :reader cui2 :hyperlink find-ucon-sui
         :print-formatter fmt-cui)
   (aui2 :value-type fixnum :initarg :aui2 :reader aui2 :hyperlink find-ucon-aui
         :print-formatter fmt-aui)
   (pfstr2 :value-type cdata :initarg :pfstr2 :reader pfstr2)
   (stype2 :value-type string  :initarg :stype2 :reader stype2)
   (rela :value-type string :initarg :rela :reader rela)
   (rui :value-type fixnum  :initarg :rui :reader rui :print-formatter fmt-rui)
   (srui :value-type string  :initarg :srui :reader srui)
   (sab :value-type string :initarg :sab :reader sab :hyperlink find-usab-rsab)
   (sl :value-type string  :initarg :sl :reader sl)
   (rg :value-type string  :initarg :rg :reader rg)
   (dir :value-type string  :initarg :dir :reader dir)
   (suppress :value-type string  :initarg :suppress :reader suppress)
   (cvf :value-type string  :initarg :cvf :reader cvf))
  (:metaclass hyperobject-class)
  (:user-name "Relationship")
  (:default-print-slots stype1 rel cui2 aui2 stype2 rela rui srui sab sl rg dir suppress pfstr2))

(defclass ucoc (umlsclass)
  ((cui1 :value-type fixnum :initarg :cui1 :reader cui1 :print-formatter fmt-cui)
   (aui1 :value-type fixnum :initarg :aui1 :reader aui1 :print-formatter fmt-aui)
   (cui2 :value-type fixnum :initarg :cui2 :reader cui2 :print-formatter fmt-cui
         :hyperlink find-ucon-cui)
   (aui2 :value-type fixnum :initarg :aui2 :reader aui2 :print-formatter fmt-aui
         :hyperlink find-ucon-aui)
   (pfstr2 :value-type cdata :initarg :pfstr2 :reader pfstr2)
   (sab :value-type string :initarg :sab :reader sab)
   (cot :value-type string :initarg :cot :reader cot)
   (cof :value-type fixnum :initarg :cof :reader cof)
   (coa :value-type cdata :initarg :coa :reader coa)
   (cvf :value-type string  :initarg :cvf :reader cvf))
  (:metaclass hyperobject-class)
  (:user-name "Co-occuring Concept")
  (:default-print-slots cot cof coa cui2 aui2 sab pfstr2))

(defclass ucon (umlsclass)
  ((cui :value-type fixnum :initarg :cui :reader cui :print-formatter fmt-cui
        :hyperlink find-ucon-cui)
   (lrl :value-type fixnum :initarg :lrl :reader lrl
        :compute-cached-value (find-lrl-cui cui))
   (pfstr :value-type cdata :initarg :pfstr :reader pfstr
          :compute-cached-value (find-pfstr-cui cui))
   (s#def :reader s#def :subobject (find-udef-cui cui))
   (s#so :reader s#so :subobject (find-uso-cui cui))
   (s#hier :reader s#hier :subobject (find-uhier-cui cui))
   (s#map :reader s#map :subobject (find-umap-cui cui))
   (s#smap :reader s#smap :subobject (find-usmap-cui cui))
   (s#sty :reader s#sty :subobject (find-usty-cui cui))
   (s#term :reader s#term :subobject (find-uterm-cui cui))
   (s#sat :reader s#sat :subobject (find-usat-ui cui))
   (s#rel :reader s#rel :subobject (find-urel-cui cui))
   (s#coc :reader s#coc :subobject (find-ucoc-cui cui)))
  (:metaclass hyperobject-class)
  (:user-name "Concept")
  (:default-print-slots cui lrl pfstr))


(defclass uconso (umlsclass)
  ((cui :value-type fixnum :initarg :cui :reader cui :print-formatter fmt-cui
        :hyperlink find-ucon-cui)
   (pfstr :value-type cdata :initarg :pfstr :reader pfstr
          :compute-cached-value (find-pfstr-cui cui))
   (lat :value-type string :initarg :lat :reader lat)
   (ts :value-type string :initarg :ts :reader ts)
   (lui :value-type fixnum :initarg :lui :reader lui :print-formatter fmt-lui)
   (stt :value-type string :initarg :stt :reader stt)
   (sui :value-type fixnum :initarg :sui :reader sui :print-formatter fmt-sui)
   (ispref :value-type string :initarg :ispref :reader ispref)
   (aui :value-type fixnum :initarg :aui :reader aui :print-formatter fmt-aui)
   (saui :value-type string :initarg :saui :reader saui)
   (scui :value-type string :initarg :scui :reader scui)
   (sdui :value-type string :initarg :sdui :reader sdui)
   (sab :value-type string :initarg :sab :reader sab)
   (tty :value-type string :initarg :tty :reader tty)
   (code :value-type string :initarg :code :reader code)
   (str :value-type string :initarg :str :reader str)
   (srl :value-type fixnum :initarg :srl :reader srl)
   (suppress :value-type string :initarg :suppress :reader suppress)
   (cvf :value-type string :initarg :cvf :reader cvf)
   (kpfeng :value-type string :initarg :kpfeng :reader kpfeng)
   (kcuisui :value-type bigint :initarg :kcuisui :reader kcuisui)
   (kcuilui :value-type bigint :initarg :kcuilui :reader kcuilui)
   (kcuilrl :value-type fixnum :initarg :kcuilrl :reader kcuilrl)
   (kluilrl :value-type fixnum :initarg :kluilrl :reader kluilrl)
   (ksuilrl :value-type fixnum :initarg :ksuilrl :reader ksuilrl)
   (s#def :reader s#def :subobject (find-udef-cui cui))
   (s#so :reader s#so :subobject (find-uso-cui cui))
   (s#hier :reader s#hier :subobject (find-uhier-cui cui))
   (s#map :reader s#map :subobject (find-umap-cui cui))
   (s#smap :reader s#smap :subobject (find-usmap-cui cui))
   (s#sty :reader s#sty :subobject (find-usty-cui cui))
   (s#term :reader s#term :subobject (find-uterm-cui cui))
   (s#sat :reader s#sat :subobject (find-usat-ui cui))
   (s#rel :reader s#rel :subobject (find-urel-cui cui))
   (s#coc :reader s#coc :subobject (find-ucoc-cui cui)))
  (:documentation "CONSO is a new concept from the RRF files. This object is a rather raw row from the MRCONSO table.")
  (:metaclass hyperobject-class)
  (:user-name "Concept")
  (:default-print-slots cui kcuilrl str sab))

(defclass umap (umlsclass)
  ((mapsetcui :value-type fixnum :initarg :mapsetcui :reader mapsetcui)
   (mapsetsab :value-type string :initarg :mapsetsab :reader mapsetsab)
   (mapsubsetid :value-type string :initarg :mapsubsetid :reader mapsubsetid)
   ;; fixme: will convert to integer
   (maprank :value-type string :initarg :maprank :reader maprank)
   (fromid :value-type string :initarg :fromid :reader fromid)
   (fromsid :value-type string :initarg :fromsid :reader fromsid)
   (fromexpr :value-type string :initarg :fromexpr :reader fromexpr)
   (fromtype :value-type string :initarg :fromtype :reader fromtype)
   (fromrule :value-type string :initarg :fromrule :reader fromrule)
   (fromres :value-type string :initarg :fromres :reader fromres)
   (rel :value-type string :initarg :rel :reader rel)
   (rela :value-type string :initarg :rela :reader rela)
   (toid :value-type string :initarg :toid :reader toid)
   (tosid :value-type string :initarg :tosid :reader tosid)
   (toexpr :value-type string :initarg :toexpr :reader toexpr)
   (totype :value-type string :initarg :totype :reader totype)
   (torule :value-type string :initarg :torule :reader torule)
   (tores :value-type string :initarg :tores :reader tores)
   (maprule :value-type string :initarg :maprule :reader maprule)
   (maptype :value-type string :initarg :maptype :reader maptype)
   (mapatn :value-type string :initarg :mapatn :reader mapatn)
   (mapatv :value-type string :initarg :mapatv :reader mapatv)
   (cvf :value-type string  :initarg :cvf :reader cvf))
  (:metaclass hyperobject-class)
  (:user-name "Mapping")
  (:default-print-slots mapsetcui mapsetsab mapsubsetid maprank fromid fromsid fromexpr fromtype
                        fromrule fromres rel rela toid tosid toexpr totype torule tores maprule
                        maptype mapatn mapatv))

(defclass usmap (umlsclass)
  ((mapsetcui :value-type fixnum :initarg :mapsetcui :reader mapsetcui)
   (mapsetsab :value-type string :initarg :mapsetsab :reader mapsetsab)
   (fromexpr :value-type string :initarg :fromexpr :reader fromexpr)
   (fromtype :value-type string :initarg :fromtype :reader fromtype)
   (rel :value-type string :initarg :rel :reader rel)
   (rela :value-type string :initarg :rela :reader rela)
   (toexpr :value-type string :initarg :toexpr :reader toexpr)
   (totype :value-type string :initarg :totype :reader totype)
   (cvf :value-type string  :initarg :cvf :reader cvf))
  (:metaclass hyperobject-class)
  (:user-name "Simple Mapping")
  (:default-print-slots mapsetcui mapsetsab fromexpr fromtype rel rela toexpr totype))

(defclass uxw (umlsclass)
  ((wd :value-type string :initarg :wd :reader wd)
   (cui :value-type fixnum :initform nil :initarg :cui :reader cui :print-formatter fmt-cui)
   (lui :value-type fixnum :initform nil :initarg :lui :reader lui :print-formatter fmt-lui)
   (sui :value-type fixnum :initform nil :initarg :sui :reader sui :print-formatter fmt-sui))
  (:metaclass hyperobject-class)
  (:user-name "XW Index" "XW Indices")
  (:default-print-slots wd cui lui sui))

(defclass uxw-noneng (umlsclass)
  ((lat :value-type string :initarg :lat :reader lat)
   (wd :value-type string :initarg :wd :reader wd)
   (cui :value-type fixnum :initform nil :initarg :cui :reader cui :print-formatter fmt-cui)
   (lui :value-type fixnum :initform nil :initarg :lui :reader lui :print-formatter fmt-lui)
   (sui :value-type fixnum :initform nil :initarg :sui :reader sui :print-formatter fmt-sui)
   (lrl :value-type fixnum :initform nil :initarg :lrl :reader lrl))
  (:metaclass hyperobject-class)
  (:user-name "XW Non-English Index" "XW Non-English Indices")
  (:default-print-slots wd cui lui sui))

(defclass uxnw (umlsclass)
  ((lat :value-type string :initarg :lat :reader lat)
   (nwd :value-type string :initarg :nwd :reader nwd)
   (cuilist :value-type list :initarg :cuilist :reader uxnw-cuilist))
  (:metaclass hyperobject-class)
  (:user-name "XNW Index" "XNW Indices")
  (:default-print-slots lat nwd cuilist))

(defclass uxns (umlsclass)
  ((lat :value-type string :initarg :lat :reader lat)
   (nstr :value-type string :initarg :nstr :reader nstr)
   (cuilist :value-type list :initarg :cuilist :reader cuilist))
  (:metaclass hyperobject-class)
  (:user-name "XNS Index" "XNS Indices")
  (:default-print-slots lat nstr cuilist))

(defclass udoc (umlsclass)
  ((dockey :value-type string :initarg :dockey :reader dockey)
   (expl :value-type cdata :initarg :expl :reader expl)
   (dtype :value-type cdata :initarg :dtype :reader dtype)
   (dvalue :value-type cdata :initarg :dvalue :reader dvalue))
  (:metaclass hyperobject-class)
  (:user-name "Documentation record")
  (:default-print-slots dockey expl dtype dvalue))


;;; LEX objects

(defclass lexterm (umlsclass)
  ((eui :value-type fixnum :initarg :eui :reader eui :print-formatter fmt-eui
        :hyperlink find-lexterm-eui)
   (wrd :value-type string :initarg :wrd :reader wrd)
   (s#abr :reader s#abr :subobject (find-labr-eui eui))
   (s#agr :reader s#agr :subobject (find-lagr-eui eui))
   (s#cmp :reader s#cmp :subobject (find-lcmp-eui eui))
   (s#mod :reader s#mod :subobject (find-lmod-eui eui))
   (s#nom :reader s#nom :subobject (find-lnom-eui eui))
   (s#prn :reader s#prn :subobject (find-lprn-eui eui))
   (s#prp :reader s#prp :subobject (find-lprp-eui eui))
   (s#spl :reader s#spl :subobject (find-lspl-eui eui))
   (s#trm :reader s#trm :subobject (find-ltrm-eui eui))
   (s#typ :reader s#typ :subobject (find-ltyp-eui eui)))
  (:metaclass hyperobject-class)
  (:user-name "Lexical Term")
  (:default-print-slots eui wrd))


(defclass labr  (umlsclass)
  ((eui :value-type integer :initarg :eui :reader eui :print-formatter fmt-eui)
   (bas :value-type string :initarg :bas :reader bas)
   (abr :value-type string :initarg :abr :reader abr)
   (eui2 :value-type integer :initarg :eui2 :reader eui2 :print-formatter fmt-eui)
   (bas2 :value-type string :initarg :bas2 :reader bas2))
  (:metaclass hyperobject-class)
  (:user-name "Abbreviations and Acronym")
  (:default-print-slots eui bas abr eui2 bas2))

(defclass lagr  (umlsclass)
  ((eui :value-type integer :initarg :eui :reader eui :print-formatter fmt-eui)
   (str :value-type string :initarg :str :reader str)
   (sca :value-type string :initarg :sca :reader sca)
   (agr :value-type string :initarg :agr :reader agr)
   (cit :value-type string :initarg :cit :reader cit)
   (bas :value-type string :initarg :bas :reader bas))
  (:metaclass hyperobject-class)
  (:user-name "Agreement and Inflection")
  (:default-print-slots eui str sca agr cit bas))

(defclass lcmp  (umlsclass)
  ((eui :value-type integer :initarg :eui :reader eui :print-formatter fmt-eui)
   (bas :value-type string :initarg :bas :reader bas)
   (sca :value-type string :initarg :sca :reader sca)
   (com :value-type string :initarg :com :reader com))
  (:metaclass hyperobject-class)
  (:user-name "Complementation")
  (:default-print-slots eui bas sca com))

(defclass lmod  (umlsclass)
  ((eui :value-type integer :initarg :eui :reader eui :print-formatter fmt-eui)
   (bas :value-type string :initarg :bas :reader bas)
   (sca :value-type string :initarg :sca :reader sca)
   (psnmod :value-type string :initarg :psnmod :reader psnmod)
   (fea :value-type string :initarg :fea :reader fea))
  (:metaclass hyperobject-class)
  (:user-name "Modifier")
  (:default-print-slots eui bas sca psnmod fea))

(defclass lnom  (umlsclass)
  ((eui :value-type integer :initarg :eui :reader eui :print-formatter fmt-eui)
   (bas :value-type string :initarg :bas :reader bas)
   (sca :value-type string :initarg :sca :reader sca)
   (eui2 :value-type integer :initarg :eui2 :reader eui2 :print-formatter fmt-eui)
   (bas2 :value-type string :initarg :bas2 :reader bas2)
   (sca2 :value-type string :initarg :sca2 :reader sca2))
  (:metaclass hyperobject-class)
  (:user-name "Nominalization")
  (:default-print-slots eui bas sca eui2 bas2 sca2))

(defclass lprn  (umlsclass)
  ((eui :value-type integer :initarg :eui :reader eui :print-formatter fmt-eui)
   (bas :value-type string :initarg :bas :reader bas)
   (num :value-type string :initarg :num :reader num)
   (gnd :value-type string :initarg :gnd :reader gnd)
   (cas :value-type string :initarg :cas :reader cas)
   (pos :value-type string :initarg :pos :reader pos)
   (qnt :value-type string :initarg :qnt :reader qnt)
   (fea :value-type string :initarg :fea :reader fea))
  (:metaclass hyperobject-class)
  (:user-name "Pronoun")
  (:default-print-slots eui bas num gnd cas pos qnt fea))

(defclass lprp  (umlsclass)
  ((eui :value-type integer :initarg :eui :reader eui :print-formatter fmt-eui)
   (bas :value-type string :initarg :bas :reader bas)
   (str :value-type string :initarg :str :reader str)
   (sca :value-type string :initarg :sca :reader sca)
   (fea :value-type string :initarg :fea :reader fea))
  (:metaclass hyperobject-class)
  (:user-name "Property" "Properties")
  (:default-print-slots eui bas str sca fea))


(defclass lspl  (umlsclass)
  ((eui :value-type integer :initarg :eui :reader eui :print-formatter fmt-eui)
   (spv :value-type string :initarg :spv :reader spv)
   (bas :value-type string :initarg :bas :reader bas))
  (:metaclass hyperobject-class)
  (:user-name "Spelling Variant")
  (:default-print-slots eui spv bas))


(defclass ltrm  (umlsclass)
  ((eui :value-type integer :initarg :eui :reader eui :print-formatter fmt-eui)
   (bas :value-type string :initarg :bas :reader bas)
   (gen :value-type string :initarg :gen :reader gen))
  (:metaclass hyperobject-class)
  (:user-name "Trade Mark")
  (:default-print-slots eui bas gen))

(defclass ltyp  (umlsclass)
  ((eui :value-type integer :initarg :eui :reader eui :print-formatter fmt-eui)
   (bas :value-type string :initarg :bas :reader bas)
   (sca :value-type string :initarg :sca :reader sca)
   (typ :value-type string :initarg :typ :reader typ))
  (:metaclass hyperobject-class)
  (:user-name "Inflection Type")
  (:default-print-slots eui bas sca typ))

(defclass lwd (umlsclass)
  ((wrd :value-type string :initarg :wrd :reader wrd)
   (euilist :value-type list :initarg :euilist :reader euilist))
  (:metaclass hyperobject-class)
  (:user-name "Lexical Word Index" "Lexical Word Indices")
  (:default-print-slots wrd euilist))

;;; Semantic NET objects

(defclass sdef (umlsclass)
  ((rt :value-type string :initarg :rt :reader rt)
   (ui :value-type integer :initarg :ui :reader ui :print-formatter fmt-tui)
   (styrl :value-type string :initarg :styrl :reader styrl)
   (stnrtn :value-type string :initarg :stnrtn :reader stnrtn)
   (def :value-type string :initarg :def :reader def)
   (ex :value-type string :initarg :ex :reader ex)
   (un :value-type string :initarg :un :reader un)
   (rh :value-type string :initarg :rh :reader rh)
   (abr :value-type string :initarg :abr :reader abr)
   (rin :value-type string :initarg :rin :reader rin))
  (:metaclass hyperobject-class)
  (:user-name "Basic information about Semantic Types and Relation")
  (:default-print-slots rt ui styrl stnrtn def ex un rh abr rin))

(defclass sstr (umlsclass)
  ((styrl :value-type string :initarg :styrl :reader styrl)
   (rl :value-type string :initarg :rl :reader rl)
   (styrl2 :value-type string :initarg :styrl2 :reader styrl2)
   (ls :value-type string :initarg :ls :reader ls))
  (:metaclass hyperobject-class)
  (:user-name "Structure of the Network")
  (:default-print-slots styrl rl styrl2 ls))

(defclass sstre1 (umlsclass)
  ((ui :value-type integer :initarg :ui :reader ui :print-formatter fmt-tui)
   (ui2 :value-type integer :initarg :ui2 :reader ui2 :print-formatter fmt-tui)
   (ui3 :value-type integer :initarg :ui3 :reader ui3 :print-formatter fmt-tui))
  (:metaclass hyperobject-class)
  (:user-name "Fully Inherited Set of Relation (TUIs)"
              "Fully Inherited Set of Relations (TUIs)")
  (:default-print-slots ui ui2 ui3))

(defclass sstre2 (umlsclass)
  ((sty :value-type string :initarg :ui :reader sty)
   (rl :value-type string :initarg :ui2 :reader rl)
   (sty2 :value-type string :initarg :ui3 :reader sty2))
  (:metaclass hyperobject-class)
  (:user-name "Fully Inherited Set of Relation (strings)"
              "Fully Inherited Set of Relations (strings)")
  (:default-print-slots sty rl sty2))


;;; **************************
;;; Local Classes
;;; **************************

(defclass ustats (umlsclass)
  ((name :value-type string :initarg :name :reader name)
   (hits :value-type integer :initarg :hits :reader hits
         :user-name "count"
         :print-formatter fmt-comma-integer)
   (srl :value-type fixnum :initarg :srl :reader srl))
  (:metaclass hyperobject-class)
  (:default-initargs :name nil :hits nil :srl nil)
  (:user-name "UMLS Statistic")
  (:default-print-slots name hits srl)
  (:documentation "Custom Table: UMLS Database statistics."))

