/* SPDX-License-Identifier: Apache-2.0 OR GPL-2.0-or-later
 * Copyright 2021 IBM Corp.
 */

#ifndef __PAU_REGS_H
#define __PAU_REGS_H

/* PAU FIR registers */
#define PAU_FIR(n)				(0x400 + (n) * 0x40)
#define PAU_FIR_MASK(n)				(0x403 + (n) * 0x40)
#define PAU_FIR_ACTION0(n)			(0x406 + (n) * 0x40)
#define PAU_FIR_ACTION1(n)			(0x407 + (n) * 0x40)
#define PAU_FIR_MAX				3

#define PAU_FIR1_NDL_BRICKS_0_5			PPC_BITMASK(0, 11)
#define PAU_FIR1_NDL_BRICKS_6_11		PPC_BITMASK(47, 58)
#define PAU_FIR2_OTL_PERR			PPC_BIT(18)

/* PAU RING: Indirect address/data port */
#define PAU_MISC_SCOM_IND_SCOM_ADDR		0x33e
#define   PAU_MISC_DA_ADDR			PPC_BITMASK(0, 23)
#define   PAU_MISC_DA_LEN			PPC_BITMASK(24, 25)
#define     PAU_MISC_DA_LEN_4B			2
#define     PAU_MISC_DA_LEN_8B			3
#define PAU_MISC_SCOM_IND_SCOM_DATA		0x33f

/* PAU RING: Indirect register blocks */
#define PAU_BLOCK(nib0, nib1)			((nib0) << 20 | (nib1) << 16)
#define PAU_REG_BLOCK(reg)			((reg) & 0xff0000)
#define PAU_REG_OFFSET(reg)			((reg) & 0xffff)

#define PAU_BLOCK_CQ_SM(n)			PAU_BLOCK(4, (n))
#define PAU_BLOCK_CQ_CTL			PAU_BLOCK(4, 4)
#define PAU_BLOCK_CQ_DAT			PAU_BLOCK(4, 5)
#define PAU_BLOCK_OTL(brk)			PAU_BLOCK(4, 0xC + (brk))
#define PAU_BLOCK_OTL_PSL(brk)			PAU_BLOCK(0, 0xC + (brk))
#define PAU_BLOCK_XSL				PAU_BLOCK(4, 0xE)
#define PAU_BLOCK_PAU_XTS			PAU_BLOCK(7, 1)
#define PAU_BLOCK_PAU_MISC			PAU_BLOCK(7, 2)
#define PAU_BLOCK_PAU_XTS_ATSD(n)		PAU_BLOCK(8, (n))

/*
 * CQ_SM block registers
 *
 * Definitions here use PAU_BLOCK_CQ_SM(0), but when pau_write() is given
 * one of these, it will do corresponding writes to every CQ_SM block.
 */
#define PAU_MCP_MISC_CFG0			(PAU_BLOCK_CQ_SM(0) + 0x000)
#define   PAU_MCP_MISC_CFG0_MA_MCRESP_OPT_WRP	PPC_BIT(9)
#define   PAU_MCP_MISC_CFG0_ENABLE_PBUS		PPC_BIT(26)
#define   PAU_MCP_MISC_CFG0_OCAPI_MODE		PPC_BITMASK(44, 48)
#define PAU_MCP_MISC_CERR_MESSAGE0		(PAU_BLOCK_CQ_SM(0) + 0x030)
#define PAU_MCP_MISC_CERR_MESSAGE1		(PAU_BLOCK_CQ_SM(0) + 0x038)
#define PAU_MCP_MISC_CERR_MESSAGE2		(PAU_BLOCK_CQ_SM(0) + 0x040)
#define PAU_MCP_MISC_CERR_MESSAGE3		(PAU_BLOCK_CQ_SM(0) + 0x048)
#define PAU_MCP_MISC_CERR_MESSAGE4		(PAU_BLOCK_CQ_SM(0) + 0x050)
#define PAU_MCP_MISC_CERR_MESSAGE5		(PAU_BLOCK_CQ_SM(0) + 0x058)
#define PAU_MCP_MISC_CERR_MESSAGE6		(PAU_BLOCK_CQ_SM(0) + 0x060)
#define PAU_MCP_MISC_CERR_MESSAGE7		(PAU_BLOCK_CQ_SM(0) + 0x068)
#define PAU_MCP_MISC_CERR_FIRST0		(PAU_BLOCK_CQ_SM(0) + 0x078)
#define PAU_MCP_MISC_CERR_FIRST1		(PAU_BLOCK_CQ_SM(0) + 0x080)
#define PAU_MCP_MISC_CERR_FIRST2		(PAU_BLOCK_CQ_SM(0) + 0x088)
#define PAU_SNP_MISC_CFG0			(PAU_BLOCK_CQ_SM(0) + 0x180)
#define   PAU_SNP_MISC_CFG0_ENABLE_PBUS		PPC_BIT(2)
#define   PAU_SNP_MISC_CFG0_OCAPI_MODE		PPC_BITMASK(32, 36)
#define   PAU_SNP_MISC_CFG0_OCAPI_C2		PPC_BITMASK(45, 49)
#define PAU_GPU_MEM_BAR(brk)			(PAU_BLOCK_CQ_SM(0) + 0x190 + (brk) * 8)
#define   PAU_GPU_MEM_BAR_ENABLE		PPC_BIT(0)
#define   PAU_GPU_MEM_BAR_ADDR_MASK		PPC_BITMASK(1, 35)
#define     PAU_GPU_MEM_BAR_ADDR		PPC_BITMASK(1, 21)
#define     PAU_GPU_MEM_BAR_SIZE		PPC_BITMASK(22, 35)
#define   PAU_GPU_MEM_BAR_POISON		PPC_BIT(45)
#define PAU_NTL_BAR(brk)			(PAU_BLOCK_CQ_SM(0) + 0x1b8 + (brk) * 8)
#define   PAU_NTL_BAR_ENABLE			PPC_BIT(0)
#define   PAU_NTL_BAR_ADDR			PPC_BITMASK(3, 35)
#define   PAU_NTL_BAR_SIZE			PPC_BITMASK(39, 43)
#define PAU_MMIO_BAR				(PAU_BLOCK_CQ_SM(0) + 0x1e0)
#define   PAU_MMIO_BAR_ENABLE			PPC_BIT(0)
#define   PAU_MMIO_BAR_ADDR			PPC_BITMASK(3, 27)
#define PAU_GENID_BAR				(PAU_BLOCK_CQ_SM(0) + 0x1e8)
#define   PAU_GENID_BAR_ENABLE			PPC_BIT(0)
#define   PAU_GENID_BAR_ADDR			PPC_BITMASK(3, 32)
#define PAU_MISC_MACHINE_ALLOC			(PAU_BLOCK_CQ_SM(0) + 0x268)
#define   PAU_MISC_MACHINE_ALLOC_ENABLE		PPC_BIT(0)

/* CQ_CTL block registers */
#define PAU_CTL_MISC_CFG0			(PAU_BLOCK_CQ_CTL + 0x000)
#define   PAU_CTL_MISC_CFG0_OTL_ENABLE		PPC_BITMASK(52, 56)
#define PAU_CTL_MISC_CFG2			(PAU_BLOCK_CQ_CTL + 0x010)
#define   PAU_CTL_MISC_CFG2_OCAPI_MODE		PPC_BITMASK(0, 4)
#define   PAU_CTL_MISC_CFG2_OCAPI_4		PPC_BITMASK(10, 14)
#define   PAU_CTL_MISC_CFG2_OCAPI_C2		PPC_BITMASK(15, 19)
#define   PAU_CTL_MISC_CFG2_OCAPI_AMO		PPC_BITMASK(20, 24)
#define   PAU_CTL_MISC_CFG2_OCAPI_MEM_OS_BIT	PPC_BITMASK(25, 29)
#define PAU_CTL_MISC_STATUS(brk)		(PAU_BLOCK_CQ_CTL + 0x060 + (brk) * 8)
#define   PAU_CTL_MISC_STATUS_AM_FENCED(brk)	(PPC_BITMASK(41, 42) << ((brk)*32))
#define PAU_CTL_MISC_GPU_MEM_BAR(brk)		(PAU_BLOCK_CQ_CTL + 0x070 + (brk) * 8)
#define PAU_CTL_MISC_MMIOPA_CONFIG(brk)		(PAU_BLOCK_CQ_CTL + 0x098 + (brk) * 8)
#define   PAU_CTL_MISC_MMIOPA_CONFIG_BAR_ADDR	PPC_BITMASK(1, 35)
#define   PAU_CTL_MISC_MMIOPA_CONFIG_BAR_SIZE	PPC_BITMASK(39, 43)
#define PAU_CTL_MISC_CERR_MESSAGE0		(PAU_BLOCK_CQ_CTL + 0x0C0)
#define PAU_CTL_MISC_CERR_MESSAGE1		(PAU_BLOCK_CQ_CTL + 0x0C8)
#define PAU_CTL_MISC_CERR_MESSAGE2		(PAU_BLOCK_CQ_CTL + 0x0D0)
#define PAU_CTL_MISC_CERR_FIRST0		(PAU_BLOCK_CQ_CTL + 0x0D8)
#define PAU_CTL_MISC_CERR_FIRST1		(PAU_BLOCK_CQ_CTL + 0x0E0)
#define PAU_CTL_MISC_FENCE_CTRL(brk)		(PAU_BLOCK_CQ_CTL + 0x108 + (brk) * 8)
#define   PAU_CTL_MISC_FENCE_REQUEST		PPC_BITMASK(0, 1)
#define PAU_CTL_MISC_CFG_ADDR(brk)		(PAU_BLOCK_CQ_CTL + 0x250 + (brk) * 8)
#define   PAU_CTL_MISC_CFG_ADDR_ENABLE		PPC_BIT(0)
#define   PAU_CTL_MISC_CFG_ADDR_STATUS		PPC_BITMASK(1, 3)
#define   PAU_CTL_MISC_CFG_ADDR_BUS_NBR		PPC_BITMASK(4, 11)
#define   PAU_CTL_MISC_CFG_ADDR_DEVICE_NBR	PPC_BITMASK(12, 16)
#define   PAU_CTL_MISC_CFG_ADDR_FUNCTION_NBR	PPC_BITMASK(17, 19)
#define   PAU_CTL_MISC_CFG_ADDR_REGISTER_NBR	PPC_BITMASK(20, 31)
#define   PAU_CTL_MISC_CFG_ADDR_TYPE		PPC_BIT(32)

/* CQ_DAT block registers */
#define PAU_DAT_MISC_CFG1			(PAU_BLOCK_CQ_DAT + 0x008)
#define   PAU_DAT_MISC_CFG1_OCAPI_MODE		PPC_BITMASK(40, 44)
#define PAU_DAT_MISC_CERR_ECC_HOLD		(PAU_BLOCK_CQ_DAT + 0x020)
#define PAU_DAT_MISC_CERR_ECC_MASK		(PAU_BLOCK_CQ_DAT + 0x028)
#define PAU_DAT_MISC_CERR_ECC_FIRST		(PAU_BLOCK_CQ_DAT + 0x030)

/* OTL block registers */
#define PAU_OTL_MISC_CFG0(brk)			(PAU_BLOCK_OTL(brk) + 0x000)
#define   PAU_OTL_MISC_CFG0_EN			PPC_BIT(0)
#define   PAU_OTL_MISC_CFG0_BLOCK_PE_HANDLE	PPC_BIT(1)
#define   PAU_OTL_MISC_CFG0_BRICKID		PPC_BITMASK(2, 3)
#define   PAU_OTL_MISC_CFG0_PE_MASK		PPC_BITMASK(4, 7)
#define   PAU_OTL_MISC_CFG0_ENABLE_4_0		PPC_BIT(51)
#define   PAU_OTL_MISC_CFG0_XLATE_RELEASE	PPC_BIT(62)
#define   PAU_OTL_MISC_CFG0_ENABLE_5_0		PPC_BIT(63)
#define PAU_OTL_MISC_ERR_RPT_HOLD0(brk)		(PAU_BLOCK_OTL(brk) + 0x030)
#define PAU_OTL_MISC_CFG_TLX_CREDITS(brk)	(PAU_BLOCK_OTL(brk) + 0x050)
#define   PAU_OTL_MISC_CFG_TLX_CREDITS_VC0	PPC_BITMASK(0, 7)
#define   PAU_OTL_MISC_CFG_TLX_CREDITS_VC1	PPC_BITMASK(8, 15)
#define   PAU_OTL_MISC_CFG_TLX_CREDITS_VC2	PPC_BITMASK(16, 23)
#define   PAU_OTL_MISC_CFG_TLX_CREDITS_VC3	PPC_BITMASK(24, 31)
#define   PAU_OTL_MISC_CFG_TLX_CREDITS_DCP0	PPC_BITMASK(32, 39)
#define   PAU_OTL_MISC_CFG_TLX_CREDITS_SPARE	PPC_BITMASK(40, 47)
#define   PAU_OTL_MISC_CFG_TLX_CREDITS_DCP2	PPC_BITMASK(48, 55)
#define   PAU_OTL_MISC_CFG_TLX_CREDITS_DCP3	PPC_BITMASK(56, 63)
#define PAU_OTL_MISC_CFG_TX(brk)		(PAU_BLOCK_OTL(brk) + 0x058)
#define   PAU_OTL_MISC_CFG_TX_TEMP1_EN		PPC_BIT(1)
#define   PAU_OTL_MISC_CFG_TX_TEMP2_EN		PPC_BIT(2)
#define   PAU_OTL_MISC_CFG_TX_TEMP3_EN		PPC_BIT(3)
#define   PAU_OTL_MISC_CFG_TX_TEMP_EN(n)	PPC_BIT(n)
#define   PAU_OTL_MISC_CFG_TX_DRDY_WAIT		PPC_BITMASK(5, 7)
#define   PAU_OTL_MISC_CFG_TX_TEMP0_RATE	PPC_BITMASK(8, 11)
#define   PAU_OTL_MISC_CFG_TX_TEMP1_RATE	PPC_BITMASK(12, 15)
#define   PAU_OTL_MISC_CFG_TX_TEMP2_RATE	PPC_BITMASK(16, 19)
#define   PAU_OTL_MISC_CFG_TX_TEMP3_RATE	PPC_BITMASK(20, 23)
#define   PAU_OTL_MISC_CFG_TX_TEMP_RATE(nib0, nib1)	PPC_BITMASK(nib0, nib1)
#define   PAU_OTL_MISC_CFG_TX_CRET_FREQ		PPC_BITMASK(32, 34)
#define PAU_OTL_MISC_OTL_REM0(brk)		(PAU_BLOCK_OTL(brk) + 0x068)
#define PAU_OTL_MISC_ERROR_SIG_RXI(brk)		(PAU_BLOCK_OTL(brk) + 0x070)
#define PAU_OTL_MISC_ERROR_SIG_RXO(brk)		(PAU_BLOCK_OTL(brk) + 0x078)
#define PAU_OTL_MISC_ERR_RPT_HOLD1(brk)		(PAU_BLOCK_OTL(brk) + 0x0B0)
#define PAU_OTL_MISC_CFG_TX2(brk)		(PAU_BLOCK_OTL(brk) + 0x0C0)
#define   PAU_OTL_MISC_CFG_TX2_SEND_EN		PPC_BIT(0)
#define PAU_OTL_MISC_PSL_DSISR_AN(brk)		(PAU_BLOCK_OTL_PSL(brk) + 0x000)
#define PAU_OTL_MISC_PSL_DAR_AN(brk)		(PAU_BLOCK_OTL_PSL(brk) + 0x008)
#define PAU_OTL_MISC_PSL_TFC_AN(brk)		(PAU_BLOCK_OTL_PSL(brk) + 0x010)
#define PAU_OTL_MISC_PSL_PEHANDLE_AN(brk)	(PAU_BLOCK_OTL_PSL(brk) + 0x018)

/* XSL block registers */
#define PAU_XSL_OSL_SPAP_AN(brk)		(PAU_BLOCK_XSL + 0x000 + (brk) * 8)
#define   PAU_XSL_OSL_SPAP_AN_EN		PPC_BIT(63)
#define PAU_XSL_GPU_MEM_BAR(brk)		(PAU_BLOCK_XSL + 0x0D0 + (brk) * 8)
#define PAU_XSL_WRAP_CFG			(PAU_BLOCK_XSL + 0x100)
#define   PAU_XSL_WRAP_CFG_CLOCK_ENABLE		PPC_BIT(0)
#define PAU_XSL_OSL_XLATE_CFG(brk)		(PAU_BLOCK_XSL + 0x040 + (brk) * 8)
#define   PAU_XSL_OSL_XLATE_CFG_AFU_DIAL	PPC_BIT(0)
#define   PAU_XSL_OSL_XLATE_CFG_OPENCAPI3	PPC_BIT(32)
#define PAU_XSL_OSL_CCINV			(PAU_BLOCK_XSL + 0x070)
#define   PAU_XSL_OSL_CCINV_REMOVE		PPC_BIT(15)
#define   PAU_XSL_OSL_CCINV_PENDING		PPC_BIT(16)
#define   PAU_XSL_OSL_CCINV_BRICK		PPC_BIT(47)
#define   PAU_XSL_OSL_CCINV_PE_HANDLE		PPC_BITMASK(48, 62)

/* XTS block registers */
#define PAU_XTS_CFG				(PAU_BLOCK_PAU_XTS + 0x020)
#define   PAU_XTS_CFG_OPENCAPI			PPC_BIT(15)
#define PAU_XTS_CFG2				(PAU_BLOCK_PAU_XTS + 0x028)
#define   PAU_XTS_CFG2_XSL2_ENA			PPC_BIT(55)
#define PAU_XTS_CFG3				(PAU_BLOCK_PAU_XTS + 0x068)
#define   PAU_XTS_CFG3_MMIOSD_OCAPI		PPC_BIT(5)
#define PAU_XTS_ATSD_HYP(n)			(PAU_BLOCK_PAU_XTS + 0x100 + (n) * 8)
#define   PAU_XTS_ATSD_HYP_MSR_HV		PPC_BIT(51)
#define   PAU_XTS_ATSD_HYP_LPARID		PPC_BITMASK(52, 63)

/* MISC block registers */
#define PAU_MISC_OPTICAL_IO_CONFIG		(PAU_BLOCK_PAU_MISC + 0x018)
#define   PAU_MISC_OPTICAL_IO_CONFIG_OTL	PPC_BITMASK(2, 3)
#define PAU_MISC_HOLD				(PAU_BLOCK_PAU_MISC + 0x020)
#define   PAU_MISC_HOLD_NDL_STALL		PPC_BITMASK(0, 3)
#define PAU_MISC_CONFIG				(PAU_BLOCK_PAU_MISC + 0x030)
#define   PAU_MISC_CONFIG_IPI_PS		PPC_BIT(11)
#define   PAU_MISC_CONFIG_IPI_PS_64K		1
#define   PAU_MISC_CONFIG_IPI_OS		PPC_BIT(12)
#define   PAU_MISC_CONFIG_IPI_OS_AIX		0
#define   PAU_MISC_CONFIG_OC_MODE		PPC_BIT(16)
#define PAU_MISC_FREEZE_1_CONFIG		(PAU_BLOCK_PAU_MISC + 0x048)
#define PAU_MISC_FENCE_1_CONFIG			(PAU_BLOCK_PAU_MISC + 0x058)
#define PAU_MISC_INT_1_CONFIG			(PAU_BLOCK_PAU_MISC + 0x068)
#define PAU_MISC_INT_BAR			(PAU_BLOCK_PAU_MISC + 0x098)
#define   PAU_MISC_INT_BAR_ADDR			PPC_BITMASK(0, 39)
#define PAU_MISC_FENCE_STATE			(PAU_BLOCK_PAU_MISC + 0x0B0)
#define   PAU_MISC_FENCE_STATE_CLEAR(brk)	PPC_BIT(0 + (brk))
#define   PAU_MISC_FENCE_STATE_SET(brk)		PPC_BIT(12 + (brk))
#define PAU_MISC_BDF2PE_CFG(n)			(PAU_BLOCK_PAU_MISC + 0x100 + (n) * 8)
#define   PAU_MISC_BDF2PE_CFG_ENABLE		PPC_BIT(0)
#define   PAU_MISC_BDF2PE_CFG_PE		PPC_BITMASK(4, 7)
#define   PAU_MISC_BDF2PE_CFG_BDF		PPC_BITMASK(8, 23)
#define PAU_MISC_INT_2_CONFIG			(PAU_BLOCK_PAU_MISC + 0x408)
#define   PAU_MISC_INT_2_CONFIG_XFAULT_2_5(n)	PPC_BIT(0 + (n))
#define   PAU_MISC_INT_2_CONFIG_XFAULT_0_1(n)	PPC_BIT(54 + (n))

/* PAU_XTS_ATSD block registers */
#define PAU_XTS_ATSD_LAUNCH(n)			(PAU_BLOCK_PAU_XTS_ATSD(n) + 0x000)
#define PAU_XTS_ATSD_MAX			16

#endif /* __PAU_REGS_H */
