/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include <libwpd-stream/libwpd-stream.h>

#include <libe-book/libe-book.h>

#include "libebook_utils.h"

namespace libebook
{

namespace
{

enum EBOOKType
{
  UNKNOWN,
  FB2,
  IMP,
  LRF,
  PDB,
  PLKR,
  PML,
  QIOO,
  TCR,
  TD,
  ZTXT,
  ZVR
};

bool isSupportedType(WPXInputStream *const input, EBOOKType &type)
{
  if (IMPDocument::isSupported(input))
  {
    type = IMP;
    return true;
  }
  else if (LRFDocument::isSupported(input))
  {
    type = LRF;
    return true;
  }
  else if (PDBDocument::isSupported(input))
  {
    type = PDB;
    return true;
  }
  else if (PLKRDocument::isSupported(input))
  {
    type = PLKR;
    return true;
  }
  else if (PMLDocument::isSupported(input))
  {
    type = PML;
    return true;
  }
  else if (TDDocument::isSupported(input))
  {
    type = TD;
    return true;
  }
  else if (ZTXTDocument::isSupported(input))
  {
    type = ZTXT;
    return true;
  }
  else if (FB2Document::isSupported(input))
  {
    type = FB2;
    return true;
  }
  else if (ZVRDocument::isSupported(input))
  {
    type = ZVR;
    return true;
  }
  else if (TCRDocument::isSupported(input))
  {
    type = TCR;
    return true;
  }
  else if (QIOODocument::isSupported(input))
  {
    type = QIOO;
    return true;
  }

  type = UNKNOWN;
  return false;
}

}

bool EBOOKDocument::isSupported(WPXInputStream *const input) try
{
  EBOOKType dummy;
  return isSupportedType(input, dummy);
}
catch (...)
{
  return false;
}

bool EBOOKDocument::parse(WPXInputStream *const input, WPXDocumentInterface *const document) try
{
  EBOOKType type;
  if (!isSupportedType(input, type))
    return false;

  input->seek(0, WPX_SEEK_SET);

  switch (type)
  {
  case FB2 :
    return FB2Document::parse(input, document);
  case IMP :
    return IMPDocument::parse(input, document);
  case LRF :
    return LRFDocument::parse(input, document);
  case PDB :
    return PDBDocument::parse(input, document);
  case PLKR :
    return PLKRDocument::parse(input, document);
  case PML :
    return PMLDocument::parse(input, document);
  case QIOO :
    return QIOODocument::parse(input, document);
  case TCR :
    return TCRDocument::parse(input, document);
  case TD :
    return TDDocument::parse(input, document);
  case ZTXT :
    return ZTXTDocument::parse(input, document);
  case ZVR :
    return ZVRDocument::parse(input, document);
  default :
    EBOOK_DEBUG_MSG(("unhandled document type %d\n", type));
    break;
  }

  return false;
}
catch (...)
{
  return false;
}

} // namespace libebook

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
