/*
 * Copyright (C) 2019 Andreas Steffen
 *
 * Copyright (C) secunet Security Networks AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the Licenseor (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be usefulbut
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <crypto/crypto_tester.h>

/**
 * NIST SP 800-90A DRBG HMAC Validation System (DRBGVS)
 */

/**
 * SHA-1 test case 1 - count 0
 */
drbg_test_vector_t drbg_hmac_sha1_1 = {
	.type = DRBG_HMAC_SHA1, .strength = 128,
	{ NULL, 0 },
	chunk_from_chars(0x79, 0x34, 0x9b, 0xbf, 0x7c, 0xdd, 0xa5, 0x79, 0x95, 0x57,
					 0x86, 0x66, 0x21, 0xc9, 0x13, 0x83, 0x11, 0x46, 0x73, 0x3a,
					 0xbf, 0x8c, 0x35, 0xc8, 0xc7, 0x21, 0x5b, 0x5b, 0x96, 0xc4,
					 0x8e, 0x9b, 0x33, 0x8c, 0x74, 0xe3, 0xe9, 0x9d, 0xfe, 0xdf),
	chunk_from_chars(0xc6, 0xa1, 0x6a, 0xb8, 0xd4, 0x20, 0x70, 0x6f, 0x0f, 0x34,
					 0xab, 0x7f, 0xec, 0x5a, 0xdc, 0xa9, 0xd8, 0xca, 0x3a, 0x13,
					 0x3e, 0x15, 0x9c, 0xa6, 0xac, 0x43, 0xc6, 0xf8, 0xa2, 0xbe,
					 0x22, 0x83, 0x4a, 0x4c, 0x0a, 0x0a, 0xff, 0xb1, 0x0d, 0x71,
					 0x94, 0xf1, 0xc1, 0xa5, 0xcf, 0x73, 0x22, 0xec, 0x1a, 0xe0,
					 0x96, 0x4e, 0xd4, 0xbf, 0x12, 0x27, 0x46, 0xe0, 0x87, 0xfd,
					 0xb5, 0xb3, 0xe9, 0x1b, 0x34, 0x93, 0xd5, 0xbb, 0x98, 0xfa,
					 0xed, 0x49, 0xe8, 0x5f, 0x13, 0x0f, 0xc8, 0xa4, 0x59, 0xb7)
};

/**
 * SHA-1 test case 3 - count 0
 */
drbg_test_vector_t drbg_hmac_sha1_2 = {
	.type = DRBG_HMAC_SHA1, .strength = 128,
	chunk_from_chars(0xb2, 0x4e, 0x39, 0x2c, 0xb1, 0xf3, 0xc1, 0x8a, 0xf2, 0xcb,
					 0x50, 0xfe, 0xac, 0x73, 0x3e, 0x32),
	chunk_from_chars(0x11, 0xc0, 0xa7, 0xe1, 0x47, 0x2c, 0xec, 0x70, 0xfa, 0x8c,
					 0x1c, 0xa1, 0x57, 0x59, 0xac, 0x5b, 0xb1, 0xc7, 0x3c, 0x22,
					 0xdb, 0x39, 0xcd, 0x7b, 0xc6, 0xab, 0x59, 0xff, 0x70, 0x8a,
					 0x5c, 0x1f, 0x59, 0x8e, 0x75, 0xdf, 0x06, 0x0e, 0x19, 0x81),
	chunk_from_chars(0x07, 0x0e, 0x60, 0x3c, 0xd4, 0x8d, 0x56, 0x43, 0x0a, 0x5a,
					 0xb4, 0x61, 0xa7, 0x51, 0xec, 0x2a, 0x4a, 0x6a, 0xa6, 0xfb,
					 0x6e, 0xe5, 0x2e, 0xfe, 0x9a, 0x41, 0xe4, 0x61, 0x1e, 0xaf,
					 0xdf, 0xc9, 0x57, 0x18, 0x4b, 0x47, 0xbb, 0xb0, 0x17, 0xe4,
					 0x84, 0xac, 0x34, 0xc7, 0xde, 0x56, 0xcd, 0x78, 0x13, 0xfe,
					 0xb3, 0x01, 0xb5, 0xbe, 0xfc, 0xe5, 0x73, 0xad, 0x0a, 0x25,
					 0x4e, 0x6c, 0xfe, 0x35, 0xb7, 0x7c, 0x30, 0xbe, 0x6b, 0x7c,
					 0xb5, 0xe7, 0xef, 0xa7, 0x28, 0x13, 0xc7, 0x54, 0x6b, 0xa5)
};

/**
 * SHA-1 test case 5 - count 0
 */
drbg_test_vector_t drbg_hmac_sha1_3 = {
	.type = DRBG_HMAC_SHA1, .strength = 128,
	{ NULL, 0 },
	chunk_from_chars(0xe5, 0xa8, 0x13, 0x57, 0xb9, 0x12, 0x15, 0xdd, 0xa0, 0xa0,
					 0x98, 0x6a, 0x3f, 0xf5, 0x12, 0x3f, 0x9c, 0x18, 0x38, 0xd7,
					 0x36, 0x0e, 0x67, 0x4b, 0x59, 0x2d, 0xd2, 0x32, 0xac, 0x83,
					 0xdb, 0x36, 0xab, 0xba, 0xcf, 0xb8, 0xc6, 0x40, 0xdc, 0x60),
	chunk_from_chars(0xa4, 0xd1, 0x84, 0x3c, 0x7a, 0xf2, 0x08, 0xf7, 0x70, 0xf8,
					 0xb5, 0xac, 0xf6, 0x45, 0x28, 0x86, 0x6d, 0x51, 0xe7, 0x31,
					 0xed, 0x5e, 0xbc, 0x75, 0x6e, 0x81, 0xef, 0xe8, 0xfb, 0x8f,
					 0x9c, 0x9a, 0xf8, 0x9d, 0x52, 0xe8, 0xe8, 0xd1, 0xc0, 0x14,
					 0x1e, 0xbe, 0xbc, 0x18, 0xb1, 0xca, 0x78, 0xc7, 0x8a, 0x2f,
					 0x21, 0xcb, 0x90, 0x9d, 0x5b, 0xac, 0x3b, 0x6a, 0xe6, 0x0c,
					 0x4a, 0x2f, 0xf4, 0x17, 0x6b, 0x14, 0x90, 0x5e, 0x4a, 0xfa,
					 0x3b, 0xa9, 0xe4, 0x58, 0x21, 0x6d, 0x57, 0x20, 0xce, 0x83)
};

/**
 * SHA-1 test case 7 - count 0
 */
drbg_test_vector_t drbg_hmac_sha1_4 = {
	.type = DRBG_HMAC_SHA1, .strength = 128,
	chunk_from_chars(0x12, 0x91, 0x26, 0xc1, 0x6d, 0x99, 0xa6, 0x84, 0xf3, 0xcb,
					 0x47, 0xe7, 0xff, 0xb2, 0x07, 0xac),
	chunk_from_chars(0x12, 0xbf, 0xb2, 0xbf, 0xf5, 0x78, 0x11, 0x28, 0x96, 0x7e,
					 0x9c, 0xe4, 0xd4, 0x29, 0x45, 0x3c, 0x5a, 0xab, 0x2b, 0xe4,
					 0xe9, 0x2e, 0xc8, 0x55, 0xfa, 0x8b, 0x0d, 0xa2, 0xcd, 0x50,
					 0x31, 0x43, 0x3d, 0x64, 0x31, 0x0b, 0xa4, 0x4d, 0x47, 0xe4),
	chunk_from_chars(0x76, 0x6c, 0x4a, 0x16, 0x70, 0xde, 0xe6, 0x3e, 0xe3, 0xaa,
					 0xfb, 0x60, 0x39, 0xe8, 0x6e, 0xc5, 0xca, 0x28, 0x90, 0x53,
					 0x19, 0x74, 0xe7, 0x0f, 0xed, 0xb3, 0xc0, 0x21, 0x8b, 0x08,
					 0xfa, 0xca, 0x07, 0xdd, 0xfa, 0xaf, 0xf0, 0xa7, 0x7f, 0x0a,
					 0xe4, 0x77, 0x62, 0xb0, 0xe1, 0xb3, 0x1c, 0x5f, 0xb0, 0x36,
					 0xdc, 0xa4, 0x13, 0xfb, 0x96, 0xf6, 0xe0, 0xec, 0x6a, 0xb7,
					 0xdb, 0xcb, 0xb5, 0x58, 0xaa, 0x5d, 0x94, 0xf4, 0x29, 0xc3,
					 0x8e, 0x0f, 0xd7, 0xb3, 0x9f, 0x12, 0x72, 0x1c, 0x99, 0x35)
};

/**
 * SHA-1 test case 9 - count 0
 */
drbg_test_vector_t drbg_hmac_sha1_5 = {
	.type = DRBG_HMAC_SHA1, .strength = 128,
	{ NULL, 0 },
	chunk_from_chars(0x23, 0xbe, 0xc0, 0x2f, 0x58, 0xa3, 0xa8, 0xd9, 0xd9, 0xfb,
					 0x36, 0x8d, 0xd2, 0x62, 0x8c, 0x56, 0xee, 0xda, 0xa5, 0x0c,
					 0xea, 0x71, 0x25, 0x46, 0x4d, 0x46, 0x6a, 0x36, 0x08, 0xa4,
					 0xac, 0xdb, 0xd6, 0xa5, 0xa7, 0xa7, 0x53, 0xa1, 0x5f, 0x9b),
	chunk_from_chars(0x98, 0x7f, 0x5c, 0xa0, 0xb5, 0x9d, 0x6f, 0xd1, 0xde, 0x8d,
					 0x7e, 0xbb, 0x07, 0xc0, 0xd9, 0x88, 0xe2, 0x46, 0xc3, 0x0f,
					 0xea, 0x5d, 0xbf, 0x5c, 0xdd, 0x5b, 0xbe, 0xf7, 0xca, 0x45,
					 0x30, 0x18, 0xe2, 0xba, 0xcb, 0xf7, 0x97, 0x7f, 0x49, 0x4c,
					 0xa9, 0xbf, 0x5c, 0x12, 0x80, 0x55, 0xe4, 0x1a, 0xb8, 0xf7,
					 0xda, 0x93, 0xd8, 0xbb, 0xfa, 0xf9, 0x04, 0x4b, 0x09, 0x4a,
					 0x6e, 0xdd, 0x0d, 0x72, 0x96, 0x3f, 0x83, 0xa3, 0xd1, 0xf4,
					 0x17, 0xfc, 0x8f, 0x5b, 0x72, 0x39, 0x11, 0xec, 0xa0, 0x62)
};

/**
 * SHA-1 test case 11 - count 0
 */
drbg_test_vector_t drbg_hmac_sha1_6 = {
	.type = DRBG_HMAC_SHA1, .strength = 128,
	chunk_from_chars(0x99, 0x9d, 0x6d, 0xa3, 0xb5, 0xdd, 0x45, 0x2c, 0x4b, 0xd9,
					 0x72, 0x01, 0x7b, 0x3f, 0xa1, 0x35),
	chunk_from_chars(0x91, 0x11, 0xd8, 0xc3, 0x3f, 0x0f, 0x58, 0x53, 0x47, 0xd5,
					 0x13, 0x54, 0xab, 0xa7, 0xaa, 0x15, 0xbd, 0x05, 0xff, 0x4f,
					 0xcb, 0x05, 0x8d, 0xe3, 0x84, 0x47, 0xd0, 0x6c, 0x5f, 0xdb,
					 0x5c, 0xf2, 0xad, 0xd7, 0x7b, 0xc5, 0x69, 0xe8, 0x94, 0x4b),
	chunk_from_chars(0x69, 0x7d, 0x46, 0x8a, 0xcc, 0xc5, 0x26, 0x55, 0x81, 0x98,
					 0xa0, 0xd0, 0xaf, 0x81, 0x85, 0x0b, 0xb9, 0x91, 0x0b, 0x70,
					 0x51, 0x8c, 0x86, 0xa7, 0xf9, 0x9b, 0x7b, 0x82, 0x65, 0x79,
					 0x66, 0xa0, 0x5c, 0x73, 0x89, 0x0c, 0x3a, 0xf4, 0x81, 0x52,
					 0x04, 0x5e, 0x66, 0x9a, 0xfd, 0xca, 0x08, 0x07, 0x39, 0x72,
					 0x1f, 0xc2, 0xbd, 0xd4, 0x9f, 0x90, 0xa1, 0xe5, 0x65, 0xa6,
					 0x90, 0x69, 0x23, 0x6e, 0x83, 0xe5, 0xe1, 0xde, 0x6c, 0x91,
					 0xbe, 0x64, 0x02, 0x0a, 0x9a, 0x95, 0xa2, 0xb7, 0x49, 0x82)
};

/**
 * SHA-256 test case 1 - count 0
 */
drbg_test_vector_t drbg_hmac_sha256_1 = {
	.type = DRBG_HMAC_SHA256, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0x06, 0x03, 0x2c, 0xd5, 0xee, 0xd3, 0x3f, 0x39, 0x26, 0x5f,
	                 0x49, 0xec, 0xb1, 0x42, 0xc5, 0x11, 0xda, 0x9a, 0xff, 0x2a,
	                 0xf7, 0x12, 0x03, 0xbf, 0xfa, 0xf3, 0x4a, 0x9c, 0xa5, 0xbd,
	                 0x9c, 0x0d, 0x0e, 0x66, 0xf7, 0x1e, 0xdc, 0x43, 0xe4, 0x2a,
					 0x45, 0xad, 0x3c, 0x6f, 0xc6, 0xcd, 0xc4, 0xdf, 0x01, 0x92,
					 0x0a, 0x4e, 0x66, 0x9e, 0xd3, 0xa8, 0x5a, 0xe8, 0xa3, 0x3b,
					 0x35, 0xa7, 0x4a, 0xd7, 0xfb, 0x2a, 0x6b, 0xb4, 0xcf, 0x39,
					 0x5c, 0xe0, 0x03, 0x34, 0xa9, 0xc9, 0xa5, 0xa5, 0xd5, 0x52),
	chunk_from_chars(0x76, 0xfc, 0x79, 0xfe, 0x9b, 0x50, 0xbe, 0xcc, 0xc9, 0x91,
					 0xa1, 0x1b, 0x56, 0x35, 0x78, 0x3a, 0x83, 0x53, 0x6a, 0xdd,
					 0x03, 0xc1, 0x57, 0xfb, 0x30, 0x64, 0x5e, 0x61, 0x1c, 0x28,
					 0x98, 0xbb, 0x2b, 0x1b, 0xc2, 0x15, 0x00, 0x02, 0x09, 0x20,
					 0x8c, 0xd5, 0x06, 0xcb, 0x28, 0xda, 0x2a, 0x51, 0xbd, 0xb0,
					 0x38, 0x26, 0xaa, 0xf2, 0xbd, 0x23, 0x35, 0xd5, 0x76, 0xd5,
					 0x19, 0x16, 0x08, 0x42, 0xe7, 0x15, 0x8a, 0xd0, 0x94, 0x9d,
					 0x1a, 0x9e, 0xc3, 0xe6, 0x6e, 0xa1, 0xb1, 0xa0, 0x64, 0xb0,
					 0x05, 0xde, 0x91, 0x4e, 0xac, 0x2e, 0x9d, 0x4f, 0x2d, 0x72,
					 0xa8, 0x61, 0x6a, 0x80, 0x22, 0x54, 0x22, 0x91, 0x82, 0x50,
					 0xff, 0x66, 0xa4, 0x1b, 0xd2, 0xf8, 0x64, 0xa6, 0xa3, 0x8c,
					 0xc5, 0xb6, 0x49, 0x9d, 0xc4, 0x3f, 0x7f, 0x2b, 0xd0, 0x9e,
					 0x1e, 0x0f, 0x8f, 0x58, 0x85, 0x93, 0x51, 0x24)
};

/**
 * SHA-256 test case 3 - count 0
 */
drbg_test_vector_t drbg_hmac_sha256_2 = {
	.type = DRBG_HMAC_SHA256, .strength = 256,
	chunk_from_chars(0xf2, 0xe5, 0x8f, 0xe6, 0x0a, 0x3a, 0xfc, 0x59, 0xda, 0xd3,
					 0x75, 0x95, 0x41, 0x5f, 0xfd, 0x31, 0x8c, 0xcf, 0x69, 0xd6,
					 0x77, 0x80, 0xf6, 0xfa, 0x07, 0x97, 0xdc, 0x9a, 0xa4, 0x3e,
					 0x14, 0x4c),
	chunk_from_chars(0xfa, 0x0e, 0xe1, 0xfe, 0x39, 0xc7, 0xc3, 0x90, 0xaa, 0x94,
					 0x15, 0x9d, 0x0d, 0xe9, 0x75, 0x64, 0x34, 0x2b, 0x59, 0x17,
					 0x77, 0xf3, 0xe5, 0xf6, 0xa4, 0xba, 0x2a, 0xea, 0x34, 0x2e,
					 0xc8, 0x40, 0xdd, 0x08, 0x20, 0x65, 0x5c, 0xb2, 0xff, 0xdb,
					 0x0d, 0xa9, 0xe9, 0x31, 0x0a, 0x67, 0xc9, 0xe5, 0xe0, 0x62,
					 0x9b, 0x6d, 0x79, 0x75, 0xdd, 0xfa, 0x96, 0xa3, 0x99, 0x64,
					 0x87, 0x40, 0xe6, 0x0f, 0x1f, 0x95, 0x57, 0xdc, 0x58, 0xb3,
					 0xd7, 0x41, 0x5f, 0x9b, 0xa9, 0xd4, 0xdb, 0xb5, 0x01, 0xf6),
	chunk_from_chars(0xf9, 0x2d, 0x4c, 0xf9, 0x9a, 0x53, 0x5b, 0x20, 0x22, 0x2a,
	 				 0x52, 0xa6, 0x8d, 0xb0, 0x4c, 0x5a, 0xf6, 0xf5, 0xff, 0xc7,
	 				 0xb6, 0x6a, 0x47, 0x3a, 0x37, 0xa2, 0x56, 0xbd, 0x8d, 0x29,
	 				 0x8f, 0x9b, 0x4a, 0xa4, 0xaf, 0x7e, 0x8d, 0x18, 0x1e, 0x02,
					 0x36, 0x79, 0x03, 0xf9, 0x3b, 0xdb, 0x74, 0x4c, 0x6c, 0x2f,
					 0x3f, 0x34, 0x72, 0x62, 0x6b, 0x40, 0xce, 0x9b, 0xd6, 0xa7,
					 0x0e, 0x7b, 0x8f, 0x93, 0x99, 0x2a, 0x16, 0xa7, 0x6f, 0xab,
					 0x6b, 0x5f, 0x16, 0x25, 0x68, 0xe0, 0x8e, 0xe6, 0xc3, 0xe8,
					 0x04, 0xae, 0xfd, 0x95, 0x2d, 0xdd, 0x3a, 0xcb, 0x79, 0x1c,
					 0x50, 0xf2, 0xad, 0x69, 0xe9, 0xa0, 0x40, 0x28, 0xa0, 0x6a,
					 0x9c, 0x01, 0xd3, 0xa6, 0x2a, 0xca, 0x2a, 0xaf, 0x6e, 0xfe,
					 0x69, 0xed, 0x97, 0xa0, 0x16, 0x21, 0x3a, 0x2d, 0xd6, 0x42,
					 0xb4, 0x88, 0x67, 0x64, 0x07, 0x2d, 0x9c, 0xbe)
};

/**
 * SHA-256 test case 5 - count 0
 */
drbg_test_vector_t drbg_hmac_sha256_3 = {
	.type = DRBG_HMAC_SHA256, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0xff, 0x0c, 0xdd, 0x55, 0x5c, 0x60, 0x46, 0x47, 0x60, 0xb2,
					 0x89, 0xb7, 0xbc, 0x1f, 0x81, 0x1a, 0x41, 0xff, 0xf7, 0x2d,
					 0xe5, 0x90, 0x83, 0x85, 0x8c, 0x02, 0x0a, 0x10, 0x53, 0xbd,
					 0xc7, 0x4a, 0x7b, 0xc0, 0x99, 0x28, 0x5a, 0xd5, 0x62, 0x19,
					 0x93, 0xb6, 0x39, 0xc4, 0xa9, 0x4c, 0x37, 0x6b, 0x14, 0xfc,
					 0x6c, 0x9b, 0x17, 0x8d, 0xb6, 0x44, 0xa8, 0xcd, 0x71, 0x30,
					 0xa4, 0xcf, 0x05, 0x16, 0x78, 0xc8, 0xf4, 0xfa, 0x8f, 0x24,
					 0xc2, 0x7b, 0x0a, 0x53, 0x13, 0x38, 0xa5, 0xce, 0x85, 0x89),
	chunk_from_chars(0x2f, 0x26, 0x20, 0x34, 0x7b, 0xdd, 0xca, 0xa2, 0x94, 0x36,
					 0x85, 0x34, 0x6b, 0xbf, 0x31, 0xc4, 0x40, 0x81, 0xf8, 0x66,
					 0x5f, 0x3d, 0xdb, 0x2b, 0x42, 0xae, 0x14, 0x16, 0xa7, 0x4c,
					 0x4b, 0x77, 0xfa, 0xb3, 0xfa, 0x19, 0xae, 0xec, 0xc5, 0x47,
					 0xe7, 0x6c, 0x8c, 0xbe, 0x6a, 0xd1, 0xf1, 0x00, 0xa3, 0xfc,
					 0x8b, 0x2c, 0xe2, 0xa1, 0xea, 0x3a, 0x3d, 0xd7, 0xcf, 0xad,
					 0x46, 0xc1, 0xb2, 0x78, 0x30, 0xb9, 0x40, 0xba, 0x18, 0xd0,
					 0x9e, 0x9b, 0x7f, 0xa9, 0x02, 0xbb, 0x76, 0x06, 0x69, 0xb1,
					 0x73, 0x5c, 0xc7, 0xb7, 0xbd, 0x39, 0x05, 0x2d, 0xa7, 0xf2,
					 0x62, 0x6f, 0xa8, 0x70, 0x00, 0xcf, 0xfa, 0xda, 0x41, 0x00,
					 0x19, 0xd0, 0x53, 0x38, 0x6a, 0xd8, 0x08, 0xbd, 0x3c, 0x0c,
					 0xfc, 0xf5, 0x6b, 0x91, 0x87, 0x9e, 0xb8, 0xd3, 0xf9, 0x32,
					 0xee, 0x2d, 0x18, 0x5e, 0x54, 0xf3, 0x1b, 0x74)
};

/**
 * SHA-256 test case 7 - count 0
 */
drbg_test_vector_t drbg_hmac_sha256_4 = {
	.type = DRBG_HMAC_SHA256, .strength = 256,
	chunk_from_chars(0x40, 0x93, 0x3f, 0xdc, 0xce, 0x41, 0x59, 0xb0, 0x95, 0x51,
					 0x11, 0xf8, 0x44, 0x47, 0x1b, 0x0d, 0xb8, 0x5b, 0x73, 0xbd,
					 0xd2, 0xb7, 0x8c, 0x46, 0x8d, 0xd3, 0x9e, 0x2a, 0x9b, 0x29,
					 0xae, 0xf2),
	chunk_from_chars(0x28, 0xba, 0x1a, 0x66, 0x16, 0x32, 0xef, 0xc8, 0xec, 0xce,
					 0xd5, 0xf5, 0x1b, 0x79, 0x13, 0x00, 0xfb, 0x3b, 0x55, 0xb0,
					 0x5d, 0x04, 0x17, 0x08, 0x63, 0x8d, 0xe4, 0xbe, 0xb7, 0x57,
					 0xa9, 0xe5, 0x76, 0x82, 0x87, 0x96, 0xaf, 0xf0, 0x7f, 0x55,
					 0x79, 0x5c, 0xb5, 0x47, 0x13, 0xc7, 0x7e, 0xd4, 0xa5, 0xf5,
					 0x42, 0xb0, 0x4a, 0xaa, 0x5d, 0xbc, 0x93, 0x1e, 0x47, 0x01,
					 0x9f, 0xeb, 0x38, 0x96, 0x26, 0x16, 0xc5, 0x7a, 0xf0, 0x9b,
					 0x7c, 0x1d, 0xf8, 0x3f, 0x2b, 0x86, 0x0f, 0xf7, 0x65, 0x86),
	chunk_from_chars(0x65, 0xe5, 0xaa, 0x47, 0xb3, 0x85, 0xf1, 0xea, 0x42, 0xb2,
					 0x31, 0xb9, 0xfe, 0x74, 0x42, 0x53, 0xb8, 0x59, 0x88, 0x59,
					 0xd7, 0x01, 0x1e, 0x52, 0x5f, 0x5a, 0x2a, 0x1a, 0xd3, 0x2a,
					 0x97, 0x2a, 0x85, 0x08, 0x02, 0xc6, 0x0a, 0x2b, 0xe1, 0x9b,
					 0xe2, 0x70, 0x06, 0x3a, 0x3c, 0xfb, 0xea, 0xae, 0x95, 0x4f,
					 0x10, 0xb1, 0x22, 0x35, 0x2d, 0xe6, 0xa0, 0x8a, 0xc4, 0x10,
					 0xe0, 0x99, 0x16, 0x53, 0xaa, 0xb2, 0x71, 0xb3, 0x60, 0xfe,
					 0x91, 0x91, 0xcf, 0x5a, 0xdd, 0xcc, 0xcc, 0xed, 0x8c, 0x4a,
					 0xcf, 0xb6, 0x14, 0x57, 0x04, 0x99, 0x92, 0x98, 0x8f, 0xd7,
					 0xa9, 0xac, 0xca, 0x1f, 0x1b, 0xca, 0x35, 0xf1, 0x47, 0x58,
					 0x13, 0x69, 0x4a, 0x39, 0x98, 0x8e, 0x5f, 0xac, 0x9f, 0x4a,
					 0xc0, 0x57, 0x22, 0x86, 0xbc, 0x46, 0x25, 0x82, 0xad, 0x0a,
					 0xf7, 0x8a, 0xb3, 0xb8, 0x5e, 0xc1, 0x7a, 0x25)
};

/**
 * SHA-256 test case 9 - count 0
 */
drbg_test_vector_t drbg_hmac_sha256_5 = {
	.type = DRBG_HMAC_SHA256, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0x6a, 0xe8, 0x03, 0x03, 0x29, 0x23, 0x91, 0x33, 0x5b, 0xf9,
					 0xc9, 0x38, 0x7f, 0xbd, 0x3b, 0xf6, 0x15, 0x75, 0x6c, 0x9c,
					 0x27, 0xc3, 0x47, 0x8c, 0x87, 0xe2, 0x60, 0xcf, 0x97, 0xd4,
					 0x71, 0x10, 0x01, 0xe1, 0x62, 0x47, 0xdd, 0x4c, 0xae, 0x64,
					 0x99, 0x33, 0x7d, 0x82, 0x78, 0x4e, 0xa5, 0x7f, 0x03, 0x57,
					 0x02, 0xef, 0x4e, 0x11, 0x2b, 0x17, 0x31, 0x12, 0xc5, 0x85,
					 0x1d, 0x07, 0xb2, 0x79, 0x30, 0x98, 0x63, 0x74, 0x0d, 0x38,
					 0xd0, 0xd0, 0x72, 0x02, 0x23, 0xe2, 0x40, 0x17, 0xbb, 0xc0),
	chunk_from_chars(0xcf, 0x43, 0x15, 0x59, 0x8f, 0xcd, 0x6a, 0xf1, 0x31, 0x55,
					 0x18, 0xc4, 0xbf, 0xba, 0xc0, 0x54, 0x0c, 0x58, 0x96, 0x35,
					 0x27, 0x35, 0x48, 0xa7, 0xb5, 0x07, 0xe7, 0xd2, 0xe6, 0x85,
					 0xe5, 0x94, 0x7b, 0x87, 0xae, 0x25, 0x7e, 0x58, 0xfa, 0xf2,
					 0x14, 0xf2, 0xb5, 0x8e, 0xd1, 0x0c, 0x3b, 0xd3, 0x5f, 0x75,
					 0xf6, 0xc3, 0x5d, 0xd6, 0xd4, 0x41, 0xc9, 0x3b, 0xcd, 0x42,
					 0xe7, 0x17, 0x20, 0x10, 0x26, 0x31, 0xb1, 0xa6, 0xa4, 0xba,
					 0x24, 0x7c, 0x17, 0x5e, 0xd8, 0x00, 0xcf, 0xca, 0x6e, 0x1e,
					 0x83, 0x9b, 0x5a, 0xa9, 0x07, 0x60, 0x4c, 0xcf, 0xe6, 0xf9,
					 0x84, 0xf6, 0x82, 0x2e, 0x00, 0x1a, 0xb0, 0x2d, 0xd6, 0x63,
					 0x49, 0x64, 0xf7, 0x89, 0xcb, 0x10, 0x7a, 0x97, 0x73, 0x46,
					 0x69, 0x3f, 0x32, 0x44, 0xc8, 0x95, 0xe8, 0x40, 0xdf, 0xa0,
					 0xed, 0xf7, 0xf1, 0x4d, 0xc6, 0x1d, 0x79, 0x4f)
};

/**
 * SHA-256 test case 11 - count 0
 */
drbg_test_vector_t drbg_hmac_sha256_6 = {
	.type = DRBG_HMAC_SHA256, .strength = 256,
	chunk_from_chars(0x9f, 0x16, 0x99, 0xc9, 0x9d, 0x60, 0xb0, 0x85, 0xbc, 0x61,
					 0xcb, 0x11, 0x0e, 0xf8, 0xab, 0x59, 0x0d, 0x82, 0xa9, 0x70,
					 0x02, 0x1c, 0x3c, 0x6a, 0x5d, 0x48, 0x02, 0x1c, 0x45, 0xde,
					 0x49, 0x56),
	chunk_from_chars(0x63, 0x3d, 0x32, 0xe3, 0x00, 0x5f, 0x78, 0x11, 0x47, 0x23,
					 0xb3, 0xea, 0x5a, 0xc1, 0x21, 0xba, 0x74, 0xaa, 0x00, 0xc5,
					 0x2d, 0x93, 0x96, 0x67, 0xe3, 0x0c, 0x33, 0x51, 0xb3, 0x85,
					 0x49, 0xf7, 0x37, 0xaf, 0xff, 0x50, 0x4a, 0x2d, 0x8a, 0xc1,
					 0x68, 0xc6, 0x8e, 0x24, 0xd0, 0xfe, 0x66, 0xf6, 0x3e, 0x33,
					 0x47, 0xc5, 0x47, 0xf1, 0x7f, 0x4d, 0x0b, 0x9f, 0x46, 0x40,
					 0x5a, 0x54, 0xee, 0xdd, 0x7e, 0x98, 0x0d, 0x06, 0xa2, 0x15,
					 0xec, 0x15, 0xe8, 0x93, 0x16, 0xab, 0x74, 0x3b, 0x75, 0x47),
	chunk_from_chars(0x6e, 0x38, 0xe8, 0x29, 0x62, 0xd7, 0x07, 0xce, 0x9a, 0x6a,
					 0xc3, 0x83, 0xa7, 0x38, 0xa7, 0x48, 0xf9, 0x75, 0xeb, 0x78,
					 0x56, 0x11, 0xfa, 0xd5, 0xe3, 0xf5, 0xa4, 0xfe, 0x44, 0xd7,
					 0xb5, 0x9a, 0x98, 0x13, 0x7a, 0x2b, 0xcd, 0xc3, 0x5f, 0x9e,
					 0xe9, 0xa1, 0xe2, 0x1b, 0xb1, 0x7d, 0xf1, 0x66, 0x5c, 0xd1,
					 0x39, 0x76, 0x25, 0xa1, 0x77, 0x24, 0x7e, 0x2e, 0x32, 0x9a,
					 0x66, 0x01, 0x40, 0x63, 0x61, 0x41, 0x56, 0x06, 0x10, 0xa3,
					 0x68, 0xbf, 0xd4, 0x99, 0xc2, 0xe2, 0x5b, 0xe3, 0x18, 0xaa,
					 0x4d, 0xa9, 0xe7, 0xa3, 0x52, 0xd1, 0x15, 0xdb, 0x82, 0x82,
					 0xed, 0x8d, 0x79, 0xec, 0xf9, 0xcd, 0x82, 0x03, 0x60, 0xd3,
					 0xd2, 0xd1, 0xa5, 0x8a, 0x93, 0xe0, 0x40, 0xf5, 0x55, 0x48,
					 0x87, 0xce, 0x6c, 0x98, 0x58, 0xbc, 0x2b, 0xb1, 0x02, 0x24,
					 0x99, 0x80, 0xa8, 0x58, 0x49, 0x8a, 0xbc, 0xda)
};

/**
 * SHA-384 test case 1 - count 0
 */
drbg_test_vector_t drbg_hmac_sha384_1 = {
	.type = DRBG_HMAC_SHA384, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0x09, 0x63, 0x49, 0x50, 0x6f, 0x3a, 0x76, 0x53, 0xd5, 0x4d,
					 0xb7, 0xec, 0x1d, 0x09, 0xe9, 0x34, 0x13, 0xed, 0xd1, 0x75,
					 0xb6, 0xdd, 0xbe, 0xb0, 0x0e, 0x56, 0x75, 0x2a, 0x52, 0x0a,
					 0xc8, 0xff, 0xfc, 0x79, 0x83, 0xb9, 0x18, 0xac, 0xad, 0xaa,
					 0x71, 0xa6, 0x7e, 0x16, 0x24, 0xf1, 0xb5, 0x02, 0x42, 0x60,
					 0xa0, 0x49, 0x5f, 0xda, 0xba, 0x58, 0xaa, 0xe4, 0x1d, 0xf8,
					 0x25, 0x05, 0x01, 0x2d, 0x48, 0x0c, 0x8e, 0x4f, 0x75, 0x1f,
					 0xd7, 0xeb, 0xc3, 0x9f, 0x9b, 0xec, 0xd6, 0x94, 0xb2, 0xa3),
	chunk_from_chars(0xf4, 0xc7, 0xbe, 0xc0, 0xc2, 0x6c, 0xf3, 0x89, 0x2d, 0x21,
					 0x45, 0x49, 0xac, 0x6f, 0x3d, 0x82, 0xf3, 0x4c, 0x69, 0x66,
					 0xd4, 0x29, 0x50, 0x99, 0xee, 0x56, 0x16, 0x6e, 0x87, 0x9a,
					 0x70, 0xec, 0xae, 0x13, 0x02, 0x51, 0xfa, 0xcd, 0xa3, 0x51,
					 0xe9, 0x03, 0xd8, 0x77, 0xb6, 0xc5, 0xea, 0xb5, 0x15, 0x3c,
					 0xe8, 0x7b, 0xa6, 0xc7, 0xcf, 0x8b, 0xcc, 0x61, 0xcb, 0xd1,
					 0x4c, 0xfb, 0xe3, 0x4c, 0xf1, 0xed, 0x43, 0x67, 0x8a, 0xee,
					 0x69, 0xcd, 0x87, 0xb6, 0x0e, 0x6b, 0xcb, 0x6f, 0xf4, 0x8e,
					 0xbd, 0x44, 0xce, 0x9e, 0x31, 0x98, 0x2d, 0x8f, 0xe2, 0x0a,
					 0xec, 0x34, 0xfa, 0x51, 0xd6, 0x25, 0xf8, 0x45, 0xf6, 0x10,
					 0x56, 0x57, 0x59, 0x69, 0xbf, 0x78, 0x5c, 0x2f, 0xfa, 0xb4,
					 0xdc, 0xc7, 0x54, 0xf1, 0x3d, 0xe6, 0x34, 0x23, 0xe9, 0x4b,
					 0xad, 0x8d, 0x5e, 0x16, 0x6d, 0x96, 0xa6, 0x2a, 0x60, 0x2d,
					 0x3e, 0xe4, 0x04, 0x5d, 0xf1, 0x62, 0x02, 0x8b, 0x89, 0xca,
					 0xc4, 0x5e, 0x62, 0x07, 0xd9, 0x09, 0x7f, 0x2b, 0x3a, 0xc0,
					 0xab, 0x17, 0x72, 0x92, 0x51, 0x98, 0x5f, 0x27, 0x6f, 0x12,
					 0x87, 0xf5, 0xc5, 0x6c, 0xc9, 0xba, 0x1a, 0x79, 0xfb, 0xdb,
					 0xb2, 0x91, 0xf3, 0xa9, 0x45, 0xfb, 0xfd, 0xbd, 0x63, 0xcf,
					 0x13, 0xb8, 0x2e, 0xc9, 0x1f, 0x7b, 0x10, 0x85, 0xb3, 0x32,
					 0x79, 0xe3)
};

/**
 * SHA-384 test case 3 - count 0
 */
drbg_test_vector_t drbg_hmac_sha384_2 = {
	.type = DRBG_HMAC_SHA384, .strength = 256,
	chunk_from_chars(0x43, 0xbf, 0x6f, 0x32, 0xb3, 0xb5, 0xf5, 0x80, 0xb5, 0x41,
					 0x79, 0xe4, 0x10, 0x2d, 0x06, 0x35, 0x36, 0xe7, 0xc4, 0x76,
					 0x81, 0xd6, 0xde, 0x3c, 0xfe, 0x88, 0xfd, 0x8e, 0xc6, 0x6e,
					 0x48, 0x73),
	chunk_from_chars(0x4d, 0x95, 0xf3, 0x1b, 0x96, 0x06, 0xa5, 0xf6, 0xd0, 0x4d,
					 0xff, 0x1d, 0x89, 0xb5, 0x0b, 0xec, 0xfd, 0x08, 0x82, 0xe6,
					 0xcf, 0x51, 0xc1, 0xc5, 0xd2, 0x4a, 0xd8, 0x43, 0xbc, 0x12,
					 0xd9, 0x77, 0xeb, 0xa4, 0x58, 0x2c, 0x39, 0xd7, 0x93, 0xa6,
					 0x3e, 0xad, 0xb6, 0x3f, 0x29, 0x25, 0x68, 0xc7, 0xfc, 0x42,
					 0x70, 0xe6, 0xc9, 0xae, 0xc8, 0x31, 0x86, 0xa2, 0x08, 0x19,
					 0xa7, 0xd3, 0x5e, 0x7f, 0x11, 0x55, 0xea, 0x10, 0x87, 0x94,
					 0x30, 0x2d, 0x59, 0x3c, 0x53, 0xce, 0x9d, 0x25, 0x42, 0x2b),
	chunk_from_chars(0xe9, 0x91, 0xd0, 0x00, 0xb2, 0x4e, 0xbd, 0xf8, 0x38, 0xba,
					 0x11, 0xf9, 0x84, 0x95, 0x91, 0xb0, 0x02, 0x9f, 0xef, 0xf3,
					 0x36, 0x04, 0xbc, 0x4d, 0x71, 0xac, 0xd9, 0x43, 0x01, 0xf8,
					 0xd0, 0x45, 0xee, 0xb1, 0xf8, 0x1f, 0x3a, 0x10, 0x1a, 0x29,
					 0x74, 0x03, 0xa3, 0x58, 0x59, 0x11, 0x3c, 0x09, 0x99, 0x39,
					 0x63, 0x86, 0x80, 0xd4, 0x81, 0xc8, 0x60, 0x67, 0xf5, 0x47,
					 0x62, 0x89, 0x2f, 0x82, 0x14, 0x6f, 0x61, 0xcc, 0xe7, 0xbc,
					 0x2c, 0x85, 0xd3, 0x95, 0x34, 0x8f, 0x3e, 0xa2, 0xab, 0xa6,
					 0xbb, 0x3e, 0x59, 0xdb, 0xcf, 0x8e, 0x41, 0xa8, 0x19, 0x18,
					 0xb6, 0xca, 0xb3, 0x04, 0xd4, 0x4e, 0xa1, 0xe3, 0x25, 0x73,
					 0xcd, 0x69, 0x36, 0xf3, 0x8c, 0xdc, 0x11, 0xd3, 0xc2, 0xf9,
					 0x62, 0x90, 0xcc, 0x27, 0xb0, 0xdf, 0xa3, 0xbb, 0xba, 0xfa,
					 0x93, 0x94, 0xac, 0xdf, 0x2f, 0x44, 0x35, 0x17, 0x0b, 0x42,
					 0x85, 0x63, 0x42, 0x7c, 0x4b, 0x02, 0xed, 0x25, 0x92, 0x42,
					 0x26, 0xed, 0xf8, 0xd5, 0xa5, 0xec, 0xa4, 0xee, 0xc4, 0xae,
					 0xcf, 0x98, 0xef, 0x2e, 0x6f, 0x75, 0xca, 0xa7, 0x0b, 0xdd,
					 0x84, 0x87, 0x7d, 0xf2, 0xe6, 0x37, 0xb7, 0xfa, 0xd6, 0x21,
					 0xc6, 0x17, 0x0c, 0xa5, 0xbd, 0x86, 0xe2, 0x1d, 0x0b, 0xb0,
					 0x1c, 0xc9, 0x0f, 0xe2, 0xe7, 0x63, 0x53, 0xa9, 0xd5, 0x68,
					 0x7b, 0xea)
};

/**
 * SHA-384 test case 5 - count 0
 */
drbg_test_vector_t drbg_hmac_sha384_3 = {
	.type = DRBG_HMAC_SHA384, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0xdc, 0x46, 0xe3, 0x17, 0xbe, 0xde, 0x8f, 0xf5, 0xb4, 0x51,
					 0x20, 0xee, 0xfa, 0x6d, 0xe7, 0x80, 0x31, 0xb0, 0x8c, 0xa7,
					 0x87, 0x6d, 0x62, 0xd1, 0x0e, 0xc8, 0x2f, 0x66, 0xa4, 0x8e,
					 0xba, 0x3f, 0x60, 0x24, 0xe1, 0x39, 0x63, 0xed, 0x40, 0x42,
					 0x29, 0xac, 0xb7, 0x96, 0xba, 0x0b, 0x2d, 0x0b, 0x04, 0x55,
					 0xc3, 0xa1, 0xae, 0xe2, 0x0c, 0xcd, 0x66, 0xa3, 0xdd, 0x96,
					 0x89, 0x68, 0x3f, 0x5c, 0xae, 0x3a, 0x7c, 0x37, 0xd0, 0x9a,
					 0xd6, 0xce, 0x74, 0x6d, 0xb6, 0x69, 0x21, 0x02, 0xc2, 0x89),
	chunk_from_chars(0xb3, 0xb5, 0x41, 0xca, 0x46, 0x2a, 0x72, 0xa2, 0xcc, 0xb9,
					 0x25, 0xf5, 0x8f, 0x40, 0xfc, 0xb1, 0xd5, 0x38, 0x51, 0x38,
					 0xb0, 0x95, 0xf7, 0x71, 0x57, 0x5e, 0x62, 0x85, 0x18, 0xd6,
					 0x94, 0xb9, 0xed, 0x47, 0xb1, 0x26, 0x34, 0x06, 0x3d, 0x9e,
					 0x9f, 0xb6, 0x4d, 0xdb, 0x20, 0xde, 0xcd, 0x20, 0xe5, 0x7f,
					 0xac, 0x66, 0x5e, 0xec, 0x16, 0x8a, 0x18, 0xf7, 0xaa, 0xf8,
					 0xc8, 0xf9, 0x25, 0xfe, 0x2c, 0x34, 0xfa, 0x9f, 0x76, 0x6d,
					 0x5a, 0x17, 0x24, 0x59, 0xf3, 0x2b, 0xcf, 0x24, 0x31, 0x99,
					 0xaa, 0xc7, 0x13, 0x9b, 0x2c, 0x1a, 0xa7, 0x98, 0x2f, 0xfb,
					 0x24, 0x24, 0xa4, 0x76, 0x5a, 0x9d, 0xd1, 0xdd, 0x46, 0x2f,
					 0x08, 0xa2, 0x80, 0x35, 0x0f, 0x0c, 0xea, 0x0c, 0x4b, 0x9c,
					 0xd8, 0x73, 0x24, 0xb0, 0xf4, 0x0c, 0x68, 0xf1, 0xa8, 0x77,
					 0xe6, 0x99, 0x18, 0x7f, 0x5c, 0x40, 0x61, 0x0d, 0x38, 0xe7,
					 0x15, 0x91, 0xe9, 0x85, 0x64, 0x01, 0x8e, 0xd2, 0xe3, 0x09,
					 0x00, 0x09, 0xe4, 0x9e, 0x1b, 0xe3, 0x6f, 0x86, 0x2b, 0xfd,
					 0x01, 0xf3, 0x8d, 0x95, 0x37, 0xda, 0xa3, 0x4d, 0x75, 0x65,
					 0xbb, 0x17, 0x61, 0xb5, 0x72, 0x7d, 0xf7, 0x55, 0x4f, 0xda,
					 0xf7, 0xf0, 0x98, 0x0d, 0xe7, 0xc2, 0x03, 0x3c, 0x1d, 0xbb,
					 0xea, 0xa2, 0x98, 0xdf, 0x1c, 0x7f, 0x34, 0x05, 0x16, 0x29,
					 0xfd, 0x4f)
};

/**
 * SHA-384 test case 7 - count 0
 */
drbg_test_vector_t drbg_hmac_sha384_4 = {
	.type = DRBG_HMAC_SHA384, .strength = 256,
	chunk_from_chars(0x8a, 0xf7, 0x9c, 0xed, 0x4b, 0x27, 0x77, 0xf4, 0x4e, 0xb3,
					 0xf9, 0x78, 0x1b, 0xab, 0x24, 0x35, 0x85, 0xb8, 0x27, 0x11,
					 0xdc, 0x1b, 0x36, 0x2e, 0xe9, 0x8c, 0x8e, 0x65, 0x77, 0x89,
					 0x1c, 0x7d),
	chunk_from_chars(0x79, 0xbe, 0x91, 0xa2, 0x24, 0x86, 0xcf, 0xd6, 0x21, 0x24,
					 0x8a, 0xed, 0x9c, 0xd8, 0x20, 0x27, 0x8f, 0x1a, 0xba, 0xc8,
					 0x2d, 0xe3, 0x77, 0xc3, 0x66, 0x4d, 0x83, 0xce, 0x2f, 0xf3,
					 0xd1, 0x75, 0x24, 0xca, 0x63, 0x07, 0x4b, 0xd6, 0x7b, 0x2e,
					 0xff, 0xc6, 0x20, 0xbd, 0xef, 0x61, 0x7b, 0x6f, 0x64, 0x3a,
					 0x86, 0x91, 0x9e, 0x5e, 0x98, 0x4d, 0x62, 0x81, 0xde, 0x71,
					 0x50, 0x40, 0xb6, 0xf5, 0x21, 0x86, 0x6e, 0xdf, 0xaa, 0xea,
					 0xd6, 0x16, 0xe1, 0xd1, 0xbd, 0x83, 0xc2, 0x21, 0x0f, 0x5e),
	chunk_from_chars(0x5b, 0x6a, 0x4a, 0xb3, 0x60, 0xfb, 0x85, 0x1b, 0xba, 0xa0,
					 0x35, 0xac, 0x55, 0xa5, 0x85, 0x3c, 0x4f, 0x06, 0xfd, 0x80,
					 0x41, 0xf2, 0x82, 0xde, 0xe1, 0x08, 0x27, 0xb5, 0x61, 0xee,
					 0x03, 0x8a, 0x3d, 0x3c, 0x53, 0x7d, 0x55, 0x43, 0x0b, 0x01,
					 0x3b, 0x97, 0x25, 0xc0, 0x11, 0xa2, 0x7b, 0x33, 0x49, 0x21,
					 0x43, 0x88, 0xb4, 0x23, 0x2d, 0xd3, 0x1a, 0x36, 0x60, 0xe5,
					 0xb6, 0xad, 0x1f, 0xab, 0x50, 0x82, 0x04, 0x1a, 0xa3, 0x4f,
					 0xc8, 0x04, 0xbf, 0x2b, 0xe3, 0x68, 0xac, 0x6d, 0xea, 0x0b,
					 0x3e, 0xbc, 0xaa, 0x5b, 0x54, 0xdb, 0xa5, 0x16, 0x82, 0xec,
					 0xa1, 0x9f, 0x9a, 0x5d, 0x59, 0x56, 0x33, 0xff, 0xac, 0xdd,
					 0x74, 0x6e, 0x10, 0x89, 0xea, 0x3d, 0x67, 0xb1, 0x56, 0xea,
					 0xa2, 0xe9, 0xf9, 0xb7, 0x9c, 0xc1, 0x78, 0x74, 0x55, 0x56,
					 0x16, 0x79, 0x2a, 0xf3, 0x70, 0x85, 0xad, 0x3f, 0xfa, 0xa2,
					 0x8b, 0xec, 0x60, 0xd5, 0xdb, 0x6e, 0xb0, 0xc2, 0xbc, 0xd6,
					 0xf8, 0x70, 0x54, 0x3d, 0x8e, 0xca, 0x7e, 0xeb, 0xfd, 0x63,
					 0xda, 0xcd, 0xb7, 0xf3, 0xd3, 0x89, 0xd0, 0xe0, 0xb2, 0xa9,
					 0xab, 0x92, 0xe7, 0xa7, 0x8d, 0x11, 0xf8, 0xa3, 0xf0, 0x64,
					 0x02, 0xa9, 0xae, 0x75, 0x10, 0xdd, 0x96, 0x48, 0xac, 0x8c,
					 0xc3, 0x3d, 0xee, 0xe4, 0x9b, 0xc6, 0xfe, 0xe9, 0x7b, 0xc8,
					 0x6d, 0xcf)
};

/**
 * SHA-384 test case 9 - count 0
 */
drbg_test_vector_t drbg_hmac_sha384_5 = {
	.type = DRBG_HMAC_SHA384, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0x54, 0x96, 0x45, 0x38, 0xca, 0x40, 0xae, 0xe8, 0xef, 0xa7,
					 0xbf, 0x33, 0x6a, 0x23, 0x78, 0x7d, 0x95, 0x06, 0x40, 0x69,
					 0x64, 0xc3, 0x27, 0x43, 0xdf, 0x9c, 0xce, 0x28, 0xe8, 0x92,
					 0xc2, 0xa6, 0x9e, 0x7b, 0xdf, 0x08, 0xe8, 0x5e, 0x8d, 0xe7,
					 0xe0, 0x49, 0xe7, 0xfc, 0x3a, 0x13, 0x88, 0x85, 0xc9, 0xc8,
					 0x4d, 0xf8, 0x1f, 0x73, 0x19, 0x6b, 0xa5, 0x08, 0x8f, 0xe2,
					 0x85, 0x43, 0xa8, 0x1b, 0xb5, 0x2c, 0x1d, 0xb4, 0x7f, 0xef,
					 0xad, 0x85, 0x51, 0x1f, 0x8d, 0x42, 0xa9, 0x84, 0x23, 0x67),
	chunk_from_chars(0x91, 0x9f, 0x95, 0x90, 0x97, 0x3f, 0xbb, 0x06, 0xea, 0xf8,
					 0xe9, 0x7c, 0x8b, 0x20, 0x2e, 0xc3, 0xbf, 0x6a, 0xd4, 0xeb,
					 0x50, 0xb9, 0xe3, 0xd6, 0x63, 0x8b, 0x66, 0x22, 0x01, 0x79,
					 0xeb, 0x8c, 0x3e, 0xf1, 0x87, 0xdd, 0x50, 0x27, 0x0f, 0xde,
					 0x90, 0x3d, 0x5c, 0x70, 0x13, 0x55, 0x2e, 0x45, 0x18, 0xad,
					 0x9c, 0x69, 0xfa, 0x5e, 0x6d, 0xd7, 0x09, 0x5d, 0xbd, 0x3c,
					 0x68, 0x03, 0xf2, 0x9d, 0x8b, 0xcf, 0x9d, 0x65, 0xcd, 0xe7,
					 0x50, 0xb1, 0x7c, 0x79, 0x73, 0x7e, 0x96, 0xd7, 0xec, 0xc2,
					 0x30, 0x40, 0x2e, 0x22, 0x6e, 0x00, 0xc7, 0x25, 0x29, 0x17,
					 0xe9, 0x54, 0x99, 0x70, 0x55, 0x23, 0xf3, 0xe1, 0x5c, 0x2c,
					 0x59, 0x05, 0x4a, 0xba, 0x3a, 0x40, 0xfa, 0x10, 0x6c, 0x85,
					 0xd2, 0xfc, 0xb1, 0xff, 0x9c, 0x11, 0x5e, 0x19, 0xac, 0xc5,
					 0x79, 0xef, 0x1a, 0x19, 0x0e, 0xf7, 0x6b, 0x75, 0x29, 0xfe,
					 0x79, 0x62, 0x30, 0xfd, 0x7c, 0x67, 0x8a, 0xc9, 0xd2, 0x29,
					 0x40, 0x13, 0xee, 0x37, 0x33, 0x1e, 0x45, 0xe0, 0x72, 0xf2,
					 0xc8, 0xb5, 0xa9, 0x27, 0x5e, 0xe9, 0x81, 0x3d, 0x51, 0x00,
					 0x6f, 0x80, 0x34, 0x3c, 0xb6, 0xa2, 0x16, 0x81, 0xbc, 0x5c,
					 0xe0, 0xee, 0x6e, 0xe3, 0xf4, 0x97, 0x10, 0xaa, 0x23, 0xc9,
					 0x25, 0xec, 0x8b, 0x01, 0x82, 0x49, 0xf0, 0xe6, 0xf0, 0x0b,
					 0x85, 0xd6)
};

/**
 * SHA-384 test case 11 - count 0
 */
drbg_test_vector_t drbg_hmac_sha384_6 = {
	.type = DRBG_HMAC_SHA384, .strength = 256,
	chunk_from_chars(0xc9, 0xb4, 0x5f, 0x12, 0x14, 0xb7, 0xfe, 0x12, 0xd5, 0xfa,
					 0x54, 0xb5, 0x79, 0xe0, 0x55, 0x02, 0x2c, 0x25, 0xd2, 0x3d,
					 0x8a, 0xf6, 0x3e, 0x39, 0x26, 0xc0, 0xec, 0xbd, 0x92, 0xab,
					 0x8d, 0x01),
	chunk_from_chars(0xe9, 0xfe, 0x33, 0xf3, 0xd1, 0x56, 0x8b, 0x14, 0x6b, 0x7a,
					 0x86, 0xfc, 0x0f, 0xe8, 0x0e, 0x11, 0xe7, 0xd3, 0xe0, 0x8c,
					 0xa3, 0x9c, 0xc5, 0xdd, 0x38, 0x67, 0x6f, 0x8a, 0xcf, 0xdb,
					 0x75, 0xbb, 0x9a, 0xa0, 0x91, 0xa0, 0x27, 0xfc, 0x5b, 0x3c,
					 0x39, 0x4c, 0x7f, 0x17, 0xc8, 0x16, 0xc3, 0xcf, 0xf6, 0x27,
					 0x0e, 0x5b, 0xa1, 0x81, 0x9e, 0x48, 0x2e, 0x42, 0x19, 0x78,
					 0x96, 0xb9, 0x24, 0xf6, 0x63, 0x00, 0xa0, 0xd4, 0x25, 0x15,
					 0x3d, 0x55, 0x1e, 0xb0, 0x00, 0xea, 0xe8, 0x09, 0xc1, 0x5a),
	chunk_from_chars(0xc7, 0xd0, 0xb2, 0x7b, 0x36, 0xbc, 0x29, 0x48, 0x31, 0x95,
					 0x40, 0x05, 0x5d, 0xf2, 0x29, 0xad, 0x3d, 0x43, 0xb5, 0x08,
					 0x6a, 0xe3, 0x45, 0xc1, 0xc3, 0xa5, 0x17, 0x01, 0x2c, 0x24,
					 0x7d, 0x5e, 0xdf, 0x25, 0xcc, 0xdf, 0x83, 0xb6, 0xfd, 0xf4,
					 0xec, 0x10, 0x41, 0x49, 0x65, 0x2b, 0x1d, 0x26, 0xed, 0x70,
					 0x36, 0x7b, 0x24, 0x6d, 0xfe, 0x9f, 0x58, 0x90, 0xf7, 0x26,
					 0xca, 0xd6, 0x77, 0x74, 0x4f, 0x64, 0x08, 0xd2, 0x43, 0xd9,
					 0x86, 0xef, 0x76, 0x8a, 0xac, 0x65, 0x61, 0x7c, 0x06, 0x20,
					 0x4d, 0x35, 0xe3, 0x1a, 0x98, 0xa0, 0x2e, 0xd3, 0x23, 0x5b,
					 0x0f, 0x98, 0x99, 0x86, 0x69, 0xca, 0xaa, 0x3f, 0xb5, 0xeb,
					 0x94, 0xdb, 0x64, 0xc2, 0xdf, 0xb3, 0xce, 0xf2, 0x31, 0xed,
					 0xa1, 0xfd, 0x59, 0xb4, 0x37, 0x7b, 0x0b, 0x00, 0x2a, 0x8c,
					 0x0b, 0x53, 0x10, 0xb9, 0x6f, 0x49, 0xe0, 0xfe, 0x3c, 0x46,
					 0x94, 0x45, 0x14, 0xe5, 0xf5, 0xae, 0xb4, 0xf6, 0x44, 0xd4,
					 0x38, 0x5f, 0x0e, 0x21, 0x09, 0xbf, 0xb9, 0xcf, 0xff, 0xa9,
					 0x96, 0x2e, 0x26, 0xab, 0x2c, 0x76, 0x49, 0xa2, 0x7e, 0xfa,
					 0x33, 0x09, 0xe7, 0x2c, 0xd1, 0x51, 0x16, 0xa5, 0x78, 0x7c,
					 0x13, 0x0c, 0xad, 0xcb, 0x4e, 0x42, 0xba, 0x6e, 0x9d, 0x0b,
					 0xd1, 0x4e, 0xed, 0x1c, 0x6a, 0x01, 0x9b, 0x5b, 0xe1, 0x3b,
					 0xe0, 0xc4)
};

/**
 * SHA-512 test case 1 - count 0
 */
drbg_test_vector_t drbg_hmac_sha512_1 = {
	.type = DRBG_HMAC_SHA512, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0x48, 0xc1, 0x21, 0xb1, 0x87, 0x33, 0xaf, 0x15, 0xc2, 0x7e,
					 0x1d, 0xd9, 0xba, 0x66, 0xa9, 0xa8, 0x1a, 0x55, 0x79, 0xcd,
					 0xba, 0x0f, 0x5b, 0x65, 0x7e, 0xc5, 0x3c, 0x2b, 0x9e, 0x90,
					 0xbb, 0xf6, 0xbb, 0xb7, 0xc7, 0x77, 0x42, 0x80, 0x68, 0xfa,
					 0xd9, 0x97, 0x08, 0x91, 0xf8, 0x79, 0xb1, 0xaf, 0xe0, 0xff,
					 0xef, 0xda, 0xdb, 0x9c, 0xcf, 0x99, 0x05, 0x04, 0xd5, 0x68,
					 0xbd, 0xb4, 0xd8, 0x62, 0xcb, 0xe1, 0x7c, 0xcc, 0xe6, 0xe2,
					 0x2d, 0xfc, 0xab, 0x8b, 0x48, 0x04, 0xfd, 0x21, 0x42, 0x1a),
	chunk_from_chars(0x05, 0xda, 0x6a, 0xac, 0x7d, 0x98, 0x0d, 0xa0, 0x38, 0xf6,
					 0x5f, 0x39, 0x28, 0x41, 0x47, 0x6d, 0x37, 0xfe, 0x70, 0xfb,
					 0xd3, 0xe3, 0x69, 0xd1, 0xf8, 0x01, 0x96, 0xe6, 0x6e, 0x54,
					 0xb8, 0xfa, 0xdb, 0x1d, 0x60, 0xe1, 0xa0, 0xf3, 0xd4, 0xdc,
					 0x17, 0x37, 0x69, 0xd7, 0x5f, 0xc3, 0x41, 0x05, 0x49, 0xd7,
					 0xa8, 0x43, 0x27, 0x0a, 0x54, 0xa0, 0x68, 0xb4, 0xfe, 0x76,
					 0x7d, 0x7d, 0x9a, 0x59, 0x60, 0x45, 0x10, 0xa8, 0x75, 0xad,
					 0x1e, 0x97, 0x31, 0xc8, 0xaf, 0xd0, 0xfd, 0x50, 0xb8, 0x25,
					 0xe2, 0xc5, 0x0d, 0x06, 0x25, 0x76, 0x17, 0x51, 0x06, 0xa9,
					 0x98, 0x1b, 0xe3, 0x7e, 0x02, 0xec, 0x7c, 0x5c, 0xd0, 0xa6,
					 0x9a, 0xa0, 0xca, 0x65, 0xbd, 0xda, 0xee, 0x1b, 0x0d, 0xe5,
					 0x32, 0xe1, 0x0c, 0xfa, 0x1f, 0x5b, 0xf6, 0xa0, 0x26, 0xe4,
					 0x73, 0x79, 0x73, 0x6a, 0x09, 0x9d, 0x67, 0x50, 0xab, 0x12,
					 0x1d, 0xbe, 0x36, 0x22, 0xb8, 0x41, 0xba, 0xf8, 0xbd, 0xcb,
					 0xe8, 0x75, 0xc8, 0x5b, 0xa4, 0xb5, 0x86, 0xb8, 0xb5, 0xb5,
					 0x7b, 0x0f, 0xec, 0xbe, 0xc0, 0x8c, 0x12, 0xff, 0x2a, 0x94,
					 0x53, 0xc4, 0x7c, 0x6e, 0x32, 0xa5, 0x21, 0x03, 0xd9, 0x72,
					 0xc6, 0x2a, 0xb9, 0xaf, 0xfb, 0x8e, 0x72, 0x8a, 0x31, 0xfc,
					 0xef, 0xbb, 0xcc, 0xc5, 0x56, 0xc0, 0xf0, 0xa3, 0x5f, 0x4b,
					 0x10, 0xac, 0xe2, 0xd9, 0x6b, 0x90, 0x6e, 0x36, 0xcb, 0xb7,
					 0x22, 0x33, 0x20, 0x1e, 0x53, 0x6d, 0x3e, 0x13, 0xb0, 0x45,
					 0x18, 0x7b, 0x41, 0x7d, 0x24, 0x49, 0xca, 0xd1, 0xed, 0xd1,
					 0x92, 0xe0, 0x61, 0xf1, 0x2d, 0x22, 0x14, 0x7b, 0x0a, 0x17,
					 0x6e, 0xa8, 0xd9, 0xc4, 0xc3, 0x54, 0x04, 0x39, 0x5b, 0x65,
					 0x02, 0xef, 0x33, 0x3a, 0x81, 0x3b, 0x65, 0x86, 0x03, 0x74,
					 0x79, 0xe0, 0xfa, 0x3c, 0x6a, 0x23)
};

/**
 * SHA-512 test case 3 - count 0
 */
drbg_test_vector_t drbg_hmac_sha512_2 = {
	.type = DRBG_HMAC_SHA512, .strength = 256,
	chunk_from_chars(0x21, 0x23, 0x00, 0xf9, 0x38, 0x99, 0xff, 0x7c, 0xb1, 0x44,
					 0xf2, 0x04, 0x26, 0x02, 0x8b, 0x97, 0x63, 0x80, 0xa3, 0x48,
					 0x25, 0x3b, 0xcc, 0x3f, 0xf4, 0x2b, 0x52, 0x8c, 0xd1, 0x97,
					 0x25, 0x49),
	chunk_from_chars(0x97, 0xae, 0xf9, 0x35, 0xea, 0x33, 0x71, 0x7e, 0x8e, 0x86,
					 0x44, 0xbb, 0x8c, 0x47, 0x89, 0xf3, 0x75, 0xc4, 0x8a, 0x94,
					 0x5d, 0xed, 0x08, 0x77, 0x11, 0x49, 0xe8, 0x28, 0xa2, 0x2d,
					 0xc8, 0x66, 0x82, 0x58, 0x0f, 0x51, 0x07, 0x0b, 0xa1, 0xe9,
					 0x91, 0xd9, 0x80, 0x3f, 0x51, 0xfd, 0x9a, 0x6f, 0x63, 0xcd,
					 0x91, 0xc1, 0xeb, 0xb2, 0xca, 0xa1, 0x5f, 0x28, 0x37, 0xdf,
					 0x8f, 0x35, 0xcb, 0xb6, 0xfe, 0x96, 0xdf, 0x26, 0x74, 0xa1,
					 0x36, 0x99, 0x0a, 0x59, 0x76, 0xcb, 0xba, 0xb6, 0x3b, 0xc1),
	chunk_from_chars(0x0e, 0x85, 0x33, 0xf6, 0x4b, 0x60, 0xc2, 0x3a, 0x26, 0x55,
					 0x82, 0x70, 0x37, 0xdb, 0x21, 0x8c, 0x2f, 0xe9, 0xce, 0x43,
					 0x0f, 0xa4, 0xed, 0x6e, 0xd9, 0xbe, 0x34, 0x9c, 0x4b, 0xdc,
					 0x6f, 0x40, 0x01, 0x8b, 0x42, 0xf4, 0x86, 0xfa, 0x04, 0x28,
					 0x8b, 0x3b, 0x0c, 0x62, 0xa1, 0x28, 0x12, 0xe7, 0x6e, 0x08,
					 0xc7, 0x60, 0x62, 0xa5, 0x10, 0xcc, 0x60, 0x84, 0x1f, 0x16,
					 0x58, 0x69, 0xef, 0xac, 0xee, 0xf9, 0x08, 0x05, 0xbd, 0xde,
					 0x2f, 0xd6, 0x6c, 0x36, 0xc3, 0x8a, 0x2a, 0xc9, 0xc3, 0xcb,
					 0x86, 0xbf, 0xd3, 0x04, 0x06, 0x56, 0x9e, 0x0a, 0xfd, 0x24,
					 0x51, 0x02, 0xf2, 0xea, 0x2d, 0x49, 0xe4, 0xee, 0x5f, 0x69,
					 0x18, 0x72, 0x27, 0xa3, 0xf0, 0xed, 0xfb, 0xc1, 0x25, 0x9c,
					 0xb6, 0x56, 0x4a, 0x2d, 0x4e, 0x82, 0x9b, 0x3f, 0xc3, 0xb6,
					 0x99, 0x6e, 0x37, 0x54, 0x6f, 0x1d, 0x8a, 0x16, 0xfc, 0xd8,
					 0x20, 0x1d, 0x1a, 0xd2, 0x86, 0x61, 0xbb, 0xb0, 0x01, 0x2d,
					 0xaa, 0xd5, 0x5d, 0x54, 0x03, 0xe8, 0x33, 0xd8, 0xa0, 0x06,
					 0x8d, 0x21, 0x6c, 0x87, 0x9b, 0xce, 0xbc, 0x05, 0x4d, 0xf0,
					 0xc9, 0xcb, 0xa1, 0x4d, 0xad, 0x48, 0x63, 0xee, 0x1f, 0x75,
					 0xb7, 0x8b, 0xc4, 0x88, 0x66, 0x2c, 0xb0, 0xc9, 0x1c, 0xa4,
					 0xfd, 0xfc, 0xe7, 0xdf, 0x59, 0x16, 0xb4, 0xe6, 0x25, 0x80,
					 0x90, 0x2c, 0x60, 0x1b, 0xe7, 0x06, 0xdc, 0xc7, 0x90, 0x38,
					 0x58, 0xe6, 0xb9, 0x92, 0x07, 0x35, 0xbd, 0xaa, 0x63, 0x5a,
					 0xdd, 0x5c, 0x06, 0x08, 0x0d, 0x82, 0x26, 0x53, 0x45, 0xb4,
					 0x90, 0x37, 0xa3, 0x2f, 0xcf, 0x0a, 0x7c, 0x9e, 0xa6, 0x06,
					 0x9e, 0x33, 0x69, 0xf9, 0xb4, 0xaa, 0x45, 0x49, 0x3e, 0xfd,
					 0x73, 0x18, 0xda, 0x2a, 0xe9, 0xb4, 0xfc, 0x30, 0x04, 0x98,
					 0x24, 0x8a, 0xfa, 0xad, 0x8d, 0x49)
};

/**
 * SHA-512 test case 5 - count 0
 */
drbg_test_vector_t drbg_hmac_sha512_3 = {
	.type = DRBG_HMAC_SHA512, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0x54, 0xda, 0xf8, 0xea, 0xd3, 0xac, 0xcb, 0x38, 0x2c, 0xdf,
					 0x25, 0x1c, 0xfb, 0xc8, 0x64, 0x4a, 0xb8, 0xbf, 0x4f, 0x99,
					 0x69, 0x37, 0x11, 0x10, 0x2c, 0x02, 0xb2, 0xb6, 0x92, 0x0c,
					 0x25, 0xc8, 0x4b, 0xcf, 0x3b, 0x02, 0xdc, 0x3e, 0x13, 0x0f,
					 0x50, 0xe8, 0x9b, 0xdf, 0x2c, 0xf7, 0x52, 0xe2, 0x45, 0x30,
					 0xe0, 0xb8, 0x95, 0x51, 0x90, 0xec, 0xd3, 0xca, 0x11, 0x7d,
					 0x26, 0x20, 0xa0, 0xf5, 0x82, 0x3a, 0x2c, 0x7c, 0x6f, 0x10,
					 0x8a, 0x72, 0xae, 0xc6, 0x9a, 0xd5, 0x91, 0x51, 0xc8, 0xec),
	chunk_from_chars(0xc9, 0x44, 0x4a, 0x93, 0x66, 0x76, 0x4f, 0x89, 0xef, 0x32,
					 0x26, 0xee, 0x70, 0xf6, 0x18, 0x49, 0x2f, 0xa9, 0xfa, 0x9a,
					 0x1c, 0x96, 0xe5, 0x3e, 0xc8, 0x17, 0xde, 0xc9, 0xc4, 0xc8,
					 0x3f, 0xb6, 0xef, 0x7f, 0xcb, 0x2c, 0xae, 0x99, 0x73, 0x29,
					 0x3e, 0xaa, 0xa5, 0x96, 0xcb, 0xbf, 0xda, 0x4c, 0x3f, 0xbf,
					 0x2d, 0xd9, 0x12, 0x53, 0xbe, 0x2c, 0xe6, 0x64, 0x8c, 0x31,
					 0xe2, 0xaa, 0x92, 0x31, 0x5a, 0xb4, 0xad, 0xb9, 0x11, 0xaa,
					 0xac, 0x15, 0xb5, 0x6a, 0x27, 0x3d, 0xa7, 0x4c, 0x11, 0x4e,
					 0x1f, 0x58, 0xb3, 0x7b, 0x47, 0x05, 0x17, 0x1c, 0x4c, 0xf7,
					 0xc7, 0x45, 0x23, 0xf5, 0x2e, 0x72, 0x1f, 0x11, 0xbd, 0x8c,
					 0x37, 0x51, 0x99, 0x95, 0xe0, 0xd1, 0x3c, 0x19, 0x0d, 0x7e,
					 0x58, 0x6c, 0xd3, 0xe3, 0xe9, 0xfc, 0x2d, 0xa9, 0x48, 0xc5,
					 0xa5, 0x2a, 0xa6, 0x9c, 0xdc, 0xc8, 0x42, 0xda, 0x17, 0xb4,
					 0x94, 0xcb, 0x33, 0x0f, 0xca, 0x9d, 0x87, 0x24, 0x96, 0x5f,
					 0xe7, 0xbb, 0xc4, 0x5c, 0xcd, 0x87, 0x72, 0x14, 0xed, 0x4d,
					 0x91, 0x71, 0x1a, 0x4e, 0xcc, 0xf9, 0x32, 0x8f, 0x7a, 0x2e,
					 0x39, 0x3c, 0xf5, 0x9e, 0x69, 0xbb, 0xa6, 0x87, 0x16, 0xab,
					 0x93, 0x0d, 0x54, 0x62, 0x74, 0xa9, 0xd7, 0xa6, 0x3f, 0xb2,
					 0x75, 0xc2, 0x07, 0x12, 0xdd, 0x6b, 0x63, 0x1b, 0x69, 0x33,
					 0x73, 0x29, 0x53, 0x11, 0x87, 0x19, 0x21, 0xc5, 0x2e, 0x2e,
					 0xc5, 0x46, 0x15, 0x9d, 0x92, 0xeb, 0x15, 0x13, 0x3b, 0xdb,
					 0x45, 0x3b, 0xa5, 0x4a, 0xef, 0x09, 0xc4, 0x79, 0x0a, 0x1b,
					 0x34, 0x28, 0xb4, 0x44, 0x75, 0xee, 0x8f, 0xab, 0x6d, 0xb2,
					 0xd5, 0xa8, 0xaa, 0xaf, 0xc0, 0x46, 0x35, 0x5e, 0xb6, 0x42,
					 0x1d, 0xbc, 0xe2, 0xf9, 0x28, 0x3a, 0x03, 0xb4, 0xc3, 0x73,
					 0x97, 0x51, 0x31, 0xae, 0xca, 0x69)
};

/**
 * SHA-512 test case 7 - count 0
 */
drbg_test_vector_t drbg_hmac_sha512_4 = {
	.type = DRBG_HMAC_SHA512, .strength = 256,
	chunk_from_chars(0x39, 0xa7, 0x19, 0x6a, 0x26, 0x16, 0xa5, 0x32, 0x9c, 0x4b,
					 0x09, 0xf3, 0xbe, 0x52, 0x1c, 0x23, 0x91, 0xfa, 0x37, 0xfc,
					 0xe9, 0x54, 0xa1, 0xfd, 0x25, 0x9b, 0xde, 0x23, 0x0c, 0x8d,
					 0x13, 0x0e),
	chunk_from_chars(0xcd, 0xed, 0x7a, 0xae, 0xab, 0x10, 0x25, 0xa6, 0x05, 0x3c,
					 0x12, 0xb8, 0xbc, 0xa1, 0x97, 0xc7, 0x96, 0x3d, 0xee, 0x09,
					 0xf5, 0x0c, 0x1c, 0xb9, 0x73, 0x34, 0x11, 0xc7, 0x2c, 0xd0,
					 0x90, 0xb9, 0x1f, 0xf7, 0x7f, 0x52, 0x36, 0x78, 0xc1, 0x92,
					 0x6b, 0x2b, 0x11, 0x35, 0x4a, 0x55, 0x51, 0x0f, 0xf9, 0x15,
					 0xac, 0x12, 0x6f, 0x3a, 0xc6, 0xed, 0xdc, 0x6f, 0x9b, 0x85,
					 0x7b, 0x12, 0x43, 0x63, 0xff, 0xb5, 0x3f, 0xbb, 0x66, 0x4a,
					 0x3d, 0x7c, 0xb9, 0x10, 0xfb, 0xf5, 0x58, 0xb8, 0x33, 0xbb),
	chunk_from_chars(0xa4, 0xc9, 0x7a, 0x54, 0xbe, 0x09, 0x1a, 0x7f, 0x1b, 0x56,
					 0x37, 0x2c, 0x08, 0xb4, 0xdb, 0xe9, 0xa1, 0x60, 0xe5, 0x9b,
					 0xfb, 0x1b, 0x38, 0xdf, 0xee, 0xff, 0x9b, 0xd8, 0xdb, 0x5b,
					 0x4f, 0x19, 0x71, 0x4c, 0x47, 0x19, 0xc9, 0x12, 0x7e, 0x8c,
					 0xbd, 0x24, 0xe7, 0x83, 0x90, 0x0e, 0x71, 0xa7, 0x7b, 0x1a,
					 0x83, 0x03, 0xc8, 0x86, 0x65, 0x5d, 0x85, 0x9a, 0x0a, 0x50,
					 0x54, 0xcd, 0x84, 0xed, 0x1c, 0x6b, 0xa1, 0x11, 0xad, 0x13,
					 0xdc, 0x12, 0x1b, 0xd0, 0x86, 0x9b, 0x21, 0x71, 0x2b, 0xc2,
					 0x42, 0x24, 0xc7, 0xe3, 0x59, 0xe2, 0xde, 0x8f, 0x09, 0x33,
					 0xc9, 0x82, 0xa1, 0x80, 0x11, 0x85, 0x64, 0x63, 0xfd, 0xb4,
					 0x9c, 0xca, 0x5c, 0x46, 0x80, 0x26, 0xce, 0x02, 0x4a, 0x56,
					 0x9c, 0x98, 0xfa, 0x3b, 0x3c, 0xc4, 0x88, 0x94, 0x6f, 0xe9,
					 0xfe, 0x07, 0xdd, 0x18, 0xfd, 0x75, 0x86, 0x3f, 0xfc, 0xfa,
					 0xd3, 0x2a, 0x2d, 0x04, 0x80, 0xe4, 0x5b, 0x16, 0x9f, 0x68,
					 0xcd, 0x6f, 0x31, 0xc6, 0x23, 0xb4, 0xfb, 0xd3, 0x35, 0x42,
					 0x70, 0xc5, 0x1c, 0x49, 0x4c, 0x61, 0x4d, 0xcc, 0xc2, 0x7d,
					 0xdb, 0x5e, 0x2f, 0x9b, 0xcb, 0xbc, 0x3f, 0xc3, 0x97, 0x38,
					 0x4c, 0x92, 0x37, 0xec, 0xaf, 0x6b, 0x47, 0x59, 0x1f, 0x67,
					 0x79, 0xa5, 0x2a, 0xb4, 0xc0, 0x38, 0x1b, 0xd7, 0xfe, 0x73,
					 0x44, 0x4c, 0x20, 0xd4, 0x7d, 0x5f, 0x25, 0xa4, 0x41, 0x3f,
					 0x80, 0x68, 0x55, 0x7e, 0x7f, 0xd2, 0x74, 0x5f, 0xdb, 0x88,
					 0xa8, 0x8d, 0x26, 0x2d, 0x1a, 0x23, 0x86, 0xc9, 0xe8, 0x4a,
					 0x11, 0x00, 0x08, 0xec, 0xd8, 0xcb, 0x9a, 0xea, 0x88, 0x3d,
					 0x43, 0x52, 0xd9, 0xe6, 0x16, 0xab, 0xe7, 0x59, 0xf1, 0x0c,
					 0x86, 0x7d, 0xa1, 0x85, 0x5e, 0xde, 0xd5, 0xa3, 0xeb, 0x5c,
					 0x4b, 0x20, 0xc9, 0x5a, 0xab, 0xe6)
};

/**
 * SHA-512 test case 9 - count 0
 */
drbg_test_vector_t drbg_hmac_sha512_5 = {
	.type = DRBG_HMAC_SHA512, .strength = 256,
	{ NULL, 0 },
	chunk_from_chars(0x60, 0xf1, 0x2a, 0x1e, 0x56, 0x3c, 0x83, 0xdd, 0xaa, 0x36,
					 0x13, 0xa5, 0x15, 0x53, 0x7f, 0x59, 0x0b, 0xf2, 0x2c, 0x23,
					 0x95, 0xc3, 0xcc, 0x67, 0x4e, 0x23, 0xcc, 0x7b, 0x3a, 0x7e,
					 0x59, 0x57, 0xb0, 0x46, 0xb7, 0x97, 0xa0, 0xaf, 0x84, 0x02,
					 0x7c, 0x69, 0x56, 0x88, 0x38, 0xf6, 0x49, 0x0e, 0x68, 0xda,
					 0x0a, 0xbd, 0xc0, 0x8a, 0xd5, 0xd9, 0xa3, 0xb8, 0x5a, 0x98,
					 0x16, 0xbe, 0xae, 0xe2, 0xda, 0x74, 0x9c, 0xa8, 0xbd, 0x75,
					 0x7b, 0x31, 0xd7, 0x56, 0xcb, 0x82, 0x0f, 0x4b, 0x52, 0x17),
	chunk_from_chars(0x33, 0x95, 0x2b, 0xde, 0x77, 0x3c, 0x0d, 0xc0, 0x09, 0x10,
					 0xce, 0x6d, 0xc5, 0x5d, 0xae, 0x77, 0xd6, 0xa5, 0xc5, 0xab,
					 0x9a, 0x93, 0xf6, 0x12, 0xa6, 0x56, 0xb9, 0xcc, 0xc1, 0xeb,
					 0x24, 0xa6, 0xcc, 0xea, 0x86, 0xd5, 0x09, 0x0d, 0xf3, 0x05,
					 0xde, 0x69, 0x29, 0x96, 0x4c, 0x45, 0x60, 0x53, 0x14, 0x81,
					 0xea, 0x83, 0x55, 0x64, 0xf1, 0xde, 0x27, 0x10, 0xa7, 0xb1,
					 0x55, 0xf3, 0x77, 0x23, 0x47, 0xa2, 0x71, 0xfb, 0x8a, 0x40,
					 0x62, 0x13, 0x53, 0xc1, 0x35, 0xbe, 0xd0, 0x18, 0x0d, 0x86,
					 0x7d, 0x4b, 0x8e, 0xe6, 0x13, 0x9f, 0x3b, 0x6f, 0x60, 0xa1,
					 0xe4, 0x07, 0x2c, 0xce, 0x31, 0x98, 0xb2, 0x01, 0xdd, 0x1a,
					 0x0a, 0x31, 0x93, 0xa2, 0xd3, 0x06, 0x1e, 0xf1, 0x21, 0xed,
					 0x1d, 0x5c, 0xc1, 0x27, 0xe7, 0xa4, 0x33, 0xae, 0x54, 0x07,
					 0xc0, 0x9b, 0x45, 0xce, 0xb1, 0xed, 0xe6, 0x9f, 0x1c, 0x5e,
					 0x27, 0xe7, 0xac, 0x7f, 0xf7, 0xdd, 0x65, 0xd0, 0x7c, 0x92,
					 0xe9, 0xb0, 0x02, 0xf4, 0x4a, 0x36, 0xb4, 0x98, 0x8b, 0x4b,
					 0xab, 0x45, 0xd2, 0x07, 0x54, 0x94, 0x3a, 0xfc, 0x0b, 0xae,
					 0x2d, 0xac, 0x20, 0xc3, 0xcc, 0x6e, 0xe0, 0x8f, 0xfd, 0x3c,
					 0x6a, 0x94, 0x4e, 0xca, 0xe7, 0xd1, 0x25, 0x1a, 0x81, 0x49,
					 0xfb, 0xb8, 0x19, 0xdb, 0x2c, 0xb7, 0xd6, 0xfe, 0x48, 0xae,
					 0x98, 0x44, 0xf1, 0xc0, 0xcf, 0xf1, 0xd7, 0x61, 0x67, 0x03,
					 0x47, 0x35, 0xc7, 0x99, 0x8a, 0x1d, 0xd0, 0xa0, 0x66, 0xe6,
					 0x7e, 0xae, 0xe8, 0x6d, 0xa5, 0x81, 0x68, 0xe4, 0xeb, 0xe0,
					 0xfd, 0x36, 0xbc, 0x1e, 0x0e, 0x46, 0x9c, 0xd3, 0x77, 0xb4,
					 0x14, 0xb6, 0xe3, 0xc8, 0x92, 0x02, 0x02, 0x40, 0xae, 0x71,
					 0xf1, 0x72, 0x86, 0xd1, 0xa4, 0x84, 0xa1, 0x6d, 0xc1, 0x8b,
					 0x25, 0xeb, 0xab, 0x47, 0x4c, 0xe6)
};

/**
 * SHA-512 test case 11 - count 0
 */
drbg_test_vector_t drbg_hmac_sha512_6 = {
	.type = DRBG_HMAC_SHA512, .strength = 256,
	chunk_from_chars(0x5e, 0x5b, 0x96, 0xb3, 0x8f, 0xff, 0xb9, 0x29, 0xbf, 0x3d,
					 0x74, 0x40, 0xe6, 0x0e, 0x9c, 0x38, 0xe6, 0xea, 0xb0, 0x5f,
					 0xd1, 0xe1, 0x2a, 0xb8, 0x2c, 0xb5, 0xfd, 0x5e, 0x6d, 0xa2,
					 0x6a, 0xb4),
	chunk_from_chars(0x29, 0xb9, 0x7c, 0x42, 0x1a, 0xea, 0x05, 0xea, 0x9b, 0x1c,
					 0x68, 0x4a, 0x8b, 0x91, 0xcd, 0x04, 0xe5, 0x76, 0xa0, 0x19,
					 0x88, 0x31, 0xc4, 0x2a, 0x18, 0x66, 0x82, 0x4a, 0x34, 0x62,
					 0xc7, 0xa8, 0x10, 0xa7, 0xce, 0x39, 0x6f, 0x4b, 0xea, 0x52,
					 0x27, 0x54, 0x5c, 0x1a, 0x87, 0xd7, 0x4e, 0xb5, 0x24, 0x6c,
					 0x56, 0x51, 0x0a, 0x65, 0x2f, 0xd9, 0x0e, 0xa3, 0x77, 0xfe,
					 0xd3, 0x15, 0xd7, 0x57, 0x6a, 0x2a, 0x23, 0x0c, 0xef, 0xdf,
					 0x4f, 0x45, 0xae, 0x28, 0x72, 0xb9, 0xd9, 0xad, 0xad, 0x3f),
	chunk_from_chars(0x61, 0x46, 0x67, 0xab, 0x1e, 0xe7, 0x5e, 0xd7, 0x13, 0x1f,
					 0x2e, 0x37, 0xda, 0x34, 0xa9, 0xb3, 0x92, 0x4d, 0xd8, 0x99,
					 0x7b, 0x35, 0x33, 0x5f, 0xfd, 0xdd, 0x67, 0xaa, 0x3a, 0x5d,
					 0xca, 0xec, 0x1b, 0x6c, 0xc9, 0xcf, 0x32, 0xab, 0x0c, 0x27,
					 0x7f, 0x2c, 0xb0, 0xcc, 0xa2, 0xae, 0xc7, 0x2d, 0x66, 0x3a,
					 0x25, 0x83, 0x08, 0xa6, 0x0c, 0xae, 0xed, 0xe4, 0x3d, 0x11,
					 0xd2, 0xd7, 0xcb, 0x81, 0x1f, 0x38, 0x67, 0x7b, 0x81, 0x31,
					 0x75, 0xd9, 0x32, 0x79, 0x0a, 0x37, 0x44, 0x9c, 0xf2, 0x8a,
					 0xa6, 0x81, 0x3a, 0xc4, 0x98, 0x96, 0x5f, 0x56, 0x92, 0x64,
					 0xc3, 0x23, 0xfa, 0x5e, 0x12, 0x0e, 0x95, 0x34, 0x88, 0x18,
					 0x2e, 0xc9, 0x7a, 0x32, 0xdd, 0xcb, 0x83, 0x5c, 0x45, 0x51,
					 0xf0, 0x85, 0x58, 0x6f, 0x3b, 0xc9, 0x68, 0x94, 0xb3, 0x0a,
					 0xfd, 0x77, 0x51, 0xe4, 0xa6, 0x90, 0xb5, 0x7a, 0xfc, 0x84,
					 0x19, 0x5f, 0xcb, 0x97, 0xf8, 0xdd, 0x90, 0x51, 0x16, 0xe6,
					 0xbc, 0x38, 0xbb, 0xaa, 0xaa, 0x54, 0x3e, 0xd0, 0x06, 0xd0,
					 0x28, 0x01, 0x68, 0x13, 0x81, 0x91, 0x89, 0x77, 0x03, 0x82,
					 0xe7, 0x5b, 0x66, 0x62, 0x0b, 0xc0, 0x73, 0x76, 0xb7, 0x89,
					 0x56, 0x20, 0xe2, 0xd4, 0x44, 0x6c, 0xc2, 0x9e, 0x22, 0xa8,
					 0x4b, 0xad, 0x1f, 0x7f, 0x87, 0x11, 0x61, 0x4c, 0x7e, 0x91,
					 0x9d, 0x48, 0x21, 0xb4, 0x0d, 0xdb, 0x46, 0x04, 0x88, 0xf1,
					 0xad, 0xd3, 0x6b, 0x40, 0xb4, 0x23, 0xfc, 0x9c, 0xed, 0xfe,
					 0x10, 0xfe, 0x12, 0x1b, 0xee, 0x24, 0xe2, 0x0c, 0x4f, 0x0b,
					 0xb1, 0x8d, 0xbd, 0x71, 0xc6, 0x02, 0xaf, 0x0b, 0x7a, 0x5a,
					 0x15, 0xa3, 0xea, 0xf3, 0xce, 0x44, 0xb2, 0xe8, 0x70, 0x90,
					 0x15, 0xf1, 0x35, 0xdb, 0xfd, 0x0f, 0x76, 0xe4, 0xb8, 0xde,
					 0xf7, 0x75, 0x80, 0x1c, 0x0e, 0x9e)
};
