/* Copyright (C) 1993,1994 by the author(s).
 
 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with ShapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
*/
/*
 * AtFStk -- Attribute Filesystem Toolkit Library
 *
 * network.c -- AtFS toolkit library
 *
 * Author: Andreas Lampen (Andreas.Lampen@cs.tu-berlin.de)
 *
 * $Header: network.c[7.0] Fri Jun 25 16:39:02 1993 andy@cs.tu-berlin.de frozen $
 */

#include <ctype.h>
#include "atfs.h"
#include "atfstk.h"

/*=========================================================================
 * Path juggling
 *=========================================================================*/

EXPORT char *atNetworkPath (aso)
     Af_key *aso;
     /* returns: network path name "<hostname>:<path>" */
{
  static char networkPath[PATH_MAX+HOST_MAX+AT_CACHEKEYLEN];
  char *realPath, curPath[PATH_MAX], tmpPath[PATH_MAX];
  FILE *mountTable;
  struct atMntent *mountEntry;
  int  curStatus, curNameLen, dirNameLen = 0, isDir = FALSE;

  getcwd (curPath, PATH_MAX-2);

  curStatus = af_retnumattr (aso, AF_ATTSTATE);

  if ((curStatus > AF_BUSY) || (curStatus == AF_DERIVED)) {
    if (chdir (aso->af_ldes->af_arpath) == -1) {
      realPath = curPath;
    }
    else {
      char *dirPtr;
      getcwd (tmpPath, PATH_MAX-2);
      /* cut archive directory name if present */
      if ((dirPtr = strrchr (tmpPath, '/')) && !strcmp (dirPtr+1, AF_SUBDIR))
	*dirPtr = '\0';
      realPath = tmpPath;
    }
  }
  else {
    /* check if aso is directory */
    if (S_ISDIR ((mode_t) af_retnumattr (aso, AF_ATTMODE))) {
      char *typePtr;
      isDir = TRUE;
      strcpy (tmpPath, af_retattr (aso, AF_ATTSPATH));
      strcat (tmpPath, "/");
      strcat (tmpPath, af_retattr (aso, AF_ATTNAME));
      typePtr = af_retattr (aso, AF_ATTTYPE);
      if (*typePtr) {
	strcat (tmpPath, ".");
	strcat (tmpPath, typePtr);
      }
      chdir (tmpPath);
    }
    else
      chdir (af_retattr (aso, AF_ATTSPATH));
    getcwd (tmpPath, PATH_MAX-2);
    realPath = tmpPath;
  }
  chdir (curPath); /* go back */
  if (realPath == tmpPath) {
    strcpy (curPath, tmpPath);
    realPath = curPath;
  }

  if ((mountTable = atSetmntent ("/etc/mtab", "r")) == NULL)
    return (NULL);
  while ((mountEntry = atGetmntent (mountTable))) {
    curNameLen = strlen (mountEntry->at_mnt_dir);
    if (!strncmp (mountEntry->at_mnt_dir, realPath, curNameLen)) {
      if (curNameLen < dirNameLen)
	continue;
      dirNameLen = curNameLen;
      strcpy (tmpPath, mountEntry->at_mnt_fsname);
    }
  }
  atEndmntent (mountTable);
  if (dirNameLen == 0) /* nothingFound */
    return (NULL);

  if (strchr (tmpPath, ':')) { /* if filesystem is remote */
    strcpy (networkPath, tmpPath);
    if (realPath[dirNameLen])
      strcat (networkPath, &realPath[dirNameLen]);
  }
  else {
    strcpy (networkPath, af_gethostname ());
    strcat (networkPath, ":");
    strcat (networkPath, realPath);
  }
  if (!isDir) {
    strcat (networkPath, "/");
    strcat (networkPath, af_retattr (aso, AF_ATTUNIXNAME));
  }
  if (curStatus == AF_BUSY) {
    strcat (networkPath, "@busy");
  }
  else if (curStatus == AF_DERIVED) {
    strcat (networkPath, "~");
    strcat (networkPath, af_retattr (aso, AT_ATTCACHEKEY));
  }
  else {
    strcat (networkPath, "@");
    strcat (networkPath, af_retattr (aso, AF_ATTVERSION));
  }
  return (networkPath);
}

EXPORT char *atLocalPath (networkPath)
     char *networkPath;
{
  static char localPath[PATH_MAX+AT_CACHEKEYLEN];
  int hostNameLen;
  char *strPtr, *tmpPtr;
  FILE *mountTable;
  struct atMntent *mountEntry;
  int  curNameLen, dirNameLen = 0;

  if (!networkPath)
    return (NULL);
  if ((strPtr = strchr (networkPath, ':')) == NULL)
    return (networkPath);

  hostNameLen = strPtr - networkPath;
  if (strncmp (networkPath, af_gethostname(), hostNameLen)) {
    /* remote path */
    if ((mountTable = atSetmntent ("/etc/mtab", "r")) == NULL)
      return (networkPath);
    while ((mountEntry = atGetmntent (mountTable))) {
      curNameLen = strlen (mountEntry->at_mnt_fsname);
      if (!strncmp (mountEntry->at_mnt_fsname, networkPath, curNameLen)) {
	if (curNameLen < dirNameLen)
	  continue;
	dirNameLen = curNameLen;
	strcpy (localPath, mountEntry->at_mnt_dir);
      }
    }
    atEndmntent (mountTable);
    if (dirNameLen == 0) /* nothingFound */
      return (networkPath);
    strcat (localPath, &networkPath[dirNameLen]);
  }
  else
    strcpy (localPath, strPtr+1);

  /* change version number representation */
  if ((strPtr = strrchr (localPath, '@'))) {
    /* saved or busy version */
    tmpPtr = strPtr+1;
    if (!strncmp (tmpPtr, "busy", 4))
      *strPtr = '\0';
    else {
      while (*tmpPtr) {
	if ((!isdigit (*tmpPtr)) && (*tmpPtr != '.'))
	  return (localPath);
	tmpPtr++;
      }
      *strPtr = '[';
      *tmpPtr++ = ']';
      *tmpPtr = '\0';
    }
  }
  else if ((strPtr = strrchr (localPath, '~'))) {
    tmpPtr = strPtr+1;
    while (*tmpPtr) {
      if ((!isdigit (*tmpPtr)) && (*tmpPtr != '.')) 
	return (localPath);
      tmpPtr++;
    }
    *strPtr = '[';
    *tmpPtr++ = ']';
    *tmpPtr = '\0';
  }

  return (localPath);
}
