#ifndef OPENMM_DRUDESCFINTEGRATOR_H_
#define OPENMM_DRUDESCFINTEGRATOR_H_

/* -------------------------------------------------------------------------- *
 *                                   OpenMM                                   *
 * -------------------------------------------------------------------------- *
 * This is part of the OpenMM molecular simulation toolkit originating from   *
 * Simbios, the NIH National Center for Physics-Based Simulation of           *
 * Biological Structures at Stanford, funded under the NIH Roadmap for        *
 * Medical Research, grant U54 GM072970. See https://simtk.org.               *
 *                                                                            *
 * Portions copyright (c) 2008-2021 Stanford University and the Authors.      *
 * Authors: Peter Eastman                                                     *
 * Contributors:                                                              *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS, CONTRIBUTORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,    *
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR      *
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE  *
 * USE OR OTHER DEALINGS IN THE SOFTWARE.                                     *
 * -------------------------------------------------------------------------- */

#include "openmm/DrudeIntegrator.h"
#include "openmm/Kernel.h"
#include "openmm/internal/windowsExportDrude.h"

namespace OpenMM {

/**
 * This is a leap-frog Verlet Integrator that simulates systems with Drude particles.  It uses the
 * self-consistent field (SCF) method: at every time step, the positions of Drude particles are
 * adjusted to minimize the potential energy.
 * 
 * This Integrator requires the System to include a DrudeForce, which it uses to identify the Drude
 * particles.
 */

class OPENMM_EXPORT_DRUDE DrudeSCFIntegrator : public DrudeIntegrator {
public:
    /**
     * Create a DrudeSCFIntegrator.
     *
     * @param stepSize       the step size with which to integrator the system (in picoseconds)
     */
    DrudeSCFIntegrator(double stepSize);
    /**
     * Get the error tolerance to use when minimizing the potential energy.  This roughly corresponds
     * to the maximum allowed force magnitude on the Drude particles after minimization.
     *
     * @return the error tolerance to use, measured in kJ/mol/nm
     */
    double getMinimizationErrorTolerance() const {
        return tolerance;
    }
    /**
     * Set the error tolerance to use when minimizing the potential energy.  This roughly corresponds
     * to the maximum allowed force magnitude on the Drude particles after minimization.
     *
     * @param tol    the error tolerance to use, measured in kJ/mol/nm
     */
    void setMinimizationErrorTolerance(double tol);
    /**
     * Advance a simulation through time by taking a series of time steps.
     *
     * @param steps   the number of time steps to take
     */
    void step(int steps) override;
protected:
    /**
     * This will be called by the Context when it is created.  It informs the Integrator
     * of what context it will be integrating, and gives it a chance to do any necessary initialization.
     * It will also get called again if the application calls reinitialize() on the Context.
     */
    void initialize(ContextImpl& context) override;
    /**
     * This will be called by the Context when it is destroyed to let the Integrator do any necessary
     * cleanup.  It will also get called again if the application calls reinitialize() on the Context.
     */
    void cleanup() override;
    /**
     * Get the names of all Kernels used by this Integrator.
     */
    std::vector<std::string> getKernelNames() override;
    /**
     * Compute the kinetic energy of the system at the current time.
     */
    double computeKineticEnergy() override;
private:
    double tolerance;
    Kernel kernel;
};

} // namespace OpenMM

#endif /*OPENMM_DRUDESCFINTEGRATOR_H_*/
