// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package filesystem

import (
	"context"
	"os"

	"code.forgejo.org/f3/gof3/v3/f3"
	helpers_repository "code.forgejo.org/f3/gof3/v3/forges/helpers/repository"
	"code.forgejo.org/f3/gof3/v3/id"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	"code.forgejo.org/f3/gof3/v3/util"
)

type repositoryDriver struct {
	nodeDriver
	h helpers_repository.Interface
}

func newRepositoryDriver(content f3.Interface) generic.NodeDriverInterface {
	n := newNodeDriver(content).(*nodeDriver)
	r := &repositoryDriver{
		nodeDriver: *n,
	}
	r.h = helpers_repository.NewHelper(r)
	return r
}

func (o *repositoryDriver) ToFormat() f3.Interface {
	from := o.GetRepositoryURL()
	o.Trace("%s", from)
	f := o.nodeDriver.ToFormat().(*f3.Repository)
	f.FetchFunc = func(ctx context.Context, destination, internalRef string) {
		o.Trace("git clone %s %s", from, destination)
		helpers_repository.GitMirror(ctx, o, from, destination, internalRef)
	}
	return f
}

func (o *repositoryDriver) GetHelper() any {
	return o.h
}

func (o *repositoryDriver) SetFetchFunc(fetchFunc func(ctx context.Context, destination, internalRef string)) {
	f := o.content.(*f3.Repository)
	f.FetchFunc = fetchFunc
}

func (o *repositoryDriver) GetRepositoryURL() string {
	return o.getBasePath()
}

func (o *repositoryDriver) GetRepositoryInternalRef() string {
	return ""
}

func (o *repositoryDriver) GetPullRequestBranch(pr *f3.PullRequestBranch) *f3.PullRequestBranch {
	panic("")
}
func (o *repositoryDriver) CreatePullRequestBranch(pr *f3.PullRequestBranch) {}
func (o *repositoryDriver) DeletePullRequestBranch(pr *f3.PullRequestBranch) {}

func (o *repositoryDriver) ensureRepository(ctx context.Context, directory string) {
	if !util.FileExists(directory) {
		if err := os.MkdirAll(directory, 0o700); err != nil {
			panic(err)
		}
		util.Command(ctx, o, "git", "-C", directory, "init", "--bare")
	}
}

func (o *repositoryDriver) Put(ctx context.Context) id.NodeID {
	return o.upsert(ctx)
}

func (o *repositoryDriver) Patch(ctx context.Context) {
	o.upsert(ctx)
}

func (o *repositoryDriver) upsert(ctx context.Context) id.NodeID {
	o.nodeDriver.upsert(ctx)
	repoDir := o.GetRepositoryURL()
	o.GetNode().Trace("%s %s", repoDir, o.GetNode().GetID())
	o.ensureRepository(ctx, repoDir)
	return o.h.Upsert(ctx, o.content.(*f3.Repository))
}

func (o *repositoryDriver) Get(ctx context.Context) bool {
	o.nodeDriver.Get(ctx)
	f := o.content.(*f3.Repository)
	f.SetID(o.GetNode().GetID().String())
	repoDir := o.GetRepositoryURL()
	o.GetNode().Trace("%s", repoDir)
	o.h.Get(ctx)
	return true
}
