/*
    serialport library
    based on:
    Cross-platform serial / RS232 library
    Version 0.21, 11/10/2015
    -> All platforms header
    -> rs232.h

    The MIT License (MIT)

    Copyright (c) 2013-2015 Frdric Meslin, Florent Touchard
    Email: fredericmeslin@hotmail.com
    Website: www.fredslab.net
    Twitter: @marzacdev

    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

#ifndef _SERIALPORT_H_
#define _SERIALPORT_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdlib.h>
#include <limits.h>

#ifdef _WIN32
#define SERIALPORT_WIN
#elif defined(__APPLE__) && defined(__MACH__)
#define SERIALPORT_MAC
#else
#define SERIALPORT_LINUX
#endif
    
    
#ifdef SERIALPORT_WIN
#define SERIALPORT_HANDLE_TYPE void *
#define SERIALPORT_HANDLE_INVALID 0
#define SERIALPORT_HANDLE_CHECK_VALID(handle) ((handle)!=0)
#else
#define SERIALPORT_HANDLE_TYPE int
#define SERIALPORT_HANDLE_INVALID -1
#define SERIALPORT_HANDLE_CHECK_VALID(handle) ((handle)>=0)
#endif

#define SERIALPORT_TIMEOUT_INFINITE UINT_MAX
    
/*****************************************************************************/
    /* Doxywizard specific */
    /**
    * \mainpage RS232
    * \section intro_sec C / C++ RS232 cross-platform serial library
    * <b>Version 0.21 - 11/10/2015</b>
    *
    * Supported platforms:  
    * - Windows (XP / Win7, Win8, Win10)
    * - Linux
    * - MacOS X
    *
    * Copyright (c) 2013-2015 Fr&eacute;d&eacute;ric Meslin, Florent Touchard <br>
    * Email: fredericmeslin@hotmail.com <br>
    * Website: www.fredslab.net <br>
    * Twitter: \@marzacdev <br>
    */
    
/*****************************************************************************/
    /**
     * \fn int serialportEnumerate()
     * \brief Enumerate available serial ports (Serial, USB serial, Bluetooth serial)
     * \return number of enumerated ports
     */
    int serialportEnumerate();
    
    /**
     * \fn int serialportGetNoPorts()
     * \brief Return the number of enumerated ports
     * \return number of enumerated ports
     */
    int serialportGetNoPorts();

    /**
     * \fn int serialportTerminate()
     * \brief Release ports and memory resources used by the library
     */    
    void serialportTerminate();

    /**
     * \fn const char * serialportGetPortName(int index)
     * \brief Get port user-friendly name
     * \param[in] index port index
     * \return null terminated port name
     */    
    const char * serialportGetPortName(int index);

    /**
     * \fn const char * serialportGetInternalName(int index)
     * \brief Get port operating-system name
     * \param[in] index port index
     * \return null terminated port name
     */        
    const char * serialportGetInternalName(int index);
    
    /**
     * \fn int serialportFindPort(const char * name)
     * \brief Try to find a port given its user-friendly name
     * \param[in] name port name (case sensitive)
     * \return index of found port or -1 if not enumerated
     */
    int serialportFindPort(const char * name);

/*****************************************************************************/
    /**
     * \fn SERIALPORT_HANDLE_TYPE serialportOpen(int index, int baudrate)
     * \brief Try to open a port at a specific baudrate
     * \brief (No parity, single stop bit, no hardware flow control)
     * \param[in] index port index
     * \param[in] baudrate port baudrate
     * \return handle if opened, SERIALPORT_HANDLE_INVALID if not available
     */
    SERIALPORT_HANDLE_TYPE serialportOpen(int index, int baudrate);

    
    /**
     * \fn void serialportClose(SERIALPORT_HANDLE_TYPE handle)
     * \brief Close an opened port
     * \param[in] handle port handle
     */            
    void serialportClose(SERIALPORT_HANDLE_TYPE handle);
    void serialportCloseAll();

    
    /**
     * \fn SERIALPORT_HANDLE_TYPE serialportOpen(int index, int baudrate)
     * \brief Try to open a port at a specific baudrate
     * \brief (No parity, single stop bit, no hardware flow control)
     * \param[in] handle port handle
     * \param[in] baudrate port baudrate
     * \return handle if opened, SERIALPORT_HANDLE_INVALID if not available
     */
    int serialportSetBaudrate(SERIALPORT_HANDLE_TYPE handle, int baudrate);

    /**
     * \fn int serialportWrite(SERIALPORT_HANDLE_TYPE handle, const char * buffer, size_t len)
     * \brief Write data to the port (non-blocking)
     * \param[in] handle port handle
     * \param[in] buffer pointer to transmit buffer
     * \param[in] len length of transmit buffer in bytes
     * \return number of bytes transferred
     */
    int serialportWrite(SERIALPORT_HANDLE_TYPE handle, const char * buffer, size_t len);
    
    
    /**
     * \fn int serialportRead(SERIALPORT_HANDLE_TYPE handle, const char * buffer, size_t len)
     * \brief Read data from the port (non-blocking)
     * \param[in] handle port handle
     * \param[in] buffer pointer to receive buffer
     * \param[in] len length of receive buffer in bytes
     * \param[in] timeout_ms timeout in milliseconds, 0 for immediate return, SERIALPORT_TIMEOUT_INFINITE for infinite
     * \return number of bytes received, 0 on timeout, -1 on error
     */
    int serialportRead(SERIALPORT_HANDLE_TYPE handle, char * buffer, size_t len, unsigned int timeout_ms);

#ifdef __cplusplus
}
#endif

#endif // _SERIALPORT_H_
